<?php

declare(strict_types=1);

if (!function_exists('moviles_disponibles_fetch_combos')) {
    /**
     * Obtiene catálogos para filtros de móviles disponibles.
     */
    function moviles_disponibles_fetch_combos(PDO $pdo): array
    {
        $movilesSql = 'SELECT id, chapa FROM para_moviles';
        if (moviles_disponibles_table_has_column($pdo, 'para_moviles', 'activo')) {
            $movilesSql .= ' WHERE activo = 1';
        }
        $movilesSql .= ' ORDER BY chapa LIMIT 400';
        $moviles = $pdo->query($movilesSql)->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $transportadorasSql = 'SELECT id, nombre FROM para_transportadoras';
        if (moviles_disponibles_table_has_column($pdo, 'para_transportadoras', 'activo')) {
            $transportadorasSql .= ' WHERE activo = 1';
        }
        $transportadorasSql .= ' ORDER BY nombre LIMIT 200';
        $transportadoras = $pdo->query($transportadorasSql)->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return [
            'moviles' => array_map(static function (array $row): array {
                $chapa = trim((string) ($row['chapa'] ?? ''));
                $label = $chapa !== '' ? $chapa : ('ID ' . (int) $row['id']);
                return [
                    'id' => (int) $row['id'],
                    'label' => $label,
                ];
            }, $moviles),
            'transportadoras' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'nombre' => trim((string) ($row['nombre'] ?? '')) ?: 'Sin transportadora',
                ];
            }, $transportadoras),
            'estados' => [
                ['value' => '', 'label' => 'Todos'],
                ['value' => '1', 'label' => 'Disponibles'],
                ['value' => '0', 'label' => 'No disponibles'],
            ],
            'binarios' => [
                ['value' => '', 'label' => 'Todos'],
                ['value' => '1', 'label' => 'Sí'],
                ['value' => '0', 'label' => 'No'],
            ],
        ];
    }
}

if (!function_exists('moviles_disponibles_fetch_data')) {
    /**
     * Consulta principal del reporte con filtros y límite.
     */
    function moviles_disponibles_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = moviles_disponibles_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        $where = [];
        $params = [];

        $where[] = 'DATE(md.fecha) >= :fecha_desde';
        $where[] = 'DATE(md.fecha) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['movil_id'] !== '') {
            $where[] = 'md.movil_id = :movil_id';
            $params[':movil_id'] = (int) $filters['movil_id'];
        }

        if ($filters['transportadora_id'] !== '') {
            $where[] = 'pt.id = :transportadora_id';
            $params[':transportadora_id'] = (int) $filters['transportadora_id'];
        }

        if ($filters['disponible'] !== '') {
            $where[] = 'md.disponible = :disponible';
            $params[':disponible'] = (int) $filters['disponible'];
        }

        if ($filters['no_asistio'] !== '') {
            $where[] = 'md.no_asistio = :no_asistio';
            $params[':no_asistio'] = (int) $filters['no_asistio'];
        }

        if ($filters['no_utilizado'] !== '') {
            $where[] = 'md.no_utilizado = :no_utilizado';
            $params[':no_utilizado'] = (int) $filters['no_utilizado'];
        }

        if ($filters['buscar'] !== '') {
            $where[] = '(md.notas LIKE :buscar OR pm.chapa LIKE :buscar OR COALESCE(pm.alias, \'\') LIKE :buscar OR pt.nombre LIKE :buscar)';
            $params[':buscar'] = '%' . $filters['buscar'] . '%';
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = '
            SELECT
                md.id,
                md.fecha,
                md.movil_id,
                md.disponible,
                md.ventana_ini,
                md.ventana_fin,
                md.llegada_time,
                md.no_asistio,
                md.no_utilizado,
                md.notas,
                md.created_at,
                md.updated_at,
                pm.chapa AS movil_chapa,
                pt.id AS transportadora_id,
                pt.nombre AS transportadora_nombre
            FROM oper_movil_disponible md
            LEFT JOIN para_moviles pm ON pm.id = md.movil_id
            LEFT JOIN para_transportadoras pt ON pt.id = pm.transportadora_id
            ' . $whereSql . '
            ORDER BY md.fecha DESC, md.id DESC
            LIMIT :limit_plus
        ';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_plus', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();

        $rawRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $truncated = false;
        if (count($rawRows) > $limit) {
            $rawRows = array_slice($rawRows, 0, $limit);
            $truncated = true;
        }

        $rows = array_map('moviles_disponibles_format_row', $rawRows);
        $summary = moviles_disponibles_build_summary($rows, $filters);
        $aggregates = moviles_disponibles_build_aggregates($rows);

        return [
            'filters' => $filters,
            'rows' => $rows,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('moviles_disponibles_build_summary')) {
    /**
     * Resumen general del reporte.
     */
    function moviles_disponibles_build_summary(array $rows, array $filters): array
    {
        if (!$rows) {
            return [
                'total_registros' => 0,
                'total_moviles' => 0,
                'total_transportadoras' => 0,
                'total_disponibles' => 0,
                'total_no_disponibles' => 0,
                'total_no_asistio' => 0,
                'total_no_utilizado' => 0,
                'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
            ];
        }

        $moviles = [];
        $transportadoras = [];
        $disponibles = 0;
        $noDisponibles = 0;
        $noAsistio = 0;
        $noUtilizado = 0;

        foreach ($rows as $row) {
            if (!empty($row['movil_id'])) {
                $moviles[(int) $row['movil_id']] = true;
            }
            if (!empty($row['transportadora_id'])) {
                $transportadoras[(int) $row['transportadora_id']] = true;
            }
            if (!empty($row['disponible'])) {
                $disponibles++;
            } else {
                $noDisponibles++;
            }
            if (!empty($row['no_asistio'])) {
                $noAsistio++;
            }
            if (!empty($row['no_utilizado'])) {
                $noUtilizado++;
            }
        }

        return [
            'total_registros' => count($rows),
            'total_moviles' => count($moviles),
            'total_transportadoras' => count($transportadoras),
            'total_disponibles' => $disponibles,
            'total_no_disponibles' => $noDisponibles,
            'total_no_asistio' => $noAsistio,
            'total_no_utilizado' => $noUtilizado,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('moviles_disponibles_build_aggregates')) {
    /**
     * Agregados por transportadora, móvil y fecha.
     */
    function moviles_disponibles_build_aggregates(array $rows): array
    {
        if (!$rows) {
            return [
                'transportadoras' => [],
                'moviles' => [],
                'fechas' => [],
            ];
        }

        $transportadoras = [];
        $moviles = [];
        $fechas = [];

        foreach ($rows as $row) {
            $transportadoraKey = (string) ($row['transportadora_id'] ?? 0);
            if (!isset($transportadoras[$transportadoraKey])) {
                $transportadoras[$transportadoraKey] = [
                    'key' => $transportadoraKey,
                    'label' => $row['transportadora_nombre'] ?? 'Sin transportadora',
                    'rows' => 0,
                    'disponibles' => 0,
                    'no_disponibles' => 0,
                    'no_asistio' => 0,
                    'no_utilizado' => 0,
                ];
            }
            $transportadoras[$transportadoraKey]['rows']++;
            if (!empty($row['disponible'])) {
                $transportadoras[$transportadoraKey]['disponibles']++;
            } else {
                $transportadoras[$transportadoraKey]['no_disponibles']++;
            }
            if (!empty($row['no_asistio'])) {
                $transportadoras[$transportadoraKey]['no_asistio']++;
            }
            if (!empty($row['no_utilizado'])) {
                $transportadoras[$transportadoraKey]['no_utilizado']++;
            }

            $movilKey = (string) ($row['movil_id'] ?? 0);
            if (!isset($moviles[$movilKey])) {
                $movilLabel = $row['movil_label'] ?? 'Sin móvil';
                $moviles[$movilKey] = [
                    'key' => $movilKey,
                    'label' => $movilLabel,
                    'rows' => 0,
                    'disponibles' => 0,
                    'no_asistio' => 0,
                    'no_utilizado' => 0,
                ];
            }
            $moviles[$movilKey]['rows']++;
            if (!empty($row['disponible'])) {
                $moviles[$movilKey]['disponibles']++;
            }
            if (!empty($row['no_asistio'])) {
                $moviles[$movilKey]['no_asistio']++;
            }
            if (!empty($row['no_utilizado'])) {
                $moviles[$movilKey]['no_utilizado']++;
            }

            $fechaKey = $row['fecha'] ?? 'Sin fecha';
            if (!isset($fechas[$fechaKey])) {
                $fechas[$fechaKey] = [
                    'fecha' => $fechaKey,
                    'rows' => 0,
                    'disponibles' => 0,
                    'no_disponibles' => 0,
                ];
            }
            $fechas[$fechaKey]['rows']++;
            if (!empty($row['disponible'])) {
                $fechas[$fechaKey]['disponibles']++;
            } else {
                $fechas[$fechaKey]['no_disponibles']++;
            }
        }

        $transportadorasList = array_values($transportadoras);
        usort($transportadorasList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });
        $transportadorasList = array_slice($transportadorasList, 0, 10);

        $movilesList = array_values($moviles);
        usort($movilesList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });
        $movilesList = array_slice($movilesList, 0, 10);

        $fechasList = array_values($fechas);
        usort($fechasList, static function (array $a, array $b): int {
            return strcmp((string) $b['fecha'], (string) $a['fecha']);
        });
        $fechasList = array_slice($fechasList, 0, 15);

        return [
            'transportadoras' => $transportadorasList,
            'moviles' => $movilesList,
            'fechas' => $fechasList,
        ];
    }
}

if (!function_exists('moviles_disponibles_format_row')) {
    /**
     * Normaliza filas para consumo del front.
     */
    function moviles_disponibles_format_row(array $row): array
    {
        $fecha = $row['fecha'] ?? null;
        if ($fecha !== null) {
            $fecha = substr((string) $fecha, 0, 10);
        }

        $movilChapa = trim((string) ($row['movil_chapa'] ?? ''));
        $movilLabel = $movilChapa !== '' ? $movilChapa : 'Sin móvil';

        $ventanaIni = $row['ventana_ini'] ?? null;
        if ($ventanaIni !== null) {
            $ventanaIni = substr((string) $ventanaIni, 0, 5);
        }
        $ventanaFin = $row['ventana_fin'] ?? null;
        if ($ventanaFin !== null) {
            $ventanaFin = substr((string) $ventanaFin, 0, 5);
        }
        $llegada = $row['llegada_time'] ?? null;
        if ($llegada !== null) {
            $llegada = substr((string) $llegada, 0, 5);
        }

        $notas = $row['notas'] ?? null;
        if ($notas !== null) {
            $notas = trim((string) $notas);
            if ($notas === '') {
                $notas = null;
            }
        }

        return [
            'id' => (int) ($row['id'] ?? 0),
            'fecha' => $fecha,
            'movil_id' => isset($row['movil_id']) ? (int) $row['movil_id'] : null,
            'movil_label' => $movilLabel,
            'transportadora_id' => isset($row['transportadora_id']) ? (int) $row['transportadora_id'] : null,
            'transportadora_nombre' => trim((string) ($row['transportadora_nombre'] ?? '')) ?: 'Sin transportadora',
            'disponible' => (int) ($row['disponible'] ?? 0) === 1,
            'ventana_ini' => $ventanaIni,
            'ventana_fin' => $ventanaFin,
            'llegada_time' => $llegada,
            'no_asistio' => (int) ($row['no_asistio'] ?? 0) === 1,
            'no_utilizado' => (int) ($row['no_utilizado'] ?? 0) === 1,
            'notas' => $notas,
            'created_at' => $row['created_at'] ?? null,
            'updated_at' => $row['updated_at'] ?? null,
        ];
    }
}

if (!function_exists('moviles_disponibles_normalize_filters')) {
    function moviles_disponibles_normalize_filters(array $input): array
    {
        $defaultDesde = date('Y-m-d', strtotime('-7 days'));
        $defaultHasta = date('Y-m-d');

        $desde = moviles_disponibles_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = moviles_disponibles_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);

        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'movil_id' => moviles_disponibles_sanitize_numeric($input['movil_id'] ?? null),
            'transportadora_id' => moviles_disponibles_sanitize_numeric($input['transportadora_id'] ?? null),
            'disponible' => moviles_disponibles_sanitize_boolean($input['disponible'] ?? null),
            'no_asistio' => moviles_disponibles_sanitize_boolean($input['no_asistio'] ?? null),
            'no_utilizado' => moviles_disponibles_sanitize_boolean($input['no_utilizado'] ?? null),
            'buscar' => moviles_disponibles_sanitize_string($input['buscar'] ?? '', 120),
        ];
    }
}

if (!function_exists('moviles_disponibles_sanitize_date')) {
    function moviles_disponibles_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('moviles_disponibles_sanitize_numeric')) {
    function moviles_disponibles_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('moviles_disponibles_sanitize_boolean')) {
    function moviles_disponibles_sanitize_boolean($value): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        if ($value === '1' || $value === '0') {
            return $value;
        }
        return '';
    }
}

if (!function_exists('moviles_disponibles_sanitize_string')) {
    function moviles_disponibles_sanitize_string(?string $value, int $maxLength): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        if (strlen($value) > $maxLength) {
            $value = substr($value, 0, $maxLength);
        }
        return $value;
    }
}

if (!function_exists('moviles_disponibles_table_has_column')) {
    function moviles_disponibles_table_has_column(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $cacheKey = $table . ':' . $column;
        if (array_key_exists($cacheKey, $cache)) {
            return $cache[$cacheKey];
        }

        if (!preg_match('/^[a-zA-Z0-9_]+$/', $table) || !preg_match('/^[a-zA-Z0-9_]+$/', $column)) {
            $cache[$cacheKey] = false;
            return false;
        }

        $tableSafe = str_replace('`', '``', $table);
        $sql = sprintf('SHOW COLUMNS FROM `%s` LIKE :column', $tableSafe);

        try {
            $stmt = $pdo->prepare($sql);
            $stmt->bindValue(':column', $column, PDO::PARAM_STR);
            $stmt->execute();
            $cache[$cacheKey] = (bool) $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Throwable $e) {
            $cache[$cacheKey] = false;
        }

        return $cache[$cacheKey];
    }
}
