<?php

declare(strict_types=1);

if (!function_exists('pedidos_hora_fetch_combos')) {
    /**
     * Obtiene catálogos para los filtros del reporte de pedidos por hora.
     */
    function pedidos_hora_fetch_combos(PDO $pdo): array
    {
        $clientes = $pdo->query('SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social LIMIT 400')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $estados = $pdo->query('SELECT id, code, nombre FROM so_pedido_estado ORDER BY orden ASC, nombre ASC')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return [
            'clientes' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'nombre' => trim((string) ($row['razon_social'] ?? 'Sin cliente')) ?: 'Sin cliente',
                ];
            }, $clientes),
            'estados' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'code' => $row['code'] ?? null,
                    'nombre' => trim((string) ($row['nombre'] ?? 'Sin estado')) ?: 'Sin estado',
                ];
            }, $estados),
        ];
    }
}

if (!function_exists('pedidos_hora_fetch_data')) {
    /**
     * Retorna el resumen del reporte, agregados por hora y detalle diario.
     */
    function pedidos_hora_fetch_data(PDO $pdo, array $rawFilters, int $dailyLimit = 90): array
    {
        $filters = pedidos_hora_normalize_filters($rawFilters);
        $dailyLimit = max(1, min($dailyLimit, 365));
        if (!empty($filters['limit_days'])) {
            $dailyLimit = max(1, min((int) $filters['limit_days'], 365));
        }

        [$whereSql, $params] = pedidos_hora_build_where($filters);

        $totals = pedidos_hora_fetch_totals($pdo, $whereSql, $params);
        $hours = pedidos_hora_fetch_hours($pdo, $whereSql, $params, (int) ($totals['total_pedidos'] ?? 0));
        $daily = pedidos_hora_fetch_daily($pdo, $whereSql, $params, $dailyLimit);
        $aggregates = pedidos_hora_fetch_aggregates($pdo, $whereSql, $params);

        $summary = pedidos_hora_build_summary($filters, $totals, $hours, $daily);
        $chart = pedidos_hora_build_chart($hours);

        return [
            'filters' => $filters,
            'summary' => $summary,
            'hours' => $hours,
            'daily' => $daily,
            'aggregates' => $aggregates,
            'chart' => $chart,
            'limit_days' => $dailyLimit,
        ];
    }
}

if (!function_exists('pedidos_hora_build_summary')) {
    /**
     * Construye la tarjeta de resumen principal.
     */
    function pedidos_hora_build_summary(array $filters, array $totals, array $hours, array $daily): array
    {
        $totalPedidos = (int) ($totals['total_pedidos'] ?? 0);
        $totalDias = (int) ($totals['total_dias'] ?? 0);
        $totalClientes = (int) ($totals['total_clientes'] ?? 0);

        $horasActivas = 0;
        $horaPico = null;
        $horaPicoPedidos = 0;
        foreach ($hours as $hour) {
            if ((int) $hour['pedidos'] > 0) {
                $horasActivas++;
                if ((int) $hour['pedidos'] > $horaPicoPedidos) {
                    $horaPicoPedidos = (int) $hour['pedidos'];
                    $horaPico = $hour['label'];
                }
            }
        }

        $promedioDiario = $totalDias > 0 ? $totalPedidos / $totalDias : 0.0;
        $promedioHora = $totalDias > 0 ? $totalPedidos / ($totalDias * 24) : 0.0;
        $promedioHoraActiva = $horasActivas > 0 ? $totalPedidos / $horasActivas : 0.0;

        $primerRegistro = $totals['primer_registro'] ?? null;
        $ultimoRegistro = $totals['ultimo_registro'] ?? null;

        return [
            'total_pedidos' => $totalPedidos,
            'total_dias' => $totalDias,
            'total_clientes' => $totalClientes,
            'horas_activas' => $horasActivas,
            'promedio_diario' => $promedioDiario,
            'promedio_por_hora' => $promedioHora,
            'promedio_por_hora_activa' => $promedioHoraActiva,
            'hora_pico' => $horaPico,
            'hora_pico_pedidos' => $horaPicoPedidos,
            'primer_registro' => $primerRegistro,
            'ultimo_registro' => $ultimoRegistro,
            'dias_detalle' => count($daily),
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('pedidos_hora_build_chart')) {
    /**
     * Prepara formato listo para Chart.js.
     */
    function pedidos_hora_build_chart(array $hours): array
    {
        return [
            'labels' => array_map(static function (array $row): string {
                return (string) ($row['label'] ?? '');
            }, $hours),
            'values' => array_map(static function (array $row): int {
                return (int) ($row['pedidos'] ?? 0);
            }, $hours),
        ];
    }
}

if (!function_exists('pedidos_hora_fetch_hours')) {
    /**
     * Agregados por hora (0 a 23).
     */
    function pedidos_hora_fetch_hours(PDO $pdo, string $whereSql, array $params, int $totalPedidos): array
    {
        $sql = '
            SELECT
                HOUR(p.fecha_pedido) AS hora,
                COUNT(*) AS pedidos,
                COUNT(DISTINCT DATE(p.fecha_pedido)) AS dias,
                COUNT(DISTINCT p.cliente_id) AS clientes
            FROM so_pedido p
            ' . $whereSql . '
            GROUP BY hora
            ORDER BY hora ASC
        ';

        $stmt = $pdo->prepare($sql);
        pedidos_hora_bind_params($stmt, $params);
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $map = [];
        foreach ($rows as $row) {
            $map[(int) $row['hora']] = $row;
        }

        $result = [];
        for ($i = 0; $i < 24; $i++) {
            $row = $map[$i] ?? ['pedidos' => 0, 'dias' => 0, 'clientes' => 0];
            $pedidos = (int) ($row['pedidos'] ?? 0);
            $dias = (int) ($row['dias'] ?? 0);
            $clientes = (int) ($row['clientes'] ?? 0);
            $promedioDia = $dias > 0 ? $pedidos / $dias : 0.0;
            $porcentaje = $totalPedidos > 0 ? ($pedidos / $totalPedidos) * 100 : 0.0;

            $result[] = [
                'hour' => $i,
                'label' => sprintf('%02d:00', $i),
                'pedidos' => $pedidos,
                'dias' => $dias,
                'clientes' => $clientes,
                'promedio_por_dia' => $promedioDia,
                'porcentaje' => $porcentaje,
            ];
        }

        return $result;
    }
}

if (!function_exists('pedidos_hora_fetch_daily')) {
    /**
     * Detalle diario con primeros/últimos movimientos.
     */
    function pedidos_hora_fetch_daily(PDO $pdo, string $whereSql, array $params, int $limit): array
    {
        $sql = '
            SELECT
                DATE(p.fecha_pedido) AS fecha,
                COUNT(*) AS pedidos,
                COUNT(DISTINCT p.cliente_id) AS clientes,
                MIN(p.fecha_pedido) AS primer_pedido,
                MAX(p.fecha_pedido) AS ultimo_pedido
            FROM so_pedido p
            ' . $whereSql . '
            GROUP BY DATE(p.fecha_pedido)
            ORDER BY fecha DESC
            LIMIT :limit_dias
        ';

        $stmt = $pdo->prepare($sql);
        pedidos_hora_bind_params($stmt, $params);
        $stmt->bindValue(':limit_dias', $limit, PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $result = [];
        foreach ($rows as $row) {
            $result[] = [
                'fecha' => $row['fecha'] ?? null,
                'pedidos' => (int) ($row['pedidos'] ?? 0),
                'clientes' => (int) ($row['clientes'] ?? 0),
                'primer_pedido' => pedidos_hora_trim_time($row['primer_pedido'] ?? null),
                'ultimo_pedido' => pedidos_hora_trim_time($row['ultimo_pedido'] ?? null),
            ];
        }

        return $result;
    }
}

if (!function_exists('pedidos_hora_fetch_aggregates')) {
    /**
     * Top de clientes y estados.
     */
    function pedidos_hora_fetch_aggregates(PDO $pdo, string $whereSql, array $params): array
    {
        $sqlClientes = '
            SELECT
                p.cliente_id AS id,
                COALESCE(cli.razon_social, "Sin cliente") AS nombre,
                COUNT(*) AS pedidos
            FROM so_pedido p
            LEFT JOIN para_clientes cli ON cli.id = p.cliente_id
            ' . $whereSql . '
            GROUP BY p.cliente_id, nombre
            ORDER BY pedidos DESC
            LIMIT 8
        ';
        $stmtClientes = $pdo->prepare($sqlClientes);
        pedidos_hora_bind_params($stmtClientes, $params);
        $stmtClientes->execute();
        $clientes = array_map(static function (array $row): array {
            return [
                'id' => isset($row['id']) ? (int) $row['id'] : null,
                'nombre' => trim((string) ($row['nombre'] ?? 'Sin cliente')) ?: 'Sin cliente',
                'pedidos' => (int) ($row['pedidos'] ?? 0),
            ];
        }, $stmtClientes->fetchAll(PDO::FETCH_ASSOC) ?: []);

        $sqlEstados = '
            SELECT
                p.estado_id AS id,
                COALESCE(est.code, "") AS code,
                COALESCE(est.nombre, "Sin estado") AS nombre,
                COUNT(*) AS pedidos
            FROM so_pedido p
            LEFT JOIN so_pedido_estado est ON est.id = p.estado_id
            ' . $whereSql . '
            GROUP BY p.estado_id, code, nombre
            ORDER BY pedidos DESC
            LIMIT 8
        ';
        $stmtEstados = $pdo->prepare($sqlEstados);
        pedidos_hora_bind_params($stmtEstados, $params);
        $stmtEstados->execute();
        $estados = array_map(static function (array $row): array {
            return [
                'id' => isset($row['id']) ? (int) $row['id'] : null,
                'code' => $row['code'] ?? '',
                'nombre' => trim((string) ($row['nombre'] ?? 'Sin estado')) ?: 'Sin estado',
                'pedidos' => (int) ($row['pedidos'] ?? 0),
            ];
        }, $stmtEstados->fetchAll(PDO::FETCH_ASSOC) ?: []);

        return [
            'clientes' => $clientes,
            'estados' => $estados,
        ];
    }
}

if (!function_exists('pedidos_hora_fetch_totals')) {
    /**
     * Totales generales para construir el resumen.
     */
    function pedidos_hora_fetch_totals(PDO $pdo, string $whereSql, array $params): array
    {
        $sql = '
            SELECT
                COUNT(*) AS total_pedidos,
                COUNT(DISTINCT DATE(p.fecha_pedido)) AS total_dias,
                COUNT(DISTINCT p.cliente_id) AS total_clientes,
                MIN(p.fecha_pedido) AS primer_registro,
                MAX(p.fecha_pedido) AS ultimo_registro
            FROM so_pedido p
            ' . $whereSql . '
        ';

        $stmt = $pdo->prepare($sql);
        pedidos_hora_bind_params($stmt, $params);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC) ?: [
            'total_pedidos' => 0,
            'total_dias' => 0,
            'total_clientes' => 0,
            'primer_registro' => null,
            'ultimo_registro' => null,
        ];
    }
}

if (!function_exists('pedidos_hora_build_where')) {
    /**
     * Genera cláusula WHERE y parámetros compartidos entre las consultas.
     */
    function pedidos_hora_build_where(array $filters): array
    {
        $where = [];
        $params = [];

        $where[] = 'DATE(p.fecha_pedido) >= :fecha_desde';
        $where[] = 'DATE(p.fecha_pedido) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['cliente_id'] !== '') {
            $where[] = 'p.cliente_id = :cliente_id';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        if ($filters['estado_id'] !== '') {
            $where[] = 'p.estado_id = :estado_id';
            $params[':estado_id'] = (int) $filters['estado_id'];
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        return [$whereSql, $params];
    }
}

if (!function_exists('pedidos_hora_bind_params')) {
    /**
     * Bind seguro para reusar en múltiples consultas.
     */
    function pedidos_hora_bind_params(PDOStatement $stmt, array $params): void
    {
        foreach ($params as $key => $value) {
            if (is_int($value)) {
                $stmt->bindValue($key, $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue($key, $value);
            }
        }
    }
}

if (!function_exists('pedidos_hora_normalize_filters')) {
    function pedidos_hora_normalize_filters(array $input): array
    {
        $defaultHasta = date('Y-m-d');
        $defaultDesde = date('Y-m-d', strtotime('-6 days'));

        $desde = pedidos_hora_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = pedidos_hora_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);

        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        $limitDays = isset($input['limit']) ? (int) $input['limit'] : 90;
        if (isset($input['limit_days'])) {
            $limitDays = (int) $input['limit_days'];
        }
        $limitDays = max(1, min($limitDays, 365));

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'cliente_id' => pedidos_hora_sanitize_numeric($input['cliente_id'] ?? null),
            'estado_id' => pedidos_hora_sanitize_numeric($input['estado_id'] ?? null),
            'limit_days' => $limitDays,
        ];
    }
}

if (!function_exists('pedidos_hora_sanitize_date')) {
    function pedidos_hora_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('pedidos_hora_sanitize_numeric')) {
    function pedidos_hora_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('pedidos_hora_trim_time')) {
    function pedidos_hora_trim_time($value): ?string
    {
        if ($value === null) {
            return null;
        }
        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }
        return substr($value, 0, 5);
    }
}
