<?php

declare(strict_types=1);

if (!function_exists('pedidos_fetch_combos')) {
    /**
     * Obtiene listas para los filtros (clientes, estados).
     */
    function pedidos_fetch_combos(PDO $pdo): array
    {
        $clientes = $pdo->query('SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social LIMIT 400')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $estados = $pdo->query('SELECT id, code, nombre FROM so_pedido_estado ORDER BY orden ASC, nombre ASC')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return [
            'clientes' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'nombre' => trim((string) ($row['razon_social'] ?? 'Sin cliente')) ?: 'Sin cliente',
                ];
            }, $clientes),
            'estados' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'code' => $row['code'] ?? null,
                    'nombre' => $row['nombre'] ?? 'Sin estado',
                ];
            }, $estados),
        ];
    }
}

if (!function_exists('pedidos_fetch_data')) {
    /**
     * Devuelve datos del reporte de pedidos según filtros y límite.
     */
    function pedidos_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = pedidos_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        $where = [];
        $params = [];

        $where[] = 'DATE(p.fecha_pedido) >= :fecha_desde';
        $where[] = 'DATE(p.fecha_pedido) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['cliente_id'] !== '') {
            $where[] = 'p.cliente_id = :cliente_id';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        if ($filters['estado_id'] !== '') {
            $where[] = 'p.estado_id = :estado_id';
            $params[':estado_id'] = (int) $filters['estado_id'];
        }

        if ($filters['codigo'] !== '') {
            $where[] = 'p.codigo LIKE :codigo';
            $params[':codigo'] = '%' . $filters['codigo'] . '%';
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = '
            SELECT
                p.id,
                p.codigo,
                p.fecha_pedido,
                p.created_at,
                p.cliente_ref,
                p.cliente_id,
                cli.razon_social AS cliente_nombre,
                est.id AS estado_id,
                est.code AS estado_code,
                est.nombre AS estado_nombre,
                agg.destinos,
                agg.destinatarios,
                agg.items,
                agg.expected_uv,
                agg.expected_uc,
                agg.shipped_uv,
                agg.shipped_uc
            FROM so_pedido p
            LEFT JOIN para_clientes cli ON cli.id = p.cliente_id
            LEFT JOIN so_pedido_estado est ON est.id = p.estado_id
            LEFT JOIN (
                SELECT
                    pd.pedido_id,
                    COUNT(DISTINCT pd.id) AS destinos,
                    COUNT(DISTINCT pd.destinatario_id) AS destinatarios,
                    COUNT(pdi.id) AS items,
                    SUM(COALESCE(pdi.expected_uv, 0)) AS expected_uv,
                    SUM(COALESCE(pdi.expected_uc, 0)) AS expected_uc,
                    SUM(COALESCE(pdi.shipped_uv, 0)) AS shipped_uv,
                    SUM(COALESCE(pdi.shipped_uc, 0)) AS shipped_uc
                FROM so_pedido_dest pd
                LEFT JOIN so_pedido_dest_item pdi ON pdi.pedido_dest_id = pd.id
                GROUP BY pd.pedido_id
            ) agg ON agg.pedido_id = p.id
            ' . $whereSql . '
            ORDER BY p.fecha_pedido DESC, p.id DESC
            LIMIT :limit_plus
        ';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_plus', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();

        $rawRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $truncated = false;
        if (count($rawRows) > $limit) {
            $rawRows = array_slice($rawRows, 0, $limit);
            $truncated = true;
        }

        $rows = array_map('pedidos_format_row', $rawRows);
        $summary = pedidos_build_summary($rows, $filters);
        $aggregates = pedidos_build_aggregates($rows);

        return [
            'filters' => $filters,
            'rows' => $rows,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('pedidos_build_summary')) {
    /**
     * Calcula totales/resumen para el front.
     */
    function pedidos_build_summary(array $rows, array $filters): array
    {
        if (!$rows) {
            return [
                'total_pedidos' => 0,
                'total_clientes' => 0,
                'total_destinatarios' => 0,
                'total_destinos' => 0,
                'total_items' => 0,
                'total_expected_uv' => 0,
                'total_expected_uc' => 0,
                'total_shipped_uv' => 0,
                'total_shipped_uc' => 0,
                'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
            ];
        }

        $clientes = [];
        $destinatarios = 0;
        $destinos = 0;
        $items = 0;
        $expectedUv = 0;
        $expectedUc = 0;
        $shippedUv = 0;
        $shippedUc = 0;

        foreach ($rows as $row) {
            if (!empty($row['cliente_id'])) {
                $clientes[(int) $row['cliente_id']] = true;
            }
            $destinatarios += (int) ($row['destinatarios'] ?? 0);
            $destinos += (int) ($row['destinos'] ?? 0);
            $items += (int) ($row['items'] ?? 0);
            $expectedUv += (int) ($row['expected_uv'] ?? 0);
            $expectedUc += (int) ($row['expected_uc'] ?? 0);
            $shippedUv += (int) ($row['shipped_uv'] ?? 0);
            $shippedUc += (int) ($row['shipped_uc'] ?? 0);
        }

        return [
            'total_pedidos' => count($rows),
            'total_clientes' => count($clientes),
            'total_destinatarios' => $destinatarios,
            'total_destinos' => $destinos,
            'total_items' => $items,
            'total_expected_uv' => $expectedUv,
            'total_expected_uc' => $expectedUc,
            'total_shipped_uv' => $shippedUv,
            'total_shipped_uc' => $shippedUc,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('pedidos_build_aggregates')) {
    /**
     * Top de clientes y estados basado en las filas devueltas.
     */
    function pedidos_build_aggregates(array $rows): array
    {
        if (!$rows) {
            return [
                'clientes' => [],
                'estados' => [],
            ];
        }

        $clientes = [];
        $estados = [];

        foreach ($rows as $row) {
            $clienteKey = (string) ($row['cliente_id'] ?? 0);
            if (!isset($clientes[$clienteKey])) {
                $clientes[$clienteKey] = [
                    'key' => $clienteKey,
                    'label' => $row['cliente_nombre'] ?? ($row['cliente_ref'] ?? 'Sin cliente'),
                    'rows' => 0,
                    'destinatarios' => 0,
                    'expected_uc' => 0,
                    'shipped_uc' => 0,
                ];
            }
            $clientes[$clienteKey]['rows']++;
            $clientes[$clienteKey]['destinatarios'] += (int) ($row['destinatarios'] ?? 0);
            $clientes[$clienteKey]['expected_uc'] += (int) ($row['expected_uc'] ?? 0);
            $clientes[$clienteKey]['shipped_uc'] += (int) ($row['shipped_uc'] ?? 0);

            $estadoKey = (string) ($row['estado_id'] ?? 0);
            if (!isset($estados[$estadoKey])) {
                $estados[$estadoKey] = [
                    'key' => $estadoKey,
                    'label' => $row['estado_nombre'] ?? 'Sin estado',
                    'code' => $row['estado_code'] ?? null,
                    'rows' => 0,
                    'expected_uc' => 0,
                    'shipped_uc' => 0,
                ];
            }
            $estados[$estadoKey]['rows']++;
            $estados[$estadoKey]['expected_uc'] += (int) ($row['expected_uc'] ?? 0);
            $estados[$estadoKey]['shipped_uc'] += (int) ($row['shipped_uc'] ?? 0);
        }

        $clientesList = array_values($clientes);
        usort($clientesList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return $b['expected_uc'] <=> $a['expected_uc'];
        });
        $clientesList = array_slice($clientesList, 0, 10);

        $estadosList = array_values($estados);
        usort($estadosList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });

        return [
            'clientes' => $clientesList,
            'estados' => $estadosList,
        ];
    }
}

if (!function_exists('pedidos_format_row')) {
    /**
     * Formatea fila cruda de la base para consumo del front.
     */
    function pedidos_format_row(array $row): array
    {
        $fechaPedido = $row['fecha_pedido'] ?? null;
        $fecha = $fechaPedido ? substr((string) $fechaPedido, 0, 10) : null;
        $hora = null;
        if ($fechaPedido) {
            $value = (string) $fechaPedido;
            if (strlen($value) >= 16) {
                $hora = substr($value, 11, 5);
            }
        }

        $clienteNombre = trim((string) ($row['cliente_nombre'] ?? ''));
        if ($clienteNombre === '' && !empty($row['cliente_ref'])) {
            $clienteNombre = (string) $row['cliente_ref'];
        }

        $expectedUv = (int) ($row['expected_uv'] ?? 0);
        $expectedUc = (int) ($row['expected_uc'] ?? 0);
        $shippedUv = (int) ($row['shipped_uv'] ?? 0);
        $shippedUc = (int) ($row['shipped_uc'] ?? 0);

        $avanceUvPct = $expectedUv > 0 ? round(($shippedUv / $expectedUv) * 100, 1) : null;
        $avanceUcPct = $expectedUc > 0 ? round(($shippedUc / $expectedUc) * 100, 1) : null;

        return [
            'pedido_id' => (int) ($row['id'] ?? 0),
            'pedido_codigo' => $row['codigo'] ?? '',
            'fecha_pedido' => $fecha,
            'hora_pedido' => $hora,
            'cliente_id' => isset($row['cliente_id']) ? (int) $row['cliente_id'] : null,
            'cliente_nombre' => $clienteNombre !== '' ? $clienteNombre : 'Sin cliente',
            'cliente_ref' => $row['cliente_ref'] ?? null,
            'estado_id' => isset($row['estado_id']) ? (int) $row['estado_id'] : null,
            'estado_code' => $row['estado_code'] ?? null,
            'estado_nombre' => $row['estado_nombre'] ?? 'Sin estado',
            'destinos' => (int) ($row['destinos'] ?? 0),
            'destinatarios' => (int) ($row['destinatarios'] ?? 0),
            'items' => (int) ($row['items'] ?? 0),
            'expected_uv' => $expectedUv,
            'expected_uc' => $expectedUc,
            'shipped_uv' => $shippedUv,
            'shipped_uc' => $shippedUc,
            'avance_uv_pct' => $avanceUvPct,
            'avance_uc_pct' => $avanceUcPct,
            'created_at' => $row['created_at'] ?? null,
        ];
    }
}

if (!function_exists('pedidos_normalize_filters')) {
    /**
     * Limpia los filtros recibidos desde GET.
     */
    function pedidos_normalize_filters(array $input): array
    {
        $defaultDesde = date('Y-m-d', strtotime('-30 days'));
        $defaultHasta = date('Y-m-d');

        $desde = pedidos_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = pedidos_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);

        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'cliente_id' => pedidos_sanitize_numeric($input['cliente_id'] ?? null),
            'estado_id' => pedidos_sanitize_numeric($input['estado_id'] ?? null),
            'codigo' => pedidos_sanitize_string($input['codigo'] ?? '', 80),
        ];
    }
}

if (!function_exists('pedidos_sanitize_date')) {
    function pedidos_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\\d{2}-[01]\\d-[0-3]\\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('pedidos_sanitize_numeric')) {
    function pedidos_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('pedidos_sanitize_string')) {
    function pedidos_sanitize_string(?string $value, int $maxLength): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        if (strlen($value) > $maxLength) {
            $value = substr($value, 0, $maxLength);
        }
        return $value;
    }
}
