<?php

declare(strict_types=1);

if (!function_exists('salidas_fetch_combos')) {
    /**
     * Obtiene catálogos para los filtros del reporte de salidas.
     */
    function salidas_fetch_combos(PDO $pdo): array
    {
        $depositos = $pdo->query('SELECT id, code, nombre FROM wh_deposito ORDER BY nombre LIMIT 200')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $estados = $pdo->query('SELECT code, nombre FROM so_embarque_estado WHERE activo = 1 ORDER BY orden, nombre')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $moviles = $pdo->query('SELECT id, chapa FROM para_moviles WHERE activo = 1 ORDER BY chapa')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $choferes = $pdo->query('SELECT id, nombre FROM para_choferes WHERE activo = 1 ORDER BY nombre')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return [
            'depositos' => array_map(static function (array $row): array {
                $nombre = trim((string) ($row['nombre'] ?? ''));
                $code = trim((string) ($row['code'] ?? ''));
                if ($nombre === '' && $code !== '') {
                    $nombre = $code;
                }
                if ($nombre === '') {
                    $nombre = 'Sin depósito';
                }
                return [
                    'id' => (int) $row['id'],
                    'nombre' => $nombre,
                    'code' => $code !== '' ? $code : null,
                ];
            }, $depositos),
            'estados' => array_map(static function (array $row): array {
                $code = trim((string) ($row['code'] ?? ''));
                return [
                    'code' => $code,
                    'nombre' => salidas_format_estado_label($code),
                ];
            }, $estados),
            'moviles' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'label' => trim((string) ($row['chapa'] ?? '')) ?: ('ID ' . (int) $row['id']),
                ];
            }, $moviles),
            'choferes' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'nombre' => trim((string) ($row['nombre'] ?? '')) ?: 'Sin chofer',
                ];
            }, $choferes),
        ];
    }
}

if (!function_exists('salidas_fetch_data')) {
    /**
     * Devuelve los datos del reporte de salidas aplicando filtros y límite.
     */
    function salidas_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = salidas_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        $where = [];
        $params = [];

        $where[] = 'DATE(e.creado_at) >= :fecha_desde';
        $where[] = 'DATE(e.creado_at) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['deposito_id'] !== '') {
            $where[] = 'e.deposito_id = :deposito_id';
            $params[':deposito_id'] = (int) $filters['deposito_id'];
        }

        if ($filters['estado'] !== '') {
            $where[] = 'est.code = :estado_code';
            $params[':estado_code'] = $filters['estado'];
        }

        if ($filters['movil_id'] !== '') {
            $where[] = 'e.movil_id = :movil_id';
            $params[':movil_id'] = (int) $filters['movil_id'];
        }

        if ($filters['chofer_id'] !== '') {
            $where[] = 'e.chofer_id = :chofer_id';
            $params[':chofer_id'] = (int) $filters['chofer_id'];
        }

        if ($filters['codigo'] !== '') {
            $where[] = 'e.codigo LIKE :codigo';
            $params[':codigo'] = '%' . $filters['codigo'] . '%';
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = '
            SELECT
                e.id,
                e.codigo,
                e.creado_at,
                e.llegada_at,
                e.carga_inicio_at,
                e.carga_fin_at,
                e.salida_at,
                e.observacion,
                e.ayudantes_cant,
                e.temp_salida_c,
                e.km_inicial,
                e.deposito_id,
                e.movil_id,
                e.chofer_id,
                e.updated_at,
                est.code AS estado_code,
                est.nombre AS estado_nombre,
                dep.code AS deposito_code,
                dep.nombre AS deposito_nombre,
                mov.chapa AS movil_chapa,
                cho.nombre AS chofer_nombre,
                dest.total_destinos,
                pre.total_preembarques,
                CASE WHEN e.llegada_at IS NOT NULL AND e.carga_inicio_at IS NOT NULL
                    THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.carga_inicio_at)
                    ELSE NULL
                END AS tiempo_espera_min,
                CASE WHEN e.carga_inicio_at IS NOT NULL AND e.carga_fin_at IS NOT NULL
                    THEN TIMESTAMPDIFF(MINUTE, e.carga_inicio_at, e.carga_fin_at)
                    ELSE NULL
                END AS tiempo_carga_min,
                CASE WHEN e.llegada_at IS NOT NULL AND e.salida_at IS NOT NULL
                    THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.salida_at)
                    ELSE NULL
                END AS tiempo_planta_min
            FROM so_embarque e
            LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
            LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
            LEFT JOIN para_moviles mov ON mov.id = e.movil_id
            LEFT JOIN para_choferes cho ON cho.id = e.chofer_id
            LEFT JOIN (
                SELECT embarque_id, COUNT(*) AS total_destinos
                FROM so_embarque_parada
                GROUP BY embarque_id
            ) dest ON dest.embarque_id = e.id
            LEFT JOIN (
                SELECT embarque_id, COUNT(*) AS total_preembarques
                FROM so_embarque_pre
                GROUP BY embarque_id
            ) pre ON pre.embarque_id = e.id
            ' . $whereSql . '
            ORDER BY e.creado_at DESC, e.id DESC
            LIMIT :limit_plus
        ';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_plus', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();

        $rawRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $truncated = false;
        if (count($rawRows) > $limit) {
            $rawRows = array_slice($rawRows, 0, $limit);
            $truncated = true;
        }

        $rows = array_map('salidas_format_row', $rawRows);
        $summary = salidas_build_summary($rows, $filters);
        $aggregates = salidas_build_aggregates($rows);

        return [
            'filters' => $filters,
            'rows' => $rows,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('salidas_build_summary')) {
    /**
     * Calcula totales generales del reporte.
     */
    function salidas_build_summary(array $rows, array $filters): array
    {
        if (!$rows) {
            return [
                'total_salidas' => 0,
                'total_depositos' => 0,
                'total_moviles' => 0,
                'total_choferes' => 0,
                'total_destinos' => 0,
                'total_preembarques' => 0,
                'avg_espera_min' => null,
                'avg_carga_min' => null,
                'avg_planta_min' => null,
                'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
            ];
        }

        $depositos = [];
        $moviles = [];
        $choferes = [];
        $totalDestinos = 0;
        $totalPreembarques = 0;
        $espera = [];
        $carga = [];
        $planta = [];

        foreach ($rows as $row) {
            if (!empty($row['deposito_id'])) {
                $depositos[(int) $row['deposito_id']] = true;
            }
            if (!empty($row['movil_id'])) {
                $moviles[(int) $row['movil_id']] = true;
            }
            if (!empty($row['chofer_id'])) {
                $choferes[(int) $row['chofer_id']] = true;
            }
            $totalDestinos += (int) ($row['destinos'] ?? 0);
            $totalPreembarques += (int) ($row['preembarques'] ?? 0);

            if (isset($row['tiempo_espera_min']) && $row['tiempo_espera_min'] !== null) {
                $espera[] = (int) $row['tiempo_espera_min'];
            }
            if (isset($row['tiempo_carga_min']) && $row['tiempo_carga_min'] !== null) {
                $carga[] = (int) $row['tiempo_carga_min'];
            }
            if (isset($row['tiempo_planta_min']) && $row['tiempo_planta_min'] !== null) {
                $planta[] = (int) $row['tiempo_planta_min'];
            }
        }

        $avgEspera = $espera ? round(array_sum($espera) / count($espera), 1) : null;
        $avgCarga = $carga ? round(array_sum($carga) / count($carga), 1) : null;
        $avgPlanta = $planta ? round(array_sum($planta) / count($planta), 1) : null;

        return [
            'total_salidas' => count($rows),
            'total_depositos' => count($depositos),
            'total_moviles' => count($moviles),
            'total_choferes' => count($choferes),
            'total_destinos' => $totalDestinos,
            'total_preembarques' => $totalPreembarques,
            'avg_espera_min' => $avgEspera,
            'avg_carga_min' => $avgCarga,
            'avg_planta_min' => $avgPlanta,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('salidas_build_aggregates')) {
    /**
     * Construye agregados de depósitos, móviles y estados.
     */
    function salidas_build_aggregates(array $rows): array
    {
        if (!$rows) {
            return [
                'depositos' => [],
                'moviles' => [],
                'estados' => [],
            ];
        }

        $depositos = [];
        $moviles = [];
        $estados = [];

        foreach ($rows as $row) {
            $depositoKey = (string) ($row['deposito_id'] ?? 0);
            if (!isset($depositos[$depositoKey])) {
                $depositos[$depositoKey] = [
                    'key' => $depositoKey,
                    'label' => $row['deposito_nombre'] ?? ($row['deposito_code'] ?? 'Sin depósito'),
                    'rows' => 0,
                    'destinos' => 0,
                    'preembarques' => 0,
                ];
            }
            $depositos[$depositoKey]['rows']++;
            $depositos[$depositoKey]['destinos'] += (int) ($row['destinos'] ?? 0);
            $depositos[$depositoKey]['preembarques'] += (int) ($row['preembarques'] ?? 0);

            $movilKey = (string) ($row['movil_id'] ?? 0);
            if (!isset($moviles[$movilKey])) {
                $movilLabel = trim((string) ($row['movil_chapa'] ?? ''));
                if ($movilLabel === '') {
                    $movilLabel = 'Sin móvil';
                }
                $moviles[$movilKey] = [
                    'key' => $movilKey,
                    'label' => $movilLabel,
                    'rows' => 0,
                    'destinos' => 0,
                    'planta_min' => 0,
                ];
            }
            $moviles[$movilKey]['rows']++;
            $moviles[$movilKey]['destinos'] += (int) ($row['destinos'] ?? 0);
            $moviles[$movilKey]['planta_min'] += (int) ($row['tiempo_planta_min'] ?? 0);

            $estadoCode = (string) ($row['estado_code'] ?? '');
            if ($estadoCode === '') {
                $estadoCode = 'SIN_ESTADO';
            }
            if (!isset($estados[$estadoCode])) {
                $estados[$estadoCode] = [
                    'code' => $estadoCode === 'SIN_ESTADO' ? null : $estadoCode,
                    'label' => $estadoCode === 'SIN_ESTADO' ? 'Sin estado' : salidas_format_estado_label($estadoCode),
                    'rows' => 0,
                    'destinos' => 0,
                    'preembarques' => 0,
                ];
            }
            $estados[$estadoCode]['rows']++;
            $estados[$estadoCode]['destinos'] += (int) ($row['destinos'] ?? 0);
            $estados[$estadoCode]['preembarques'] += (int) ($row['preembarques'] ?? 0);
        }

        $depositosList = array_values($depositos);
        usort($depositosList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            $cmp = $b['destinos'] <=> $a['destinos'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });
        $depositosList = array_slice($depositosList, 0, 10);

        $movilesList = array_values($moviles);
        usort($movilesList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return $b['destinos'] <=> $a['destinos'];
        });
        $movilesList = array_slice($movilesList, 0, 10);

        $estadosList = array_values($estados);
        usort($estadosList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });

        return [
            'depositos' => $depositosList,
            'moviles' => $movilesList,
            'estados' => $estadosList,
        ];
    }
}

if (!function_exists('salidas_format_row')) {
    /**
     * Normaliza la fila cruda del SELECT para el front.
     */
    function salidas_format_row(array $row): array
    {
        $creadoAt = $row['creado_at'] ?? null;
        $fecha = $creadoAt ? substr((string) $creadoAt, 0, 10) : null;
        $hora = null;
        if ($creadoAt) {
            $value = (string) $creadoAt;
            if (strlen($value) >= 16) {
                $hora = substr($value, 11, 5);
            }
        }

        $llegadaAt = $row['llegada_at'] ?? null;
        $cargaInicio = $row['carga_inicio_at'] ?? null;
        $cargaFin = $row['carga_fin_at'] ?? null;
        $salidaAt = $row['salida_at'] ?? null;

        $depositoNombre = trim((string) ($row['deposito_nombre'] ?? ''));
        if ($depositoNombre === '' && !empty($row['deposito_code'])) {
            $depositoNombre = (string) $row['deposito_code'];
        }
        if ($depositoNombre === '') {
            $depositoNombre = 'Sin depósito';
        }

        $estadoCode = trim((string) ($row['estado_code'] ?? ''));
        $estadoNombre = $estadoCode !== '' ? salidas_format_estado_label($estadoCode) : 'Sin estado';

        $observacion = $row['observacion'] ?? null;
        if ($observacion !== null) {
            $observacion = trim((string) $observacion);
            if ($observacion === '') {
                $observacion = null;
            }
        }

        $tempSalida = $row['temp_salida_c'] ?? null;
        if ($tempSalida !== null && !is_numeric($tempSalida)) {
            $tempSalida = null;
        }

        $kmInicial = $row['km_inicial'] ?? null;
        if ($kmInicial !== null && !is_numeric($kmInicial)) {
            $kmInicial = null;
        }

        return [
            'embarque_id' => (int) ($row['id'] ?? 0),
            'codigo' => $row['codigo'] ?? '',
            'fecha_creado' => $fecha,
            'hora_creado' => $hora,
            'llegada_hora' => $llegadaAt !== null ? substr((string) $llegadaAt, 11, 5) : null,
            'carga_inicio_hora' => $cargaInicio !== null ? substr((string) $cargaInicio, 11, 5) : null,
            'carga_fin_hora' => $cargaFin !== null ? substr((string) $cargaFin, 11, 5) : null,
            'salida_hora' => $salidaAt !== null ? substr((string) $salidaAt, 11, 5) : null,
            'deposito_id' => isset($row['deposito_id']) ? (int) $row['deposito_id'] : null,
            'deposito_nombre' => $depositoNombre,
            'deposito_code' => $row['deposito_code'] ?? null,
            'estado_code' => $estadoCode !== '' ? $estadoCode : null,
            'estado_nombre' => $estadoNombre,
            'movil_id' => isset($row['movil_id']) ? (int) $row['movil_id'] : null,
            'movil_chapa' => $row['movil_chapa'] ?? null,
            'chofer_id' => isset($row['chofer_id']) ? (int) $row['chofer_id'] : null,
            'chofer_nombre' => $row['chofer_nombre'] ?? null,
            'destinos' => (int) ($row['total_destinos'] ?? 0),
            'preembarques' => (int) ($row['total_preembarques'] ?? 0),
            'tiempo_espera_min' => isset($row['tiempo_espera_min']) ? ($row['tiempo_espera_min'] !== null ? (int) $row['tiempo_espera_min'] : null) : null,
            'tiempo_carga_min' => isset($row['tiempo_carga_min']) ? ($row['tiempo_carga_min'] !== null ? (int) $row['tiempo_carga_min'] : null) : null,
            'tiempo_planta_min' => isset($row['tiempo_planta_min']) ? ($row['tiempo_planta_min'] !== null ? (int) $row['tiempo_planta_min'] : null) : null,
            'ayudantes_cant' => isset($row['ayudantes_cant']) ? (int) $row['ayudantes_cant'] : null,
            'temp_salida_c' => $tempSalida !== null ? (float) $tempSalida : null,
            'km_inicial' => $kmInicial !== null ? (float) $kmInicial : null,
            'observacion' => $observacion,
            'updated_at' => $row['updated_at'] ?? null,
        ];
    }
}

if (!function_exists('salidas_normalize_filters')) {
    /**
     * Normaliza los filtros de entrada.
     */
    function salidas_normalize_filters(array $input): array
    {
        $defaultDesde = date('Y-m-d', strtotime('-30 days'));
        $defaultHasta = date('Y-m-d');

        $desde = salidas_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = salidas_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);

        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'deposito_id' => salidas_sanitize_numeric($input['deposito_id'] ?? null),
            'estado' => salidas_sanitize_estado($input['estado'] ?? null),
            'movil_id' => salidas_sanitize_numeric($input['movil_id'] ?? null),
            'chofer_id' => salidas_sanitize_numeric($input['chofer_id'] ?? null),
            'codigo' => salidas_sanitize_string($input['codigo'] ?? '', 80),
        ];
    }
}

if (!function_exists('salidas_sanitize_date')) {
    function salidas_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('salidas_sanitize_numeric')) {
    function salidas_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('salidas_sanitize_string')) {
    function salidas_sanitize_string(?string $value, int $maxLength): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        if (strlen($value) > $maxLength) {
            $value = substr($value, 0, $maxLength);
        }
        return $value;
    }
}

if (!function_exists('salidas_sanitize_estado')) {
    function salidas_sanitize_estado(?string $value): string
    {
        $value = strtoupper(trim((string) $value));
        if ($value === '') {
            return '';
        }
        if (!preg_match('/^[A-Z_]{3,40}$/', $value)) {
            return '';
        }
        return $value;
    }
}

if (!function_exists('salidas_format_estado_label')) {
    function salidas_format_estado_label(string $code): string
    {
        static $map = [
            'CREADO' => 'Creado',
            'PROGRAMADO' => 'Programado',
            'PLANIFICADO' => 'Planificado',
            'EN_PLANTA' => 'En planta',
            'CARGA_INICIADA' => 'Carga iniciada',
            'CARGA_FINALIZADA' => 'Carga finalizada',
            'EN_TRANSITO' => 'En tránsito',
            'CERRADO' => 'Cerrado',
            'FINALIZADO' => 'Finalizado',
            'ANULADO' => 'Anulado',
        ];

        $code = strtoupper(trim($code));
        if ($code === '') {
            return 'Sin estado';
        }
        if (isset($map[$code])) {
            return $map[$code];
        }
        $label = str_replace('_', ' ', strtolower($code));
        return ucwords($label);
    }
}
