<?php

declare(strict_types=1);

require_once dirname(__DIR__) . '/InventarioReport.php';
require_once __DIR__ . '/DistribucionReport.php';

if (!function_exists('servicio_cliente_fetch_filters')) {
    /**
     * Devuelve catálogos reutilizados desde el reporte de distribución.
     */
    function servicio_cliente_fetch_filters(PDO $pdo): array
    {
        return distribucion_fetch_filters($pdo);
    }
}

if (!function_exists('servicio_cliente_normalize_filters')) {
    /**
     * @param array<string, mixed> $raw
     * @return array<string, mixed>
     */
    function servicio_cliente_normalize_filters(array $raw): array
    {
        $defaultHasta = date('Y-m-d');
        $defaultDesde = date('Y-m-d', strtotime('-29 days'));

        $fechaDesde = inventario_sanitize_date($raw['fecha_desde'] ?? null);
        $fechaHasta = inventario_sanitize_date($raw['fecha_hasta'] ?? null);
        if ($fechaDesde === '') {
            $fechaDesde = $defaultDesde;
        }
        if ($fechaHasta === '') {
            $fechaHasta = $defaultHasta;
        }
        if ($fechaDesde > $fechaHasta) {
            [$fechaDesde, $fechaHasta] = [$fechaHasta, $fechaDesde];
        }

        $toleranciaMin = isset($raw['tolerancia_min']) ? (int) $raw['tolerancia_min'] : 60;
        if ($toleranciaMin <= 0) {
            $toleranciaMin = 60;
        }

        $metaOtif = isset($raw['meta_otif']) && $raw['meta_otif'] !== '' ? (float) $raw['meta_otif'] : 98.0;
        $metaReclamaciones = isset($raw['meta_reclamaciones']) && $raw['meta_reclamaciones'] !== ''
            ? max(0.0, (float) $raw['meta_reclamaciones'])
            : 1.0;
        $metaSatisfaccion = isset($raw['meta_satisfaccion']) && $raw['meta_satisfaccion'] !== ''
            ? (float) $raw['meta_satisfaccion']
            : 95.0;

        return [
            'fecha_desde'        => $fechaDesde,
            'fecha_hasta'        => $fechaHasta,
            'cliente_id'         => distribucion_sanitize_id($raw['cliente_id'] ?? null),
            'deposito_id'        => distribucion_sanitize_id($raw['deposito_id'] ?? null),
            'movil_id'           => distribucion_sanitize_id($raw['movil_id'] ?? null),
            'tolerancia_min'     => $toleranciaMin,
            'meta_otif'          => $metaOtif,
            'meta_reclamaciones' => $metaReclamaciones,
            'meta_satisfaccion'  => $metaSatisfaccion,
        ];
    }
}

if (!function_exists('servicio_cliente_format_periodo')) {
    /**
     * @param array<string, mixed> $filters
     */
    function servicio_cliente_format_periodo(array $filters): string
    {
        return $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'];
    }
}

if (!function_exists('servicio_cliente_fetch_data')) {
    /**
     * @param array<string, mixed> $rawFilters
     * @return array<string, mixed>
     */
    function servicio_cliente_fetch_data(PDO $pdo, array $rawFilters): array
    {
        $filters = servicio_cliente_normalize_filters($rawFilters);

        $otif = servicio_cliente_metric_otif($pdo, $filters);
        $reclamaciones = servicio_cliente_metric_reclamaciones($pdo, $filters, $otif);
        $satisfaccion = servicio_cliente_metric_satisfaccion($pdo, $filters);

        $metrics = [];
        $metrics[] = servicio_cliente_build_metric([
            'clave'        => 'otif',
            'indicador'    => 'OTIF (On Time In Full)',
            'formula'      => '(Pedidos completos y a tiempo / Total de pedidos) × 100',
            'meta'         => $filters['meta_otif'],
            'valor'        => $otif['porcentaje'],
            'unidad'       => '%',
            'observaciones'=> $otif['observacion'] ?? '',
            'direccion'    => 'max',
            'detalle'      => $otif,
        ]);

        $metrics[] = servicio_cliente_build_metric([
            'clave'        => 'reclamaciones',
            'indicador'    => 'Tasa de reclamaciones',
            'formula'      => '(Reclamaciones / Pedidos entregados) × 100',
            'meta'         => $filters['meta_reclamaciones'],
            'valor'        => $reclamaciones['porcentaje'],
            'unidad'       => '%',
            'observaciones'=> $reclamaciones['observacion'] ?? '',
            'direccion'    => 'min',
            'detalle'      => $reclamaciones,
        ]);

        $metrics[] = servicio_cliente_build_metric([
            'clave'        => 'satisfaccion',
            'indicador'    => 'Nivel de satisfacción del cliente',
            'formula'      => 'Encuesta post-entrega',
            'meta'         => $filters['meta_satisfaccion'],
            'valor'        => $satisfaccion['porcentaje'],
            'unidad'       => '%',
            'observaciones'=> $satisfaccion['observacion'] ?? '',
            'direccion'    => 'max',
            'detalle'      => $satisfaccion,
        ]);

        return [
            'filters' => $filters,
            'periodo' => servicio_cliente_format_periodo($filters),
            'metrics' => $metrics,
            'summary' => servicio_cliente_build_summary($otif, $reclamaciones, $satisfaccion),
        ];
    }
}

if (!function_exists('servicio_cliente_metric_otif')) {
    /**
     * @param array<string, mixed> $filters
     * @return array<string, mixed>
     */
    function servicio_cliente_metric_otif(PDO $pdo, array $filters): array
    {
        $distFilters = [
            'fecha_desde'      => $filters['fecha_desde'],
            'fecha_hasta'      => $filters['fecha_hasta'],
            'cliente_id'       => $filters['cliente_id'],
            'deposito_id'      => $filters['deposito_id'],
            'movil_id'         => $filters['movil_id'],
            'tolerancia_min'   => $filters['tolerancia_min'],
            'combustible_meta' => 0,
        ];

        $detalle = distribucion_metric_entregas_tiempo($pdo, $distFilters);

        $totalPedidos = (int) ($detalle['total_registros'] ?? 0);
        $entregas = (int) ($detalle['entregas'] ?? 0);
        $enPlazo = (int) ($detalle['en_plazo'] ?? 0);

        $base = $totalPedidos > 0 ? $totalPedidos : $entregas;
        $porcentaje = null;
        if ($base > 0) {
            $porcentaje = round(($enPlazo / $base) * 100, 2);
        }

        $observacion = $base > 0
            ? sprintf('%d pedidos a tiempo sobre %d evaluados', $enPlazo, $base)
            : 'Sin pedidos en el periodo seleccionado';

        return [
            'total_pedidos'      => $totalPedidos,
            'pedidos_entregados' => $entregas,
            'pedidos_otif'       => $enPlazo,
            'porcentaje'         => $porcentaje,
            'observacion'        => $observacion,
            'detalle'            => $detalle,
        ];
    }
}

if (!function_exists('servicio_cliente_metric_reclamaciones')) {
    /**
     * @param array<string, mixed> $filters
     * @param array<string, mixed> $otif
     * @return array<string, mixed>
     */
    function servicio_cliente_metric_reclamaciones(PDO $pdo, array $filters, array $otif): array
    {
        $entregas = (int) ($otif['pedidos_entregados'] ?? ($otif['detalle']['entregas'] ?? 0));

        $source = servicio_cliente_detect_reclamos_source($pdo);
        if ($source === null) {
            $deliveries = ['entregas' => $entregas];
            $fallback = distribucion_metric_devoluciones($pdo, [
                'fecha_desde'  => $filters['fecha_desde'],
                'fecha_hasta'  => $filters['fecha_hasta'],
                'cliente_id'   => $filters['cliente_id'],
                'deposito_id'  => $filters['deposito_id'],
                'movil_id'     => $filters['movil_id'],
            ], $deliveries);

            $observacion = $fallback['observacion'] ?? 'Sin datos de devoluciones';
            if (($fallback['devoluciones'] ?? 0) > 0) {
                $observacion .= ' · se utilizan devoluciones como proxy de reclamaciones';
            } else {
                $observacion .= ' · no hay fuente de reclamaciones configurada';
            }

            return [
                'reclamos'     => (int) ($fallback['devoluciones'] ?? 0),
                'entregas'     => $entregas,
                'porcentaje'   => $fallback['porcentaje'] ?? null,
                'observacion'  => $observacion,
                'fuente'       => 'devoluciones',
            ];
        }

        $table = $source['table'];
        $alias = 'recl';

        $where = [
            'DATE(' . $alias . '.`' . $source['date_column'] . '`) >= :fecha_desde',
            'DATE(' . $alias . '.`' . $source['date_column'] . '`) <= :fecha_hasta',
        ];
        $params = [
            ':fecha_desde' => $filters['fecha_desde'],
            ':fecha_hasta' => $filters['fecha_hasta'],
        ];

        $joins = [];
        if (!empty($source['deleted_column'])) {
            $where[] = $alias . '.`' . $source['deleted_column'] . '` IS NULL';
        }

        if ($filters['cliente_id'] !== '') {
            if (!empty($source['cliente_column'])) {
                $where[] = $alias . '.`' . $source['cliente_column'] . '` = :cliente_id';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            } elseif (!empty($source['pedido_column'])
                && inventario_has_table($pdo, 'so_pedido')
                && inventario_has_column($pdo, 'so_pedido', 'cliente_id')) {
                $joins[] = 'LEFT JOIN so_pedido sc_p ON sc_p.id = ' . $alias . '.`' . $source['pedido_column'] . '`';
                $where[] = 'sc_p.`cliente_id` = :cliente_id';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            }
        }

        if ($filters['deposito_id'] !== '') {
            if (!empty($source['deposito_column'])) {
                $where[] = $alias . '.`' . $source['deposito_column'] . '` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            } elseif (!empty($source['pedido_column'])
                && inventario_has_table($pdo, 'so_pedido')
                && inventario_has_column($pdo, 'so_pedido', 'deposito_id')) {
                if (!in_array('LEFT JOIN so_pedido sc_p ON sc_p.id = ' . $alias . '.`' . $source['pedido_column'] . '`', $joins, true)) {
                    $joins[] = 'LEFT JOIN so_pedido sc_p ON sc_p.id = ' . $alias . '.`' . $source['pedido_column'] . '`';
                }
                $where[] = 'sc_p.`deposito_id` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            }
        }

        $sql = 'SELECT COUNT(*) AS total FROM `' . $table . '` ' . $alias;
        if ($joins) {
            $sql .= ' ' . implode(' ', $joins);
        }
        if ($where) {
            $sql .= ' WHERE ' . implode(' AND ', $where);
        }

        $stmt = $pdo->prepare($sql);
        foreach ($params as $placeholder => $value) {
            if (is_int($value)) {
                $stmt->bindValue($placeholder, $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue($placeholder, (string) $value);
            }
        }
        $stmt->execute();
        $totalReclamos = (int) ($stmt->fetchColumn() ?: 0);

        $porcentaje = null;
        if ($entregas > 0) {
            $porcentaje = round(($totalReclamos / $entregas) * 100, 2);
        }

        $observacion = $entregas > 0
            ? sprintf('%d reclamaciones sobre %d entregas', $totalReclamos, $entregas)
            : 'Sin entregas para evaluar reclamaciones';

        return [
            'reclamos'     => $totalReclamos,
            'entregas'     => $entregas,
            'porcentaje'   => $porcentaje,
            'observacion'  => $observacion,
            'fuente'       => $table,
        ];
    }
}

if (!function_exists('servicio_cliente_metric_satisfaccion')) {
    /**
     * @param array<string, mixed> $filters
     * @return array<string, mixed>
     */
    function servicio_cliente_metric_satisfaccion(PDO $pdo, array $filters): array
    {
        $source = servicio_cliente_detect_satisfaccion_source($pdo);
        if ($source === null) {
            return [
                'encuestas'   => 0,
                'porcentaje'  => null,
                'promedio'    => null,
                'observacion' => 'Sin registros de encuestas de satisfacción',
            ];
        }

        $alias = 'enc';
        $table = $source['table'];

        $select = 'SELECT COUNT(*) AS total_encuestas, AVG(' . $alias . '.`' . $source['score_column'] . '`) AS avg_score, MAX(' . $alias . '.`' . $source['score_column'] . '`) AS max_score';
        if (!empty($source['max_column'])) {
            $select .= ', MAX(' . $alias . '.`' . $source['max_column'] . '`) AS scale_score';
        }

        $sql = $select . ' FROM `' . $table . '` ' . $alias;

        $joins = [];
        $where = [
            'DATE(' . $alias . '.`' . $source['date_column'] . '`) >= :fecha_desde',
            'DATE(' . $alias . '.`' . $source['date_column'] . '`) <= :fecha_hasta',
        ];
        $params = [
            ':fecha_desde' => $filters['fecha_desde'],
            ':fecha_hasta' => $filters['fecha_hasta'],
        ];

        if (!empty($source['deleted_column'])) {
            $where[] = $alias . '.`' . $source['deleted_column'] . '` IS NULL';
        }

        if ($filters['cliente_id'] !== '') {
            if (!empty($source['cliente_column'])) {
                $where[] = $alias . '.`' . $source['cliente_column'] . '` = :cliente_id';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            } elseif (!empty($source['pedido_column'])
                && inventario_has_table($pdo, 'so_pedido')
                && inventario_has_column($pdo, 'so_pedido', 'cliente_id')) {
                $joins[] = 'LEFT JOIN so_pedido sc_p ON sc_p.id = ' . $alias . '.`' . $source['pedido_column'] . '`';
                $where[] = 'sc_p.`cliente_id` = :cliente_id';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            }
        }

        if ($filters['deposito_id'] !== '') {
            if (!empty($source['deposito_column'])) {
                $where[] = $alias . '.`' . $source['deposito_column'] . '` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            } elseif (!empty($source['pedido_column'])
                && inventario_has_table($pdo, 'so_pedido')
                && inventario_has_column($pdo, 'so_pedido', 'deposito_id')) {
                if (!in_array('LEFT JOIN so_pedido sc_p ON sc_p.id = ' . $alias . '.`' . $source['pedido_column'] . '`', $joins, true)) {
                    $joins[] = 'LEFT JOIN so_pedido sc_p ON sc_p.id = ' . $alias . '.`' . $source['pedido_column'] . '`';
                }
                $where[] = 'sc_p.`deposito_id` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            }
        }

        if ($joins) {
            $sql .= ' ' . implode(' ', $joins);
        }
        if ($where) {
            $sql .= ' WHERE ' . implode(' AND ', $where);
        }

        $stmt = $pdo->prepare($sql);
        foreach ($params as $placeholder => $value) {
            if (is_int($value)) {
                $stmt->bindValue($placeholder, $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue($placeholder, (string) $value);
            }
        }
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        $totalEncuestas = (int) ($row['total_encuestas'] ?? 0);
        $avgScore = isset($row['avg_score']) && $row['avg_score'] !== null ? (float) $row['avg_score'] : null;
        $maxScore = isset($row['max_score']) && $row['max_score'] !== null ? (float) $row['max_score'] : null;
        $scaleScore = isset($row['scale_score']) && $row['scale_score'] !== null ? (float) $row['scale_score'] : null;

        $porcentaje = null;
        if ($avgScore !== null) {
            $scale = null;
            if ($scaleScore !== null && $scaleScore > 0.0) {
                $scale = $scaleScore;
            } elseif ($maxScore !== null && $maxScore > 0.0) {
                $scale = $maxScore;
            }

            if ($source['scale_hint'] === 'percent') {
                $porcentaje = round($avgScore, 2);
            } else {
                if ($scale === null || $scale <= 0.0) {
                    $scale = 5.0;
                }
                if ($avgScore > $scale && $scale < 100.0) {
                    $scale = $avgScore;
                }
                if ($scale > 100.0) {
                    $scale = 100.0;
                }
                $porcentaje = $scale > 0.0 ? round(($avgScore / $scale) * 100, 2) : null;
            }
        }

        $observacion = $totalEncuestas > 0
            ? sprintf('Promedio %.2f pts (%d encuestas)', $avgScore ?? 0.0, $totalEncuestas)
            : 'Sin respuestas de encuestas en el periodo';

        return [
            'encuestas'   => $totalEncuestas,
            'promedio'    => $avgScore,
            'porcentaje'  => $porcentaje,
            'observacion' => $observacion,
            'fuente'      => $table,
        ];
    }
}

if (!function_exists('servicio_cliente_build_metric')) {
    /**
     * @param array<string, mixed> $config
     * @return array<string, mixed>
     */
    function servicio_cliente_build_metric(array $config): array
    {
        $valor = isset($config['valor']) && $config['valor'] !== null ? (float) $config['valor'] : null;
        $meta = isset($config['meta']) && $config['meta'] !== null ? (float) $config['meta'] : null;
        $direccion = isset($config['direccion']) && $config['direccion'] === 'min' ? 'min' : 'max';
        $estado = servicio_cliente_indicator_estado($valor, $meta, $direccion);

        return [
            'clave'         => (string) ($config['clave'] ?? ''),
            'indicador'     => (string) ($config['indicador'] ?? ''),
            'formula'       => (string) ($config['formula'] ?? ''),
            'meta'          => $meta,
            'valor'         => $valor,
            'unidad'        => (string) ($config['unidad'] ?? ''),
            'estado'        => $estado,
            'observaciones' => (string) ($config['observaciones'] ?? ''),
            'direccion'     => $direccion,
            'detalle'       => $config['detalle'] ?? [],
        ];
    }
}

if (!function_exists('servicio_cliente_indicator_estado')) {
    function servicio_cliente_indicator_estado(?float $valor, ?float $meta, string $direccion): string
    {
        if ($valor === null || $meta === null) {
            return 'SIN_DATOS';
        }

        if ($direccion === 'min') {
            if ($valor <= $meta) {
                return 'OK';
            }
            if ($meta <= 0.0) {
                return 'ALERTA';
            }
            $ratio = $valor / $meta;
            if ($ratio <= 1.6) {
                return 'ALERTA';
            }
            return 'CRITICO';
        }

        if ($valor >= $meta) {
            return 'OK';
        }
        if ($meta <= 0.0) {
            return 'ALERTA';
        }
        $ratio = $valor / $meta;
        if ($ratio >= 0.4) {
            return 'ALERTA';
        }
        return 'CRITICO';
    }
}

if (!function_exists('servicio_cliente_build_summary')) {
    /**
     * @param array<string, mixed> $otif
     * @param array<string, mixed> $reclamaciones
     * @param array<string, mixed> $satisfaccion
     * @return array<string, mixed>
     */
    function servicio_cliente_build_summary(array $otif, array $reclamaciones, array $satisfaccion): array
    {
        return [
            'pedidos_totales'   => $otif['total_pedidos'] ?? 0,
            'pedidos_otif'      => $otif['pedidos_otif'] ?? 0,
            'otif_pct'          => $otif['porcentaje'] ?? null,
            'reclamos'          => $reclamaciones['reclamos'] ?? 0,
            'reclamos_pct'      => $reclamaciones['porcentaje'] ?? null,
            'encuestas'         => $satisfaccion['encuestas'] ?? 0,
            'satisfaccion_pct'  => $satisfaccion['porcentaje'] ?? null,
        ];
    }
}

if (!function_exists('servicio_cliente_detect_reclamos_source')) {
    /**
     * @return array<string, string|null>|null
     */
    function servicio_cliente_detect_reclamos_source(PDO $pdo): ?array
    {
        $tables = [
            'so_reclamos',
            'so_reclamaciones',
            'cliente_reclamos',
            'clientes_reclamos',
            'reclamos_cliente',
            'reclamos',
            'sc_reclamos',
        ];

        foreach ($tables as $table) {
            if (!inventario_has_table($pdo, $table)) {
                continue;
            }

            $dateColumn = inventario_detect_column($pdo, $table, ['fecha', 'fecha_reclamo', 'respondido_at', 'created_at', 'creado_at', 'registrado_at']);
            if ($dateColumn === null) {
                continue;
            }

            return [
                'table'           => $table,
                'date_column'     => $dateColumn,
                'cliente_column'  => inventario_detect_column($pdo, $table, ['cliente_id', 'destinatario_id', 'clienteid']),
                'pedido_column'   => inventario_detect_column($pdo, $table, ['pedido_id', 'so_pedido_id', 'embarque_id', 'orden_id']),
                'deposito_column' => inventario_detect_column($pdo, $table, ['deposito_id', 'warehouse_id']),
                'deleted_column'  => inventario_detect_column($pdo, $table, ['deleted_at', 'eliminado_at']),
            ];
        }

        return null;
    }
}

if (!function_exists('servicio_cliente_detect_satisfaccion_source')) {
    /**
     * @return array<string, string|null>|null
     */
    function servicio_cliente_detect_satisfaccion_source(PDO $pdo): ?array
    {
        $tables = [
            'so_encuesta_satisfaccion',
            'encuesta_satisfaccion',
            'servicio_cliente_encuestas',
            'encuestas_post_entrega',
            'cliente_encuestas',
            'clientes_encuestas',
            'customer_satisfaction',
        ];

        foreach ($tables as $table) {
            if (!inventario_has_table($pdo, $table)) {
                continue;
            }

            $dateColumn = inventario_detect_column($pdo, $table, ['fecha', 'fecha_respuesta', 'respondido_at', 'created_at', 'creado_at']);
            if ($dateColumn === null) {
                continue;
            }

            $scoreColumn = inventario_detect_column($pdo, $table, ['puntaje', 'puntaje_total', 'valoracion', 'calificacion', 'score', 'porcentaje', 'nps', 'satisfaccion']);
            if ($scoreColumn === null) {
                continue;
            }

            $scaleHint = (strpos($scoreColumn, 'porcentaje') !== false || strpos($scoreColumn, 'pct') !== false) ? 'percent' : null;

            return [
                'table'           => $table,
                'date_column'     => $dateColumn,
                'score_column'    => $scoreColumn,
                'cliente_column'  => inventario_detect_column($pdo, $table, ['cliente_id', 'destinatario_id']),
                'pedido_column'   => inventario_detect_column($pdo, $table, ['pedido_id', 'embarque_id', 'so_pedido_id']),
                'deposito_column' => inventario_detect_column($pdo, $table, ['deposito_id', 'warehouse_id']),
                'max_column'      => inventario_detect_column($pdo, $table, ['puntaje_maximo', 'max_score', 'escala', 'escala_maxima']),
                'deleted_column'  => inventario_detect_column($pdo, $table, ['deleted_at', 'eliminado_at']),
                'scale_hint'      => $scaleHint,
            ];
        }

        return null;
    }
}
