<?php

declare(strict_types=1);
if (!function_exists('stock_quiebre_fetch_combos')) {
    /**
     * Devuelve combos (clientes, operativas) reutilizables en el reporte.
     */
    function stock_quiebre_fetch_combos(PDO $pdo): array
    {
        return [
            'clientes'   => stock_quiebre_fetch_clientes($pdo),
            'operativas' => stock_quiebre_fetch_operativas($pdo),
        ];
    }
}

if (!function_exists('stock_quiebre_fetch_data')) {
    /**
     * Genera los datos del reporte de quiebre de stock según filtros y límite.
     *
     * @return array{
     *     rows: array<int, array<string, mixed>>,
     *     totals: array<string, float|int|null>,
     *     limit: int,
     *     truncated: bool,
     *     filters: array<string, string|int>,
     *     criterion: string
     * }
     */
    function stock_quiebre_fetch_data(PDO $pdo, array $rawFilters, int $limit = 500): array
    {
        $filters = stock_quiebre_normalize_filters($rawFilters);

        $limit = max(1, min($limit, 2000));

        $hasMax = stock_quiebre_has_column($pdo, 'para_productos', 'stock_max_default');
        $useNewSchema = stock_quiebre_has_table($pdo, 'wh_pallet_items');

        if ($useNewSchema) {
            $sub = "
                SELECT
                    it.producto_id,
                    SUM(COALESCE(it.uc_total_cache,
                        (COALESCE(it.uc_por_caja,0) * COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                    )) AS stock,
                    SUM(CASE WHEN it.estado = 'RESERVADO' THEN
                        COALESCE(it.uc_total_cache,
                            (COALESCE(it.uc_por_caja,0) * COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                        )
                    ELSE 0 END) AS reservados
                FROM wh_pallet_items it
                GROUP BY it.producto_id
            ";
        } else {
            $sub = "
                SELECT
                    s.producto_id,
                    SUM(COALESCE(s.qty_uc,0)) AS stock,
                    0 AS reservados
                FROM wh_stock s
                GROUP BY s.producto_id
            ";
        }

        $sql = "
            SELECT
                c.razon_social AS cliente,
                op.nombre      AS operativa,
                p.id           AS producto_id,
                p.sku,
                p.denominacion,
                COALESCE(p.stock_min_default, 0) AS minimo,
                " . ($hasMax ? "COALESCE(p.stock_max_default, 0)" : "NULL") . " AS maximo,
                COALESCE(agg.stock, 0)      AS stock,
                COALESCE(agg.reservados, 0) AS reservados,
                (COALESCE(agg.stock, 0) - COALESCE(agg.reservados, 0)) AS disponibles
            FROM para_productos p
            LEFT JOIN ({$sub}) agg ON agg.producto_id = p.id
            LEFT JOIN para_clientes c ON c.id = p.cliente_id
            LEFT JOIN sys_operativas op ON op.id = p.operativa_id
            WHERE (p.deleted_at IS NULL OR p.deleted_at IS NULL)
        ";

        $params = [];

        if ($filters['cliente_id'] !== '') {
            $sql .= " AND p.cliente_id = :cliente_id";
            $params[':cliente_id'] = $filters['cliente_id'];
        }
        if ($filters['operativa_id'] !== '') {
            $sql .= " AND p.operativa_id = :operativa_id";
            $params[':operativa_id'] = $filters['operativa_id'];
        }
        if ($filters['q'] !== '') {
            $sql .= " AND (p.sku LIKE :q OR p.denominacion LIKE :q)";
            $params[':q'] = '%' . $filters['q'] . '%';
        }
        if ($filters['solo_con_min']) {
            $sql .= " AND COALESCE(p.stock_min_default,0) > 0";
        }

        $exprObserved = $filters['criterio'] === 'stock' ? 't.stock' : 't.disponibles';

        $outer = "
            SELECT
                t.cliente,
                t.operativa,
                t.producto_id,
                t.sku,
                t.denominacion,
                COALESCE(t.minimo, 0)     AS minimo,
                COALESCE(t.maximo, NULL)  AS maximo,
                COALESCE(t.stock, 0)      AS stock,
                COALESCE(t.reservados, 0) AS reservados,
                COALESCE(t.disponibles,0) AS disponibles
            FROM (
                {$sql}
            ) t
            WHERE ({$exprObserved}) < COALESCE(t.minimo, 0)
              AND COALESCE(t.minimo, 0) > 0
            ORDER BY t.operativa ASC, t.sku ASC
            LIMIT :row_limit
        ";

        $stmt = $pdo->prepare($outer);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':row_limit', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $truncated = false;
        if (count($rows) > $limit) {
            $rows = array_slice($rows, 0, $limit);
            $truncated = true;
        }

        $processed = [];
        $totals = [
            'productos'          => 0,
            'faltante_total'     => 0.0,
            'disponibles_total'  => 0.0,
            'stock_total'        => 0.0,
            'reservados_total'   => 0.0,
            'minimo_total'       => 0.0,
            'cobertura_promedio' => null,
        ];

        $coverageSum = 0.0;
        $coverageCount = 0;
        $criterionKey = $filters['criterio'] === 'stock' ? 'stock' : 'disponibles';

        foreach ($rows as $row) {
            $minimo      = (float)($row['minimo'] ?? 0);
            $stock       = (float)($row['stock'] ?? 0);
            $reservados  = (float)($row['reservados'] ?? 0);
            $disponibles = (float)($row['disponibles'] ?? 0);
            $observado   = $criterionKey === 'stock' ? $stock : $disponibles;
            $faltante    = max($minimo - $observado, 0);

            $coverage = null;
            if ($minimo > 0) {
                $coverage = $observado / $minimo * 100.0;
                $coverageSum += $coverage;
                $coverageCount++;
            }

            $processed[] = [
                'cliente'        => $row['cliente'] ?? '-',
                'operativa'      => $row['operativa'] ?? '-',
                'producto_id'    => (int)($row['producto_id'] ?? 0),
                'sku'            => $row['sku'] ?? '',
                'denominacion'   => $row['denominacion'] ?? '',
                'minimo'         => $minimo,
                'maximo'         => $row['maximo'] !== null ? (float)$row['maximo'] : null,
                'stock'          => $stock,
                'reservados'     => $reservados,
                'disponibles'    => $disponibles,
                'faltante_min'   => $faltante,
                'cobertura_pct'  => $coverage !== null ? round($coverage, 1) : null,
                'criterio_base'  => $criterionKey,
                'criterio_valor' => $observado,
            ];

            $totals['faltante_total']    += $faltante;
            $totals['disponibles_total'] += $disponibles;
            $totals['stock_total']       += $stock;
            $totals['reservados_total']  += $reservados;
            $totals['minimo_total']      += $minimo;
        }

        $totals['productos'] = count($processed);
        if ($coverageCount > 0) {
            $totals['cobertura_promedio'] = round($coverageSum / $coverageCount, 1);
        }

        return [
            'rows'      => $processed,
            'totals'    => $totals,
            'limit'     => $limit,
            'truncated' => $truncated,
            'filters'   => $filters,
            'criterion' => $criterionKey,
        ];
    }
}

if (!function_exists('stock_quiebre_normalize_filters')) {
    /** @return array<string, string|int> */
    function stock_quiebre_normalize_filters(array $raw): array
    {
        $filters = [
            'cliente_id'   => trim((string)($raw['cliente_id'] ?? '')),
            'operativa_id' => trim((string)($raw['operativa_id'] ?? '')),
            'q'            => trim((string)($raw['q'] ?? '')),
            'criterio'     => strtolower(trim((string)($raw['criterio'] ?? 'disponibles'))),
            'solo_con_min' => !empty($raw['solo_con_min']) ? 1 : 0,
        ];

        if (!in_array($filters['criterio'], ['stock', 'disponibles'], true)) {
            $filters['criterio'] = 'disponibles';
        }

        return $filters;
    }
}

if (!function_exists('stock_quiebre_has_table')) {
    function stock_quiebre_has_table(PDO $pdo, string $table): bool
    {
        static $cache = [];
        if (array_key_exists($table, $cache)) {
            return $cache[$table];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$table]);
        $cache[$table] = (int)$stmt->fetchColumn() > 0;
        return $cache[$table];
    }
}

if (!function_exists('stock_quiebre_has_column')) {
    function stock_quiebre_has_column(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = $table . '|' . $column;
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        $cache[$key] = (int)$stmt->fetchColumn() > 0;
        return $cache[$key];
    }
}

if (!function_exists('stock_quiebre_fetch_clientes')) {
    function stock_quiebre_fetch_clientes(PDO $pdo): array
    {
        $sql = 'SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social';
        return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}

if (!function_exists('stock_quiebre_fetch_operativas')) {
    function stock_quiebre_fetch_operativas(PDO $pdo): array
    {
        $sql = 'SELECT id, nombre FROM sys_operativas WHERE deleted_at IS NULL ORDER BY nombre';
        return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}
