<?php

declare(strict_types=1);

if (!function_exists('traza_detect_table')) {
    /**
     * Determina el primer nombre de tabla existente dentro del listado indicado.
     *
     * @param array<int,string> $candidates
     */
    function traza_detect_table(PDO $pdo, array $candidates): ?string
    {
        foreach ($candidates as $table) {
            $table = trim((string) $table);
            if ($table === '') {
                continue;
            }
            if (traza_table_exists($pdo, $table)) {
                return $table;
            }
        }
        return null;
    }
}

if (!function_exists('traza_pick_column')) {
    /**
     * Devuelve el primer nombre de columna existente en la tabla indicada.
     *
     * @param array<int,string> $candidates
     */
    function traza_pick_column(PDO $pdo, string $table, array $candidates): ?string
    {
        foreach ($candidates as $column) {
            $column = trim((string) $column);
            if ($column === '') {
                continue;
            }
            if (traza_column_exists($pdo, $table, $column)) {
                return $column;
            }
        }
        return null;
    }
}

if (!function_exists('traza_resolve_schema')) {
    /**
     * Detecta las tablas efectivas a utilizar según la base actual.
     *
     * @return array{pallets:string,pallet_items:string,moves:string,move_items:?string}
     */
    function traza_resolve_schema(PDO $pdo): array
    {
        $pallets = traza_detect_table($pdo, ['wh_pallets', 'wh_pallet']);
        $palletItems = traza_detect_table($pdo, ['wh_pallet_items', 'wh_pallet_item']);
        $moves = traza_detect_table($pdo, ['wh_moves', 'wh_move']);
        $moveItems = traza_detect_table($pdo, ['wh_move_items', 'wh_move_item']);

        $missing = [];
        if (!$pallets) {
            $missing[] = 'wh_pallets/wh_pallet';
        }
        if (!$palletItems) {
            $missing[] = 'wh_pallet_items/wh_pallet_item';
        }
        if (!$moves) {
            $missing[] = 'wh_moves/wh_move';
        }

        if ($missing) {
            throw new RuntimeException('El esquema de warehouse no está disponible en esta base (faltan: ' . implode(', ', $missing) . ').');
        }

        return [
            'pallets' => $pallets,
            'pallet_items' => $palletItems,
            'moves' => $moves,
            'move_items' => $moveItems,
        ];
    }
}

if (!function_exists('traza_fetch_data')) {
    /**
     * Punto de entrada principal: aplica filtros, agrupa pallets y obtiene movimientos.
     *
     * @param array<string,mixed> $input
     * @return array<string,mixed>
     */
    function traza_fetch_data(PDO $pdo, array $input): array
    {
        $filters = traza_normalize_filters($input);
        if ($filters['sku'] === '' && $filters['lote'] === '' && $filters['pallet'] === '') {
            throw new RuntimeException('Ingrese al menos un SKU, lote o código de pallet para buscar.');
        }

        $schema = traza_resolve_schema($pdo);

        $items = traza_fetch_matching_items($pdo, $filters, $schema);
        $grouped = traza_group_items_by_pallet($items, $filters);
        $pallets = $grouped['pallets'];
        $stats = $grouped['stats'];

        $fallbackUsed = false;
        if (!$pallets && $filters['pallet'] !== '') {
            $fallback = traza_fetch_pallet_fallback($pdo, $filters['pallet'], $schema);
            if ($fallback) {
                $pallets = $fallback['pallets'];
                $stats = traza_merge_stats($stats, $fallback['stats']);
                $fallbackUsed = true;
            }
        }

        $palletIds = array_values(array_filter(array_unique(array_map(
            static fn(array $pallet): int => (int) ($pallet['pallet_id'] ?? 0),
            $pallets
        ))));

        $movements = $palletIds ? traza_fetch_movements($pdo, $palletIds, $filters, $schema) : [];
        $summary = traza_build_summary($pallets, $movements, $filters, $stats);

        return [
            'filters' => $filters,
            'pallets' => $pallets,
            'movements' => $movements,
            'summary' => $summary,
            'meta' => [
                'fallback_used' => $fallbackUsed,
                'matched_items' => count($items),
            ],
        ];
    }
}

if (!function_exists('traza_fetch_matching_items')) {
    /**
     * Recupera el detalle de pallet_items que cumplen con los filtros.
     *
     * @return array<int,array<string,mixed>>
     */
    function traza_fetch_matching_items(PDO $pdo, array $filters, array $schema): array
    {
        $palletItemsTable = $schema['pallet_items'];
        $palletsTable = $schema['pallets'];

        $hasPalletEstado = traza_column_exists($pdo, $palletsTable, 'estado');
        $hasPalletCreated = traza_column_exists($pdo, $palletsTable, 'created_at');
        $hasUvCajas = traza_column_exists($pdo, $palletItemsTable, 'uv_cajas');
        $hasUcPorCaja = traza_column_exists($pdo, $palletItemsTable, 'uc_por_caja');
        $hasUcSueltas = traza_column_exists($pdo, $palletItemsTable, 'uc_sueltas');
        $hasUcTotalCache = traza_column_exists($pdo, $palletItemsTable, 'uc_total_cache');
        $hasUcUnidades = traza_column_exists($pdo, $palletItemsTable, 'uc_unidades');
        $hasItemEstado = traza_column_exists($pdo, $palletItemsTable, 'estado');
        $hasFechaProd = traza_column_exists($pdo, $palletItemsTable, 'fecha_produccion');
        $hasFechaVenc = traza_column_exists($pdo, $palletItemsTable, 'fecha_vencimiento');

        $hasLoteDirect = traza_column_exists($pdo, $palletItemsTable, 'lote');
        $hasLoteId = traza_column_exists($pdo, $palletItemsTable, 'lote_id');
        $loteExpr = $hasLoteDirect ? 'pi.lote' : null;
        $fechaProdExpr = $hasFechaProd ? 'pi.fecha_produccion' : null;
        $fechaVencExpr = $hasFechaVenc ? 'pi.fecha_vencimiento' : null;

        $hasOccupancy = traza_table_exists($pdo, 'wh_position_occupancy');
        $hasPositions = $hasOccupancy && traza_table_exists($pdo, 'wh_positions');
        $hasDeposito = $hasPositions && traza_table_exists($pdo, 'wh_deposito');

        $joins = [];
        if ($hasOccupancy) {
            $joins[] = 'LEFT JOIN wh_position_occupancy occ ON occ.pallet_id = pa.id AND occ.hasta IS NULL';
        }
        if ($hasPositions) {
            $joins[] = 'LEFT JOIN wh_positions pos ON pos.id = occ.position_id';
        }
        if ($hasDeposito) {
            $joins[] = 'LEFT JOIN wh_deposito dep ON dep.id = pos.deposito_id';
        }

        if (!$loteExpr && $hasLoteId && traza_table_exists($pdo, 'wh_lote')) {
            $joins[] = 'LEFT JOIN wh_lote lt ON lt.id = pi.lote_id';
            if (traza_column_exists($pdo, 'wh_lote', 'codigo')) {
                $loteExpr = 'lt.codigo';
            } elseif (traza_column_exists($pdo, 'wh_lote', 'code')) {
                $loteExpr = 'lt.code';
            }
            if (!$fechaProdExpr && traza_column_exists($pdo, 'wh_lote', 'fecha_produccion')) {
                $fechaProdExpr = 'lt.fecha_produccion';
            }
            if (!$fechaVencExpr && traza_column_exists($pdo, 'wh_lote', 'fecha_vencimiento')) {
                $fechaVencExpr = 'lt.fecha_vencimiento';
            }
        }

        $loteExpr = $loteExpr ?? 'NULL';
        $fechaProdExpr = $fechaProdExpr ?? 'NULL';
        $fechaVencExpr = $fechaVencExpr ?? 'NULL';

        $selectParts = [
            'pi.id AS pallet_item_id',
            'pi.pallet_id AS pallet_id',
            'pa.codigo AS pallet_codigo',
            $hasPalletEstado ? 'pa.estado AS pallet_estado' : 'NULL AS pallet_estado',
            $hasPalletCreated ? 'pa.created_at AS pallet_created_at' : 'NULL AS pallet_created_at',
            'pr.id AS producto_id',
            'pr.sku AS producto_sku',
            'pr.denominacion AS producto_nombre',
            $loteExpr . ' AS lote',
            $fechaProdExpr . ' AS fecha_produccion',
            $fechaVencExpr . ' AS fecha_vencimiento',
            $hasUvCajas ? 'pi.uv_cajas AS uv_cajas' : '0 AS uv_cajas',
            $hasUcPorCaja ? 'pi.uc_por_caja AS uc_por_caja' : '0 AS uc_por_caja',
            $hasUcSueltas ? 'pi.uc_sueltas AS uc_sueltas' : '0 AS uc_sueltas',
            $hasUcTotalCache ? 'pi.uc_total_cache AS uc_total_cache' : 'NULL AS uc_total_cache',
            $hasUcUnidades ? 'pi.uc_unidades AS uc_unidades' : 'NULL AS uc_unidades',
            $hasItemEstado ? 'pi.estado AS item_estado' : 'NULL AS item_estado',
            $hasOccupancy ? 'occ.position_id AS position_id' : 'NULL AS position_id',
            $hasOccupancy ? 'occ.desde AS ocupando_desde' : 'NULL AS ocupando_desde',
            $hasPositions ? 'pos.rack AS pos_rack' : 'NULL AS pos_rack',
            $hasPositions ? 'pos.columna AS pos_columna' : 'NULL AS pos_columna',
            $hasPositions ? 'pos.nivel AS pos_nivel' : 'NULL AS pos_nivel',
            $hasPositions ? 'pos.fondo AS pos_fondo' : 'NULL AS pos_fondo',
            $hasPositions ? 'pos.deposito_id AS pos_deposito_id' : 'NULL AS pos_deposito_id',
            $hasDeposito ? 'dep.code AS deposito_code' : 'NULL AS deposito_code',
            $hasDeposito ? 'dep.nombre AS deposito_nombre' : 'NULL AS deposito_nombre',
        ];

        $sql = "
            SELECT
                " . implode(",\n                ", $selectParts) . "
            FROM {$palletItemsTable} pi
            JOIN {$palletsTable} pa ON pa.id = pi.pallet_id
            JOIN para_productos pr ON pr.id = pi.producto_id
        " . ($joins ? "\n            " . implode("\n            ", $joins) : '') . "
            WHERE 1 = 1
        ";

        $conditions = [];
        $params = [];

        if ($filters['sku'] !== '') {
            $conditions[] = '(UPPER(pr.sku) LIKE ? OR UPPER(pr.denominacion) LIKE ?)';
            $like = traza_like($filters['sku']);
            $params[] = $like;
            $params[] = $like;
        }

        if ($filters['lote'] !== '') {
            $conditions[] = 'UPPER(COALESCE(' . $loteExpr . ", '')) LIKE ?";
            $params[] = traza_like($filters['lote']);
        }

        if ($filters['pallet'] !== '') {
            $conditions[] = 'UPPER(pa.codigo) LIKE ?';
            $params[] = traza_like($filters['pallet']);
        }

        if ($conditions) {
            $sql .= ' AND ' . implode(' AND ', $conditions);
        }

        $orderExpr = $loteExpr !== 'NULL' ? $loteExpr : 'pa.codigo';
        $sql .= ' ORDER BY pa.codigo ASC, pr.sku ASC, ' . $orderExpr . ' ASC LIMIT 600';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $index => $value) {
            $stmt->bindValue($index + 1, $value);
        }
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}

if (!function_exists('traza_group_items_by_pallet')) {
    /**
     * Agrupa los ítems obtenidos por pallet.
     *
     * @return array{pallets: array<int,array<string,mixed>>, stats: array<string,mixed>}
     */
    function traza_group_items_by_pallet(array $rows, array $filters): array
    {
        $result = [];
        $stats = [
            'productos' => [],
            'lotes' => [],
            'pallets' => [],
        ];

        foreach ($rows as $row) {
            $pid = (int) ($row['pallet_id'] ?? 0);
            if ($pid <= 0) {
                continue;
            }

            if (!isset($result[$pid])) {
                $result[$pid] = [
                    'pallet_id' => $pid,
                    'codigo' => (string) ($row['pallet_codigo'] ?? ''),
                    'estado' => (string) ($row['pallet_estado'] ?? ''),
                    'created_at' => $row['pallet_created_at'] ?? null,
                    'items' => [],
                    'uv_total' => 0,
                    'uc_total' => 0,
                    'position_label' => traza_format_position_label($row),
                    'deposito_label' => traza_format_deposito_label($row),
                    'ocupando_desde' => $row['ocupando_desde'] ?? null,
                    'match_reasons' => [],
                ];
                $stats['pallets'][] = $result[$pid]['codigo'];
            }

            $uv = (int) ($row['uv_cajas'] ?? 0);
            $ucPorCaja = (int) ($row['uc_por_caja'] ?? 0);
            $ucSueltas = (int) ($row['uc_sueltas'] ?? 0);
            $ucUnidades = isset($row['uc_unidades']) ? (int) $row['uc_unidades'] : null;

            if ($row['uc_total_cache'] !== null) {
                $ucTotal = (int) $row['uc_total_cache'];
            } elseif ($uv > 0 && $ucPorCaja > 0) {
                $ucTotal = ($uv * $ucPorCaja) + $ucSueltas;
            } elseif ($ucUnidades !== null) {
                $ucTotal = $ucUnidades;
            } else {
                $ucTotal = $ucSueltas;
            }

            $item = [
                'pallet_item_id' => (int) $row['pallet_item_id'],
                'producto_id' => (int) $row['producto_id'],
                'sku' => (string) ($row['producto_sku'] ?? ''),
                'nombre' => (string) ($row['producto_nombre'] ?? ''),
                'lote' => (string) ($row['lote'] ?? ''),
                'estado' => (string) ($row['item_estado'] ?? ''),
                'uv_cajas' => $uv,
                'uc_total' => $ucTotal,
                'uc_por_caja' => $ucPorCaja,
                'uc_sueltas' => $ucSueltas,
                'uc_unidades' => $ucUnidades,
                'fecha_produccion' => $row['fecha_produccion'] ?? null,
                'fecha_vencimiento' => $row['fecha_vencimiento'] ?? null,
            ];

            $result[$pid]['items'][] = traza_enrich_item_summary($item);
            $result[$pid]['uv_total'] += $uv;
            $result[$pid]['uc_total'] += $ucTotal;

            if ($filters['sku'] !== '' && str_contains(strtoupper($item['sku']), $filters['sku'])) {
                $result[$pid]['match_reasons']['sku'] = 'SKU';
            }
            if ($filters['lote'] !== '' && str_contains(strtoupper($item['lote']), $filters['lote'])) {
                $result[$pid]['match_reasons']['lote'] = 'Lote';
            }
            if ($filters['pallet'] !== '' && str_contains(strtoupper((string) $result[$pid]['codigo']), $filters['pallet'])) {
                $result[$pid]['match_reasons']['pallet'] = 'Pallet';
            }

            $stats['productos'][$item['sku']] = $item['nombre'];
            if ($item['lote'] !== '') {
                $stats['lotes'][$item['lote']] = true;
            }
        }

        $list = array_values(array_map(static function (array $pallet): array {
            $pallet['items_summary'] = array_map(static fn(array $item): string => $item['summary'], $pallet['items']);
            $pallet['match_summary'] = $pallet['match_reasons'] ? implode(', ', $pallet['match_reasons']) : '';
            return $pallet;
        }, $result));

        return [
            'pallets' => $list,
            'stats' => $stats,
        ];
    }
}

if (!function_exists('traza_enrich_item_summary')) {
    /**
     * Agrega textos formateados para un ítem de pallet.
     */
    function traza_enrich_item_summary(array $item): array
    {
        $parts = [];
        if ($item['sku'] !== '') {
            $parts[] = $item['sku'];
        }
        if ($item['lote'] !== '') {
            $parts[] = 'Lote ' . $item['lote'];
        }
        $qtyParts = [];
        if ($item['uv_cajas'] > 0) {
            $qtyParts[] = $item['uv_cajas'] . ' cajas';
        }
        if ($item['uc_total'] > 0) {
            $qtyParts[] = $item['uc_total'] . ' UC';
        }
        if ($qtyParts) {
            $parts[] = implode(' · ', $qtyParts);
        }
        if ($item['estado'] !== '') {
            $parts[] = '(' . $item['estado'] . ')';
        }
        $item['summary'] = implode(' · ', $parts);
        return $item;
    }
}

if (!function_exists('traza_fetch_pallet_fallback')) {
    /**
     * Busca un pallet por código aunque no tenga ítems actuales.
     */
    function traza_fetch_pallet_fallback(PDO $pdo, string $palletCode, array $schema): ?array
    {
        $palletsTable = $schema['pallets'];
        $hasEstado = traza_column_exists($pdo, $palletsTable, 'estado');
        $hasCreated = traza_column_exists($pdo, $palletsTable, 'created_at');

        $columns = [
            'pa.id AS pallet_id',
            'pa.codigo AS codigo',
        ];
        $columns[] = $hasEstado ? 'pa.estado AS estado' : 'NULL AS estado';
        $columns[] = $hasCreated ? 'pa.created_at AS created_at' : 'NULL AS created_at';

        $selectSql = implode(",\n                ", $columns);

        $sql = "
            SELECT
                {$selectSql}
            FROM {$palletsTable} pa
            WHERE UPPER(pa.codigo) LIKE :code
            LIMIT 10
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':code', traza_like($palletCode));
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        if (!$rows) {
            return null;
        }
        $pallets = array_map(static function (array $row): array {
            return [
                'pallet_id' => (int) $row['pallet_id'],
                'codigo' => (string) ($row['codigo'] ?? ''),
                'estado' => (string) ($row['estado'] ?? ''),
                'created_at' => $row['created_at'] ?? null,
                'items' => [],
                'uv_total' => 0,
                'uc_total' => 0,
                'items_summary' => [],
                'position_label' => 'Sin datos',
                'deposito_label' => '',
                'ocupando_desde' => null,
                'match_reasons' => ['pallet' => 'Pallet'],
                'match_summary' => 'Pallet',
            ];
        }, $rows);

        return [
            'pallets' => $pallets,
            'stats' => [
                'productos' => [],
                'lotes' => [],
                'pallets' => array_map(static fn(array $p): string => $p['codigo'], $pallets),
            ],
        ];
    }
}

if (!function_exists('traza_fetch_movements')) {
    /**
     * Devuelve movimientos de los pallets indicados.
     *
     * @param array<int> $palletIds
     * @return array<int,array<string,mixed>>
     */
    function traza_fetch_movements(PDO $pdo, array $palletIds, array $filters, array $schema): array
    {
        $placeholders = implode(',', array_fill(0, count($palletIds), '?'));

        $fechaDesde = $filters['fecha_desde'] . ' 00:00:00';
        $fechaHasta = $filters['fecha_hasta'] . ' 23:59:59';

        $hasMoveItems = !empty($schema['move_items']);
        $movesTable = $schema['moves'];
        $palletsTable = $schema['pallets'];
        $moveItemsTable = $schema['move_items'];
        $palletItemsTable = $schema['pallet_items'];

        $positionTable = traza_detect_table($pdo, ['wh_positions', 'wh_posicion']);
        $depositTable = traza_detect_table($pdo, ['wh_depositos', 'wh_deposito']);

        $fromPosColumn = traza_pick_column($pdo, $movesTable, ['desde_position_id', 'from_position_id', 'from_pos_id']);
        $toPosColumn = traza_pick_column($pdo, $movesTable, ['hasta_position_id', 'to_position_id', 'to_pos_id']);

        $joinFromPosition = $positionTable !== null && $fromPosColumn !== null;
        $joinToPosition = $positionTable !== null && $toPosColumn !== null;
        $hasAnyPosition = $joinFromPosition || $joinToPosition;

        $posRackCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['rack']) : null;
        $posColCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['columna', 'column', 'col']) : null;
        $posNivelCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['nivel', 'level']) : null;
        $posFondoCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['fondo', 'depth']) : null;
        $posCodeFullCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['code_full', 'pos_code_full']) : null;
        $posCodeCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['pos_code', 'code']) : null;
        $posDepositCol = $hasAnyPosition ? traza_pick_column($pdo, $positionTable, ['deposito_id', 'deposit_id']) : null;

        $hasDeposito = $posDepositCol !== null && $depositTable !== null;
        $depCodeCol = $hasDeposito ? traza_pick_column($pdo, $depositTable, ['code', 'codigo']) : null;
        $depNameCol = $hasDeposito ? traza_pick_column($pdo, $depositTable, ['nombre', 'name']) : null;

        $hasRefTipo = traza_column_exists($pdo, $movesTable, 'ref_tipo');
        $hasRefId = traza_column_exists($pdo, $movesTable, 'ref_id');
        $hasUvCajas = traza_column_exists($pdo, $movesTable, 'uv_cajas');
        $hasUcUnidades = traza_column_exists($pdo, $movesTable, 'uc_unidades');
        $hasDeltaUv = traza_column_exists($pdo, $movesTable, 'delta_uv');
        $hasDeltaUc = traza_column_exists($pdo, $movesTable, 'delta_uc');
        $hasSolicitadoPor = traza_column_exists($pdo, $movesTable, 'solicitado_por');
        $hasAsignadoA = traza_column_exists($pdo, $movesTable, 'asignado_a');
        $hasEjecutadoPor = traza_column_exists($pdo, $movesTable, 'ejecutado_por');
        $hasAsignadoAt = traza_column_exists($pdo, $movesTable, 'asignado_at');
        $hasIniciadoAt = traza_column_exists($pdo, $movesTable, 'iniciado_at');
        $hasFinalizadoAt = traza_column_exists($pdo, $movesTable, 'finalizado_at');

        $selectParts = [
            'm.id',
            'm.pallet_id',
            'pa.codigo AS pallet_codigo',
            'm.tipo',
            'm.motivo',
            ($hasRefTipo ? 'm.ref_tipo' : 'NULL') . ' AS ref_tipo',
            ($hasRefId ? 'm.ref_id' : 'NULL') . ' AS ref_id',
            ($hasUvCajas ? 'm.uv_cajas' : 'NULL') . ' AS uv_cajas',
            ($hasUcUnidades ? 'm.uc_unidades' : 'NULL') . ' AS uc_unidades',
            ($hasDeltaUv ? 'm.delta_uv' : 'NULL') . ' AS delta_uv',
            ($hasDeltaUc ? 'm.delta_uc' : 'NULL') . ' AS delta_uc',
        ];

        if ($hasSolicitadoPor) {
            $selectParts[] = 'm.solicitado_por AS solicitado_por';
            $selectParts[] = 'sol.nombre AS solicitado_nombre';
        } else {
            $selectParts[] = 'NULL AS solicitado_por';
            $selectParts[] = 'NULL AS solicitado_nombre';
        }

        if ($hasAsignadoA) {
            $selectParts[] = 'm.asignado_a AS asignado_a';
            $selectParts[] = 'asg.nombre AS asignado_nombre';
        } else {
            $selectParts[] = 'NULL AS asignado_a';
            $selectParts[] = 'NULL AS asignado_nombre';
        }

        if ($hasEjecutadoPor) {
            $selectParts[] = 'm.ejecutado_por AS ejecutado_por';
            $selectParts[] = 'eje.nombre AS ejecutado_nombre';
        } else {
            $selectParts[] = 'NULL AS ejecutado_por';
            $selectParts[] = 'NULL AS ejecutado_nombre';
        }

        $selectParts[] = ($hasAsignadoAt ? 'm.asignado_at' : 'NULL') . ' AS asignado_at';
        $selectParts[] = ($hasIniciadoAt ? 'm.iniciado_at' : 'NULL') . ' AS iniciado_at';
        $selectParts[] = ($hasFinalizadoAt ? 'm.finalizado_at' : 'NULL') . ' AS finalizado_at';
        $selectParts[] = 'm.created_at';

        $fechaSources = [];
        if ($hasFinalizadoAt) {
            $fechaSources[] = 'm.finalizado_at';
        }
        if ($hasIniciadoAt) {
            $fechaSources[] = 'm.iniciado_at';
        }
        if ($hasAsignadoAt) {
            $fechaSources[] = 'm.asignado_at';
        }
        $fechaSources[] = 'm.created_at';
        $fechaRefExpr = 'COALESCE(' . implode(', ', $fechaSources) . ')';
        $selectParts[] = $fechaRefExpr . ' AS fecha_ref';

        $selectParts[] = ($fromPosColumn ? 'm.' . $fromPosColumn : 'NULL') . ' AS desde_position_id';
        $selectParts[] = ($toPosColumn ? 'm.' . $toPosColumn : 'NULL') . ' AS hasta_position_id';

        if ($joinFromPosition && $posRackCol) {
            $selectParts[] = 'posd.' . $posRackCol . ' AS desde_rack';
        } else {
            $selectParts[] = 'NULL AS desde_rack';
        }
        if ($joinFromPosition && $posColCol) {
            $selectParts[] = 'posd.' . $posColCol . ' AS desde_columna';
        } else {
            $selectParts[] = 'NULL AS desde_columna';
        }
        if ($joinFromPosition && $posNivelCol) {
            $selectParts[] = 'posd.' . $posNivelCol . ' AS desde_nivel';
        } else {
            $selectParts[] = 'NULL AS desde_nivel';
        }
        if ($joinFromPosition && $posFondoCol) {
            $selectParts[] = 'posd.' . $posFondoCol . ' AS desde_fondo';
        } else {
            $selectParts[] = 'NULL AS desde_fondo';
        }
        if ($joinFromPosition && $posCodeFullCol) {
            $selectParts[] = 'posd.' . $posCodeFullCol . ' AS desde_code_full';
        } else {
            $selectParts[] = 'NULL AS desde_code_full';
        }
        if ($joinFromPosition && $posCodeCol) {
            $selectParts[] = 'posd.' . $posCodeCol . ' AS desde_code';
        } else {
            $selectParts[] = 'NULL AS desde_code';
        }

        if ($joinToPosition && $posRackCol) {
            $selectParts[] = 'posh.' . $posRackCol . ' AS hasta_rack';
        } else {
            $selectParts[] = 'NULL AS hasta_rack';
        }
        if ($joinToPosition && $posColCol) {
            $selectParts[] = 'posh.' . $posColCol . ' AS hasta_columna';
        } else {
            $selectParts[] = 'NULL AS hasta_columna';
        }
        if ($joinToPosition && $posNivelCol) {
            $selectParts[] = 'posh.' . $posNivelCol . ' AS hasta_nivel';
        } else {
            $selectParts[] = 'NULL AS hasta_nivel';
        }
        if ($joinToPosition && $posFondoCol) {
            $selectParts[] = 'posh.' . $posFondoCol . ' AS hasta_fondo';
        } else {
            $selectParts[] = 'NULL AS hasta_fondo';
        }
        if ($joinToPosition && $posCodeFullCol) {
            $selectParts[] = 'posh.' . $posCodeFullCol . ' AS hasta_code_full';
        } else {
            $selectParts[] = 'NULL AS hasta_code_full';
        }
        if ($joinToPosition && $posCodeCol) {
            $selectParts[] = 'posh.' . $posCodeCol . ' AS hasta_code';
        } else {
            $selectParts[] = 'NULL AS hasta_code';
        }

        if ($joinFromPosition && $hasDeposito && $depCodeCol) {
            $selectParts[] = 'depd.' . $depCodeCol . ' AS desde_deposito_code';
        } else {
            $selectParts[] = 'NULL AS desde_deposito_code';
        }
        if ($joinFromPosition && $hasDeposito && $depNameCol) {
            $selectParts[] = 'depd.' . $depNameCol . ' AS desde_deposito_nombre';
        } else {
            $selectParts[] = 'NULL AS desde_deposito_nombre';
        }
        if ($joinToPosition && $hasDeposito && $depCodeCol) {
            $selectParts[] = 'deph.' . $depCodeCol . ' AS hasta_deposito_code';
        } else {
            $selectParts[] = 'NULL AS hasta_deposito_code';
        }
        if ($joinToPosition && $hasDeposito && $depNameCol) {
            $selectParts[] = 'deph.' . $depNameCol . ' AS hasta_deposito_nombre';
        } else {
            $selectParts[] = 'NULL AS hasta_deposito_nombre';
        }

        $joins = [
            "JOIN {$palletsTable} pa ON pa.id = m.pallet_id",
        ];

        if ($hasSolicitadoPor) {
            $joins[] = 'LEFT JOIN sys_users sol ON sol.id = m.solicitado_por';
        }
        if ($hasAsignadoA) {
            $joins[] = 'LEFT JOIN sys_users asg ON asg.id = m.asignado_a';
        }
        if ($hasEjecutadoPor) {
            $joins[] = 'LEFT JOIN sys_users eje ON eje.id = m.ejecutado_por';
        }

        if ($joinFromPosition) {
            $joins[] = 'LEFT JOIN ' . $positionTable . ' posd ON posd.id = m.' . $fromPosColumn;
            if ($hasDeposito) {
                $joins[] = 'LEFT JOIN ' . $depositTable . ' depd ON depd.id = posd.' . $posDepositCol;
            }
        }
        if ($joinToPosition) {
            $joins[] = 'LEFT JOIN ' . $positionTable . ' posh ON posh.id = m.' . $toPosColumn;
            if ($hasDeposito) {
                $joins[] = 'LEFT JOIN ' . $depositTable . ' deph ON deph.id = posh.' . $posDepositCol;
            }
        }

        $selectItems = '';
        if ($hasMoveItems) {
            $loteExpr = null;
            $needsLoteJoin = false;
            if (traza_column_exists($pdo, $palletItemsTable, 'lote')) {
                $loteExpr = 'pi.lote';
            } elseif (traza_column_exists($pdo, $palletItemsTable, 'lote_id') && traza_table_exists($pdo, 'wh_lote')) {
                if (traza_column_exists($pdo, 'wh_lote', 'codigo')) {
                    $loteExpr = 'lt.codigo';
                    $needsLoteJoin = true;
                } elseif (traza_column_exists($pdo, 'wh_lote', 'code')) {
                    $loteExpr = 'lt.code';
                    $needsLoteJoin = true;
                }
            }
            $loteExpr = $loteExpr ?? "''";

            $selectItems = ",
                det.match_skus,
                det.match_lotes,
                det.total_uv,
                det.total_uc
            ";

            $loteJoin = $needsLoteJoin ? 'LEFT JOIN wh_lote lt ON lt.id = pi.lote_id' : '';

            $joins[] = "LEFT JOIN (
                    SELECT
                        mi.move_id,
                        GROUP_CONCAT(DISTINCT UPPER(pr.sku) ORDER BY pr.sku SEPARATOR ', ') AS match_skus,
                        GROUP_CONCAT(DISTINCT UPPER({$loteExpr}) ORDER BY {$loteExpr} SEPARATOR ', ') AS match_lotes,
                        SUM(mi.uv_cajas) AS total_uv,
                        SUM(mi.uc_unidades) AS total_uc
                    FROM {$moveItemsTable} mi
                    JOIN {$palletItemsTable} pi ON pi.id = mi.pallet_item_id
                    JOIN para_productos pr ON pr.id = pi.producto_id
                    {$loteJoin}
                    GROUP BY mi.move_id
                ) det ON det.move_id = m.id";
        }

        $sql = "
            SELECT
                " . implode(",\n                ", $selectParts) . $selectItems . "
            FROM {$movesTable} m
            " . implode("\n            ", $joins) . "
                        WHERE m.pallet_id IN ($placeholders)
                            AND {$fechaRefExpr}
                                    BETWEEN ? AND ?
            ORDER BY fecha_ref DESC, m.id DESC
            LIMIT 1200
        ";

        $stmt = $pdo->prepare($sql);
        $index = 1;
        foreach ($palletIds as $pid) {
            $stmt->bindValue($index, $pid, PDO::PARAM_INT);
            $index++;
        }
        $stmt->bindValue($index, $fechaDesde);
        $stmt->bindValue($index + 1, $fechaHasta);
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return array_map(static function (array $row): array {
            $row['fecha_ref_label'] = traza_format_datetime($row['fecha_ref']);
            $row['asignado_label'] = traza_format_user_label($row['asignado_nombre'], $row['asignado_a']);
            $row['ejecutado_label'] = traza_format_user_label($row['ejecutado_nombre'], $row['ejecutado_por']);
            $row['solicitado_label'] = traza_format_user_label($row['solicitado_nombre'], $row['solicitado_por']);
            $row['desde_label'] = traza_format_position_from_parts([
                'deposito_code' => $row['desde_deposito_code'] ?? null,
                'deposito_nombre' => $row['desde_deposito_nombre'] ?? null,
                'pos_rack' => $row['desde_rack'] ?? null,
                'pos_columna' => $row['desde_columna'] ?? null,
                'pos_nivel' => $row['desde_nivel'] ?? null,
                'pos_fondo' => $row['desde_fondo'] ?? null,
                'pos_code_full' => $row['desde_code_full'] ?? null,
                'pos_code' => $row['desde_code'] ?? null,
            ]);
            $row['hasta_label'] = traza_format_position_from_parts([
                'deposito_code' => $row['hasta_deposito_code'] ?? null,
                'deposito_nombre' => $row['hasta_deposito_nombre'] ?? null,
                'pos_rack' => $row['hasta_rack'] ?? null,
                'pos_columna' => $row['hasta_columna'] ?? null,
                'pos_nivel' => $row['hasta_nivel'] ?? null,
                'pos_fondo' => $row['hasta_fondo'] ?? null,
                'pos_code_full' => $row['hasta_code_full'] ?? null,
                'pos_code' => $row['hasta_code'] ?? null,
            ]);

            $row['detalle_items'] = traza_format_move_items($row['match_skus'] ?? null, $row['match_lotes'] ?? null);
            $row['cantidad_label'] = traza_format_move_quantities(
                $row['total_uv'] ?? null,
                $row['total_uc'] ?? null,
                $row['uv_cajas'] ?? null,
                $row['uc_unidades'] ?? null,
                $row['delta_uv'] ?? null,
                $row['delta_uc'] ?? null
            );
            $row['ref_label'] = traza_format_reference($row['ref_tipo'] ?? null, $row['ref_id'] ?? null);
            $row['duracion_label'] = traza_format_duration($row['asignado_at'] ?? null, $row['finalizado_at'] ?? null);
            return $row;
        }, $rows);
    }
}

if (!function_exists('traza_format_move_items')) {
    function traza_format_move_items(?string $skus, ?string $lotes): string
    {
        $parts = [];
        if ($skus) {
            $parts[] = 'SKU: ' . $skus;
        }
        if ($lotes) {
            $parts[] = 'Lotes: ' . $lotes;
        }
        return $parts ? implode(' · ', $parts) : '';
    }
}

if (!function_exists('traza_format_move_quantities')) {
    function traza_format_move_quantities($totalUv, $totalUc, $legacyUv, $legacyUc, $deltaUv = null, $deltaUc = null): string
    {
        $uv = $totalUv ?? $legacyUv;
        $uc = $totalUc ?? $legacyUc;
        $parts = [];
        if ($uv !== null && (int) $uv !== 0) {
            $parts[] = (int) $uv . ' cajas';
        }
        if ($uc !== null && (int) $uc !== 0) {
            $parts[] = (int) $uc . ' UC';
        }
        if ($deltaUv !== null && (int) $deltaUv !== 0) {
            $deltaUvVal = (int) $deltaUv;
            $parts[] = 'Delta UV ' . ($deltaUvVal > 0 ? '+' : '') . $deltaUvVal;
        }
        if ($deltaUc !== null) {
            $deltaUcVal = (int) $deltaUc;
            $parts[] = 'Delta UC ' . ($deltaUcVal > 0 ? '+' : '') . $deltaUcVal;
        }
        return $parts ? implode(' · ', $parts) : '';
    }
}

if (!function_exists('traza_format_reference')) {
    function traza_format_reference(?string $tipo, ?string $id): string
    {
        if (!$tipo && !$id) {
            return '';
        }
        $tipo = $tipo ? strtoupper(trim($tipo)) : 'REF';
        return $id ? ($tipo . ' #' . $id) : $tipo;
    }
}

if (!function_exists('traza_format_duration')) {
    function traza_format_duration(?string $inicio, ?string $fin): string
    {
        if (!$inicio || !$fin) {
            return '';
        }
        $start = strtotime($inicio);
        $end = strtotime($fin);
        if ($start === false || $end === false || $end <= $start) {
            return '';
        }
        $minutes = (int) round(($end - $start) / 60);
        if ($minutes < 60) {
            return $minutes . ' min';
        }
        $hours = $minutes / 60;
        return number_format($hours, ($hours - floor($hours)) > 0 ? 1 : 0) . ' h';
    }
}

if (!function_exists('traza_build_summary')) {
    /**
     * Construye métricas generales.
     */
    function traza_build_summary(array $pallets, array $movements, array $filters, array $matchStats): array
    {
        $totalPallets = count($pallets);
        $totalItems = 0;
        $uvTotal = 0;
        $ucTotal = 0;

        foreach ($pallets as $pallet) {
            $totalItems += count($pallet['items']);
            $uvTotal += (int) ($pallet['uv_total'] ?? 0);
            $ucTotal += (int) ($pallet['uc_total'] ?? 0);
        }

        $totalMoves = count($movements);
        $fechas = array_column($movements, 'fecha_ref');
        sort($fechas);
        $primerMov = $fechas ? traza_format_datetime(reset($fechas)) : null;
        $ultimoMov = $fechas ? traza_format_datetime(end($fechas)) : null;

        return [
            'total_pallets' => $totalPallets,
            'total_items' => $totalItems,
            'total_movimientos' => $totalMoves,
            'uv_total' => $uvTotal,
            'uc_total' => $ucTotal,
            'primer_movimiento' => $primerMov,
            'ultimo_movimiento' => $ultimoMov,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
            'match_productos' => array_map(static fn(string $sku, string $nombre): array => [
                'sku' => $sku,
                'nombre' => $nombre,
            ], array_keys($matchStats['productos']), array_values($matchStats['productos'])),
            'match_lotes' => array_keys($matchStats['lotes']),
            'match_pallets' => $matchStats['pallets'],
        ];
    }
}

if (!function_exists('traza_merge_stats')) {
    function traza_merge_stats(array $base, array $extra): array
    {
        $merged = $base;
        foreach ($extra['productos'] ?? [] as $sku => $nombre) {
            $merged['productos'][$sku] = $nombre;
        }
        foreach ($extra['lotes'] ?? [] as $lote => $_) {
            $merged['lotes'][$lote] = true;
        }
        foreach ($extra['pallets'] ?? [] as $codigo) {
            if (!in_array($codigo, $merged['pallets'], true)) {
                $merged['pallets'][] = $codigo;
            }
        }
        return $merged;
    }
}

if (!function_exists('traza_normalize_filters')) {
    /**
     * Limpia y normaliza los filtros recibidos desde la petición.
     *
     * @param array<string,mixed> $input
     * @return array{sku:string,lote:string,pallet:string,fecha_desde:string,fecha_hasta:string}
     */
    function traza_normalize_filters(array $input): array
    {
        $skuRaw = trim((string) ($input['sku'] ?? ''));
        if ($skuRaw !== '' && str_contains($skuRaw, '·')) {
            $skuRaw = trim((string) explode('·', $skuRaw, 2)[0]);
        }
        $sku = strtoupper(substr($skuRaw, 0, 80));

        $loteRaw = trim((string) ($input['lote'] ?? ''));
        $lote = strtoupper(substr($loteRaw, 0, 80));

        $palletRaw = trim((string) ($input['pallet'] ?? ''));
        $pallet = strtoupper(substr($palletRaw, 0, 80));

        $today = date('Y-m-d');
        $defaultDesde = date('Y-m-d', strtotime('-90 days'));
        $fechaDesde = traza_sanitize_date($input['fecha_desde'] ?? '', $defaultDesde);
        $fechaHasta = traza_sanitize_date($input['fecha_hasta'] ?? '', $today);

        if ($fechaDesde > $fechaHasta) {
            [$fechaDesde, $fechaHasta] = [$fechaHasta, $fechaDesde];
        }

        if ($fechaHasta > $today) {
            $fechaHasta = $today;
        }

        return [
            'sku' => $sku,
            'lote' => $lote,
            'pallet' => $pallet,
            'fecha_desde' => $fechaDesde,
            'fecha_hasta' => $fechaHasta,
        ];
    }
}

if (!function_exists('traza_search_productos')) {
    /**
     * Devuelve hasta 20 productos que coinciden con el texto indicado.
     */
    function traza_search_productos(PDO $pdo, string $query): array
    {
        $query = trim($query);
        if ($query === '') {
            return [];
        }
        $like = traza_like(strtoupper($query));
        $sql = "
            SELECT id, sku, denominacion
            FROM para_productos
            WHERE UPPER(sku) LIKE ? OR UPPER(denominacion) LIKE ?
            ORDER BY sku ASC
            LIMIT 20
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(1, $like);
        $stmt->bindValue(2, $like);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(static function (array $row): array {
            return [
                'id' => (int) ($row['id'] ?? 0),
                'sku' => (string) ($row['sku'] ?? ''),
                'nombre' => (string) ($row['denominacion'] ?? ''),
            ];
        }, $rows);
    }
}

if (!function_exists('traza_column_exists')) {
    function traza_column_exists(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = spl_object_id($pdo) . ':' . strtolower($table) . ':' . strtolower($column);
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        return $cache[$key] = ((int) $stmt->fetchColumn() > 0);
    }
}

if (!function_exists('traza_table_exists')) {
    function traza_table_exists(PDO $pdo, string $table): bool
    {
        static $cache = [];
        $key = spl_object_id($pdo) . ':' . strtolower($table);
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$table]);
        return $cache[$key] = ((int) $stmt->fetchColumn() > 0);
    }
}

if (!function_exists('traza_format_position_label')) {
    function traza_format_position_label(array $row): string
    {
        return traza_format_position_from_parts([
            'deposito_code' => $row['deposito_code'] ?? null,
            'deposito_nombre' => $row['deposito_nombre'] ?? null,
            'pos_rack' => $row['pos_rack'] ?? null,
            'pos_columna' => $row['pos_columna'] ?? null,
            'pos_nivel' => $row['pos_nivel'] ?? null,
            'pos_fondo' => $row['pos_fondo'] ?? null,
            'pos_code_full' => $row['pos_code_full'] ?? null,
            'pos_code' => $row['pos_code'] ?? null,
        ]);
    }
}

if (!function_exists('traza_format_deposito_label')) {
    function traza_format_deposito_label(array $row): string
    {
        $code = trim((string) ($row['deposito_code'] ?? ''));
        $name = trim((string) ($row['deposito_nombre'] ?? ''));
        if ($code === '' && $name === '') {
            return '';
        }
        if ($code !== '' && $name !== '') {
            return $code . ' · ' . $name;
        }
        return $code !== '' ? $code : $name;
    }
}

if (!function_exists('traza_format_position_from_parts')) {
    function traza_format_position_from_parts(array $parts): string
    {
        $segments = [];
        $dep = '';
        if (!empty($parts['deposito_code'])) {
            $dep = (string) $parts['deposito_code'];
        }
        if (!empty($parts['deposito_nombre'])) {
            $dep = $dep !== '' ? ($dep . ' · ' . $parts['deposito_nombre']) : (string) $parts['deposito_nombre'];
        }
        if ($dep !== '') {
            $segments[] = $dep;
        }
        $rack = $parts['pos_rack'] ?? null;
        $col = $parts['pos_columna'] ?? null;
        $niv = $parts['pos_nivel'] ?? null;
        $fon = $parts['pos_fondo'] ?? null;
        $gridLabel = '';
        if ($rack !== null || $col !== null || $niv !== null) {
            $grid = [];
            if ($rack !== null) {
                $grid[] = 'R' . $rack;
            }
            if ($col !== null) {
                $grid[] = 'C' . $col;
            }
            if ($niv !== null) {
                $grid[] = 'N' . $niv;
            }
            if ($fon !== null) {
                $grid[] = 'F' . $fon;
            }
            if ($grid) {
                $gridLabel = implode('-', $grid);
            }
        }

        $codeFull = trim((string) ($parts['pos_code_full'] ?? ''));
        $codeShort = trim((string) ($parts['pos_code'] ?? ''));
        $codeAdded = false;
        if ($codeFull !== '') {
            $segments[] = $codeFull;
            $codeAdded = true;
        } elseif ($codeShort !== '') {
            $segments[] = $codeShort;
            $codeAdded = true;
        }

        if ($gridLabel !== '' && !$codeAdded) {
            $segments[] = $gridLabel;
        }

        return $segments ? implode(' · ', $segments) : 'Sin posición';
    }
}

if (!function_exists('traza_format_datetime')) {
    function traza_format_datetime($value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }
        $ts = strtotime((string) $value);
        if ($ts === false) {
            return null;
        }
        return date('d/m/Y H:i', $ts);
    }
}

if (!function_exists('traza_format_user_label')) {
    function traza_format_user_label(?string $name, $id): string
    {
        $name = trim((string) $name);
        if ($name !== '') {
            return $name;
        }
        if ($id) {
            return 'Usuario #' . $id;
        }
        return '';
    }
}

if (!function_exists('traza_like')) {
    function traza_like(string $value): string
    {
        $escaped = str_replace(['%', '_'], ['\\%', '\\_'], $value);
        return '%' . $escaped . '%';
    }
}

if (!function_exists('traza_sanitize_date')) {
    function traza_sanitize_date($value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}
