<?php

declare(strict_types=1);

if (!function_exists('so_prep_err_table_name')) {
    function so_prep_err_table_name(): string
    {
        return 'so_pre_error_log';
    }
}

if (!function_exists('so_prep_err_table_exists')) {
    function so_prep_err_table_exists(PDO $pdo, string $table): bool
    {
        static $cache = [];
        if (array_key_exists($table, $cache)) {
            return $cache[$table];
        }

        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$table]);
        $cache[$table] = ((int) $stmt->fetchColumn() > 0);

        return $cache[$table];
    }
}

if (!function_exists('so_prep_err_column_exists')) {
    function so_prep_err_column_exists(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = $table . ':' . $column;

        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }

        if (!so_prep_err_table_exists($pdo, $table)) {
            return $cache[$key] = false;
        }

        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        $cache[$key] = ((int) $stmt->fetchColumn() > 0);

        return $cache[$key];
    }
}

if (!function_exists('so_prep_err_detect_column')) {
    function so_prep_err_detect_column(PDO $pdo, string $table, array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            if ($candidate && so_prep_err_column_exists($pdo, $table, $candidate)) {
                return $candidate;
            }
        }

        return null;
    }
}

if (!function_exists('so_prep_err_fetch_motivos')) {
    function so_prep_err_fetch_motivos(PDO $pdo): array
    {
        $table = 'para_motivos_errores';
        if (!so_prep_err_table_exists($pdo, $table)) {
            return [];
        }

        $labelCol = so_prep_err_detect_column($pdo, $table, ['descripcion', 'nombre', 'detalle', 'motivo', 'titulo', 'label']);
        $activoCol = so_prep_err_detect_column($pdo, $table, ['activo', 'habilitado', 'enabled', 'is_active']);

        $selectLabel = $labelCol ? "`{$labelCol}`" : "CONCAT('Motivo #', id)";
        $sql = "SELECT id, {$selectLabel} AS nombre FROM {$table}";
        if ($activoCol) {
            $sql .= " WHERE COALESCE({$table}.`{$activoCol}`, 1) IN (1, '1', 'S', 'Y', 'SI', 'YES', 'ACTIVO')";
        }
        $sql .= ' ORDER BY nombre';

        $stmt = $pdo->query($sql);
        $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];

        return array_map(static function (array $row): array {
            return [
                'id' => (int) ($row['id'] ?? 0),
                'nombre' => trim((string) ($row['nombre'] ?? '')),
            ];
        }, $rows ?: []);
    }
}

if (!function_exists('so_prep_err_resolve_pre_info')) {
    function so_prep_err_resolve_pre_info(PDO $pdo, int $soId): ?array
    {
        if ($soId <= 0) {
            return null;
        }

        $tblPre = so_prep_err_table_exists($pdo, 'so_preembarque') ? 'so_preembarque' : null;
        if (!$tblPre) {
            return null;
        }

        $pedidoFk = so_prep_err_detect_column($pdo, $tblPre, ['pedido_id', 'so_pedido_id']);
        if ($pedidoFk) {
            $stmt = $pdo->prepare("SELECT id, codigo FROM {$tblPre} WHERE `{$pedidoFk}` = ? ORDER BY id DESC LIMIT 1");
            $stmt->execute([$soId]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($row) {
                return ['id' => (int) $row['id'], 'codigo' => (string) ($row['codigo'] ?? '')];
            }
        }

        if (!so_prep_err_table_exists($pdo, 'so_pedido')) {
            return null;
        }

        $codigoCol = so_prep_err_detect_column($pdo, 'so_pedido', ['codigo', 'code']);
        if (!$codigoCol) {
            return null;
        }

        $stCodigo = $pdo->prepare("SELECT `{$codigoCol}` FROM so_pedido WHERE id = ? LIMIT 1");
        $stCodigo->execute([$soId]);
        $codigo = (string) ($stCodigo->fetchColumn() ?: '');
        if ($codigo === '') {
            return null;
        }

        $stmt = $pdo->prepare("SELECT id, codigo FROM {$tblPre} WHERE codigo = ? ORDER BY id DESC LIMIT 1");
        $stmt->execute(['PRE-' . $codigo]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            return ['id' => (int) $row['id'], 'codigo' => (string) ($row['codigo'] ?? '')];
        }

        return null;
    }
}

if (!function_exists('so_prep_err_fetch_rows')) {
    function so_prep_err_fetch_rows(PDO $pdo, int $soId): array
    {
        $table = so_prep_err_table_name();
        if (!so_prep_err_table_exists($pdo, $table)) {
            return [];
        }

        $motivos = 'para_motivos_errores';
        $motivoLabelCol = so_prep_err_detect_column($pdo, $motivos, ['descripcion', 'nombre', 'detalle', 'motivo', 'titulo', 'label']);
        $usersTable = so_prep_err_table_exists($pdo, 'sys_users') ? 'sys_users' : null;
        $userNameCol = $usersTable ? so_prep_err_detect_column($pdo, $usersTable, ['full_name', 'nombre', 'name']) : null;
        $userAltCol = $usersTable ? so_prep_err_detect_column($pdo, $usersTable, ['username', 'user', 'email']) : null;

        $select = [
            'e.id',
            'e.so_id',
            'e.pre_id',
            'e.motivo_id',
            'e.responsable_user_id',
            'e.responsable_nombre',
            'e.observacion',
            'e.logged_at',
            'e.created_at',
            "DATE_FORMAT(e.logged_at, '%Y-%m-%d %H:%i') AS logged_at_fmt",
            "DATE_FORMAT(e.created_at, '%Y-%m-%d %H:%i') AS created_at_fmt",
        ];

        $joins = [];
        if (so_prep_err_table_exists($pdo, $motivos)) {
            $select[] = $motivoLabelCol ? "m.`{$motivoLabelCol}` AS motivo_nombre" : "CONCAT('Motivo #', m.id) AS motivo_nombre";
            $joins[] = "LEFT JOIN {$motivos} m ON m.id = e.motivo_id";
        } else {
            $select[] = "CONCAT('Motivo #', e.motivo_id) AS motivo_nombre";
        }

        if ($usersTable) {
            $parts = [];
            if ($userNameCol) {
                $parts[] = "u.`{$userNameCol}`";
            }
            if ($userAltCol) {
                $parts[] = "u.`{$userAltCol}`";
            }
            if (!$parts) {
                $parts[] = "CONCAT('Usuario #', u.id)";
            }
            $select[] = 'COALESCE(e.responsable_nombre, ' . implode(', ', $parts) . ') AS responsable_label';
            $joins[] = 'LEFT JOIN ' . $usersTable . ' u ON u.id = e.responsable_user_id';
        } else {
            $select[] = 'e.responsable_nombre AS responsable_label';
        }

        $sql = 'SELECT ' . implode(', ', $select) . " FROM {$table} e";
        if ($joins) {
            $sql .= ' ' . implode(' ', $joins);
        }
        $sql .= ' WHERE e.so_id = :so_id ORDER BY e.logged_at DESC, e.id DESC';

        $stmt = $pdo->prepare($sql);
        $stmt->execute(['so_id' => $soId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return array_map(static function (array $row): array {
            return [
                'id' => (int) ($row['id'] ?? 0),
                'so_id' => (int) ($row['so_id'] ?? 0),
                'pre_id' => isset($row['pre_id']) ? (int) $row['pre_id'] : null,
                'motivo_id' => (int) ($row['motivo_id'] ?? 0),
                'motivo_nombre' => trim((string) ($row['motivo_nombre'] ?? '')),
                'responsable_user_id' => isset($row['responsable_user_id']) ? (int) $row['responsable_user_id'] : null,
                'responsable_nombre' => trim((string) ($row['responsable_nombre'] ?? '')),
                'responsable_label' => trim((string) ($row['responsable_label'] ?? '')),
                'observacion' => trim((string) ($row['observacion'] ?? '')),
                'logged_at' => (string) ($row['logged_at'] ?? ''),
                'logged_at_fmt' => (string) ($row['logged_at_fmt'] ?? ''),
                'created_at' => (string) ($row['created_at'] ?? ''),
                'created_at_fmt' => (string) ($row['created_at_fmt'] ?? ''),
            ];
        }, $rows);
    }
}

if (!function_exists('so_prep_err_insert')) {
    function so_prep_err_insert(PDO $pdo, array $payload): int
    {
        $table = so_prep_err_table_name();
        if (!so_prep_err_table_exists($pdo, $table)) {
            throw new RuntimeException('Tabla de errores no encontrada. Ejecute la migración correspondiente.');
        }

        $stmt = $pdo->prepare(
            "INSERT INTO {$table} (so_id, pre_id, motivo_id, responsable_user_id, responsable_nombre, observacion, logged_at, created_at) " .
            'VALUES (:so_id, :pre_id, :motivo_id, :responsable_user_id, :responsable_nombre, :observacion, :logged_at, NOW())'
        );
        $stmt->execute([
            'so_id' => $payload['so_id'],
            'pre_id' => $payload['pre_id'] ?? null,
            'motivo_id' => $payload['motivo_id'],
            'responsable_user_id' => $payload['responsable_user_id'] ?? null,
            'responsable_nombre' => $payload['responsable_nombre'] ?? null,
            'observacion' => $payload['observacion'] ?? null,
            'logged_at' => $payload['logged_at'],
        ]);

        return (int) $pdo->lastInsertId();
    }
}
