<?php

declare(strict_types=1);

if (!function_exists('so_pre_has_table')) {
    function so_pre_has_table(PDO $pdo, string $name): bool
    {
        static $cache = [];
        if (array_key_exists($name, $cache)) {
            return $cache[$name];
        }

        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$name]);
        $cache[$name] = ((int) $stmt->fetchColumn() > 0);

        return $cache[$name];
    }
}

if (!function_exists('so_pre_has_column')) {
    function so_pre_has_column(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = $table . ':' . $column;

        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }

        if (!so_pre_has_table($pdo, $table)) {
            return $cache[$key] = false;
        }

        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        $cache[$key] = ((int) $stmt->fetchColumn() > 0);

        return $cache[$key];
    }
}

if (!function_exists('so_pre_detect_table')) {
    function so_pre_detect_table(PDO $pdo, array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            if ($candidate && so_pre_has_table($pdo, $candidate)) {
                return $candidate;
            }
        }

        return null;
    }
}

if (!function_exists('so_pre_detect_column')) {
    function so_pre_detect_column(PDO $pdo, string $table, array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            if ($candidate && so_pre_has_column($pdo, $table, $candidate)) {
                return $candidate;
            }
        }

        return null;
    }
}

if (!function_exists('so_pre_fetch_dataset')) {
    function so_pre_fetch_dataset(PDO $pdo, array $opts = []): array
    {
        $preId = isset($opts['pre_id']) ? (int) $opts['pre_id'] : 0;
        $soId  = isset($opts['so_id']) ? (int) $opts['so_id'] : 0;

        $tblPre = 'so_preembarque';
        $tblPrePick = 'so_pre_pick';

        if (!so_pre_has_table($pdo, $tblPre)) {
            return [
                'pre' => null,
                'pedido' => null,
                'rows' => [],
                'totals' => ['uv' => 0, 'uc' => 0],
            ];
        }

        $prePedidoFk = so_pre_detect_column($pdo, $tblPre, ['pedido_id', 'so_pedido_id']);

        if ($preId <= 0 && $soId > 0) {
            if ($prePedidoFk) {
                $st = $pdo->prepare("SELECT id FROM {$tblPre} WHERE `{$prePedidoFk}` = ? ORDER BY id DESC LIMIT 1");
                $st->execute([$soId]);
                $preId = (int) ($st->fetchColumn() ?: 0);
            }

            if ($preId <= 0 && so_pre_has_table($pdo, 'so_pedido') && so_pre_has_column($pdo, 'so_pedido', 'codigo')) {
                $stCodigo = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id = ? LIMIT 1');
                $stCodigo->execute([$soId]);
                $soCodigo = $stCodigo->fetchColumn();
                if ($soCodigo) {
                    $stPre = $pdo->prepare("SELECT id FROM {$tblPre} WHERE codigo = ? ORDER BY id DESC LIMIT 1");
                    $stPre->execute(['PRE-' . $soCodigo]);
                    $preId = (int) ($stPre->fetchColumn() ?: 0);
                }
            }
        }

        if ($preId <= 0) {
            return [
                'pre' => null,
                'pedido' => null,
                'rows' => [],
                'totals' => ['uv' => 0, 'uc' => 0],
            ];
        }

        $select = ['pre.id AS pre_id'];
        $joins = [];

    $preCols = ['codigo', 'pedido_id', 'deposito_id', 'zona_posicion_id', 'estado_id', 'asignado_at', 'inicio_at', 'fin_at', 'created_at', 'updated_at'];
        foreach ($preCols as $col) {
            if (so_pre_has_column($pdo, $tblPre, $col)) {
                $select[] = "pre.`{$col}` AS pre_{$col}";
            }
        }

        $pedidoData = null;
        $pedidoFkExpr = null;
        if ($prePedidoFk && so_pre_has_table($pdo, 'so_pedido')) {
            $joins[] = "LEFT JOIN so_pedido so ON so.id = pre.`{$prePedidoFk}`";
            $pedidoData = [];
            $pedidoFkExpr = "pre.`{$prePedidoFk}`";
            foreach (['codigo', 'cliente_id', 'cliente_nombre', 'cliente', 'fecha_pedido', 'estado_id', 'estado_code'] as $col) {
                if (so_pre_has_column($pdo, 'so_pedido', $col)) {
                    $select[] = "so.`{$col}` AS pedido_{$col}";
                }
            }
        }

        $sqlPre = 'SELECT ' . implode(', ', $select) . " FROM {$tblPre} pre " . implode(' ', $joins) . ' WHERE pre.id = ? LIMIT 1';
        $stPre = $pdo->prepare($sqlPre);
        $stPre->execute([$preId]);
        $preRow = $stPre->fetch(PDO::FETCH_ASSOC) ?: null;

        if (!$preRow) {
            return [
                'pre' => null,
                'pedido' => null,
                'rows' => [],
                'totals' => ['uv' => 0, 'uc' => 0],
            ];
        }

        $preInfo = ['id' => (int) $preRow['pre_id']];
        foreach ($preRow as $key => $value) {
            if (strpos($key, 'pre_') === 0) {
                $field = substr($key, 4);
                $preInfo[$field] = $value;
            }
        }

        $pedidoInfo = null;
        if ($pedidoData !== null) {
            $pedidoInfo = ['id' => null];
            foreach ($preRow as $key => $value) {
                if (strpos($key, 'pedido_') === 0) {
                    $field = substr($key, 7);
                    $pedidoInfo[$field] = $value;
                }
            }
            if ($pedidoInfo !== null && $pedidoFkExpr) {
                $pedidoInfo['id'] = isset($preInfo[$prePedidoFk]) ? (int) $preInfo[$prePedidoFk] : null;
            }
        }

        if (isset($preInfo['deposito_id']) && so_pre_has_table($pdo, 'wh_deposito')) {
            $depCols = [];
            foreach (['codigo', 'code', 'nombre', 'descripcion'] as $col) {
                if (so_pre_has_column($pdo, 'wh_deposito', $col)) {
                    $depCols[] = $col;
                }
            }
            if ($depCols) {
                $colsExpr = [];
                foreach ($depCols as $col) {
                    $colsExpr[] = "d.`{$col}`";
                }
                $depSql = 'SELECT ' . implode(', ', $colsExpr) . ' FROM wh_deposito d WHERE d.id = ? LIMIT 1';
                $stDep = $pdo->prepare($depSql);
                $stDep->execute([(int) $preInfo['deposito_id']]);
                $depRow = $stDep->fetch(PDO::FETCH_ASSOC);
                if ($depRow) {
                    $parts = array_filter([
                        $depRow['codigo'] ?? $depRow['code'] ?? null,
                        $depRow['nombre'] ?? $depRow['descripcion'] ?? null,
                    ], static function ($value) {
                        return $value !== null && $value !== '';
                    });
                    $preInfo['deposito'] = $parts ? implode(' · ', $parts) : '';
                }
            }
            if (!isset($preInfo['deposito']) || !is_string($preInfo['deposito'])) {
                $preInfo['deposito'] = $preInfo['deposito'] ?? '';
                if (!is_string($preInfo['deposito'])) {
                    $preInfo['deposito'] = '';
                }
            }
        }

        if (!so_pre_has_table($pdo, $tblPrePick)) {
            return [
                'pre' => $preInfo,
                'pedido' => $pedidoInfo,
                'rows' => [],
                'totals' => ['uv' => 0, 'uc' => 0],
            ];
        }

        $uvCol = so_pre_detect_column($pdo, $tblPrePick, ['uv_cajas', 'uv', 'cantidad_uv', 'qty_uv']);
        $ucCol = so_pre_detect_column($pdo, $tblPrePick, ['uc_unidades', 'uc', 'cantidad_uc', 'qty_uc']);
        $palletCol = so_pre_detect_column($pdo, $tblPrePick, ['pallet_id']);
        $fromCol = so_pre_detect_column($pdo, $tblPrePick, ['from_pos_id', 'posicion_origen_id', 'posicion_desde_id']);
        $toCol = so_pre_detect_column($pdo, $tblPrePick, ['to_pos_id', 'posicion_destino_id', 'posicion_hasta_id']);
        $itemCol = so_pre_detect_column($pdo, $tblPrePick, ['pedido_dest_item_id', 'pedido_item_id', 'dest_item_id']);
        $loteCol = so_pre_detect_column($pdo, $tblPrePick, ['lote_id']);
        $createdCol = so_pre_detect_column($pdo, $tblPrePick, ['created_at', 'creado_at']);

        $selectRows = ['ppk.id AS pick_id'];
        $selectRows[] = $uvCol ? "COALESCE(ppk.`{$uvCol}`, 0) AS uv_cajas" : '0 AS uv_cajas';
        $selectRows[] = $ucCol ? "COALESCE(ppk.`{$ucCol}`, 0) AS uc_unidades" : '0 AS uc_unidades';

        if ($palletCol) {
            $selectRows[] = "ppk.`{$palletCol}` AS pallet_id";
        } else {
            $selectRows[] = 'NULL AS pallet_id';
        }

        if ($fromCol) {
            $selectRows[] = "ppk.`{$fromCol}` AS from_pos_id";
        } else {
            $selectRows[] = 'NULL AS from_pos_id';
        }

        if ($toCol) {
            $selectRows[] = "ppk.`{$toCol}` AS to_pos_id";
        } else {
            $selectRows[] = 'NULL AS to_pos_id';
        }

        if ($itemCol) {
            $selectRows[] = "ppk.`{$itemCol}` AS pedido_dest_item_id";
        } else {
            $selectRows[] = 'NULL AS pedido_dest_item_id';
        }

        if ($createdCol) {
            $selectRows[] = "ppk.`{$createdCol}` AS created_at";
        }

        $joins = [];

        $pedidoItemCols = [];
        if ($itemCol && so_pre_has_table($pdo, 'so_pedido_dest_item')) {
            $joins[] = "LEFT JOIN so_pedido_dest_item di ON di.id = ppk.`{$itemCol}`";
            foreach (['producto_id', 'lote_codigo', 'descripcion', 'producto_txt', 'producto_nombre'] as $col) {
                if (so_pre_has_column($pdo, 'so_pedido_dest_item', $col)) {
                    $pedidoItemCols[$col] = true;
                }
            }
            if (so_pre_has_column($pdo, 'so_pedido_dest_item', 'pedido_dest_id') && so_pre_has_table($pdo, 'so_pedido_dest')) {
                $joins[] = 'LEFT JOIN so_pedido_dest dest ON dest.id = di.pedido_dest_id';
                if (so_pre_has_column($pdo, 'so_pedido_dest', 'destinatario_nombre')) {
                    $selectRows[] = 'dest.destinatario_nombre AS destinatario_nombre';
                } elseif (so_pre_has_column($pdo, 'so_pedido_dest', 'destinatario')) {
                    $selectRows[] = 'dest.destinatario AS destinatario_nombre';
                }
            }
        }

        $prodTable = null;
        $productoParts = [];
        if (!empty($pedidoItemCols['descripcion'])) {
            $productoParts[] = 'di.descripcion';
        }
        if (!empty($pedidoItemCols['producto_txt'])) {
            $productoParts[] = 'di.producto_txt';
        }
        if (!empty($pedidoItemCols['producto_nombre'])) {
            $productoParts[] = 'di.producto_nombre';
        }
            if ($itemCol && !empty($pedidoItemCols['producto_id'])) {
            $prodTable = so_pre_detect_table($pdo, ['para_productos']);
            if ($prodTable && so_pre_has_column($pdo, $prodTable, 'denominacion')) {
                $joins[] = "LEFT JOIN {$prodTable} prod ON prod.id = di.producto_id";
                $productoParts[] = 'prod.denominacion';
            } elseif ($prodTable && so_pre_has_column($pdo, $prodTable, 'nombre')) {
                $joins[] = "LEFT JOIN {$prodTable} prod ON prod.id = di.producto_id";
                $productoParts[] = 'prod.nombre';
            }
        }
        if (!$productoParts) {
            if ($itemCol && !empty($pedidoItemCols['producto_id'])) {
                $productoParts[] = 'CONCAT("Producto ", di.producto_id)';
            } else {
                $productoParts[] = 'CONCAT("Pick #", ppk.id)';
            }
        }
        $selectRows[] = 'COALESCE(' . implode(', ', $productoParts) . ') AS producto';

        $loteExprParts = [];
        $vtoExprParts = [];
        $loteJoinExpr = null;
        $loteTable = so_pre_detect_table($pdo, ['wh_lote']);
        if ($loteTable) {
            if ($loteCol && so_pre_has_column($pdo, $loteTable, 'id')) {
                $loteJoinExpr = "ppk.`{$loteCol}`";
            }
            if ($itemCol && so_pre_has_column($pdo, 'so_pedido_dest_item', 'lote_id')) {
                if ($loteJoinExpr) {
                    $loteJoinExpr = "COALESCE({$loteJoinExpr}, di.lote_id)";
                } else {
                    $loteJoinExpr = 'di.lote_id';
                }
            }
            if ($loteJoinExpr) {
                $joins[] = "LEFT JOIN {$loteTable} lot ON lot.id = {$loteJoinExpr}";
                foreach (['codigo', 'lote', 'numero', 'code'] as $col) {
                    if (so_pre_has_column($pdo, $loteTable, $col)) {
                        $loteExprParts[] = "lot.`{$col}`";
                    }
                }
                foreach (['fecha_vencimiento', 'vencimiento', 'fecha_vto'] as $col) {
                    if (so_pre_has_column($pdo, $loteTable, $col)) {
                        $vtoExprParts[] = "lot.`{$col}`";
                    }
                }
            }
        }
        if ($itemCol && !empty($pedidoItemCols['lote_codigo'])) {
            $loteExprParts[] = 'di.lote_codigo';
        }
        if (!$loteExprParts) {
            $loteExprParts[] = 'NULL';
        }
        $selectRows[] = 'COALESCE(' . implode(', ', $loteExprParts) . ') AS lote';

        if (!$vtoExprParts && $itemCol) {
            foreach (['fecha_vencimiento', 'vencimiento'] as $col) {
                if (so_pre_has_column($pdo, 'so_pedido_dest_item', $col)) {
                    $vtoExprParts[] = "di.`{$col}`";
                }
            }
        }
        if ($vtoExprParts) {
            $selectRows[] = 'DATE_FORMAT(COALESCE(' . implode(', ', $vtoExprParts) . '), "%Y-%m-%d") AS vto';
        } else {
            $selectRows[] = 'NULL AS vto';
        }

        $posTable = so_pre_detect_table($pdo, ['wh_posicion', 'wh_positions']);
        $posCols = ['code_full', 'pos_code', 'pos_code_full', 'codigo', 'code', 'nombre', 'title'];
        if ($posTable && $fromCol) {
            $joins[] = "LEFT JOIN {$posTable} pf ON pf.id = ppk.`{$fromCol}`";
        }
        if ($posTable && $toCol) {
            $joins[] = "LEFT JOIN {$posTable} pt ON pt.id = ppk.`{$toCol}`";
        }
        $fromParts = [];
        $toParts = [];
        if ($posTable) {
            foreach ($posCols as $col) {
                if ($fromCol && so_pre_has_column($pdo, $posTable, $col)) {
                    $fromParts[] = "pf.`{$col}`";
                }
                if ($toCol && so_pre_has_column($pdo, $posTable, $col)) {
                    $toParts[] = "pt.`{$col}`";
                }
            }
        }
        if (!$fromParts) {
            $fromParts[] = 'NULL';
        }
        if (!$toParts) {
            $toParts[] = 'NULL';
        }
        $selectRows[] = 'COALESCE(' . implode(', ', $fromParts) . ') AS from_pos';
        $selectRows[] = 'COALESCE(' . implode(', ', $toParts) . ') AS to_pos';

        $palletParts = [];
        $palletTable = $palletCol ? so_pre_detect_table($pdo, ['wh_pallet', 'wh_pallets']) : null;
        if ($palletTable && $palletCol) {
            $joins[] = "LEFT JOIN {$palletTable} pal ON pal.id = ppk.`{$palletCol}`";
            foreach (['codigo', 'code', 'codigo_pallet', 'numero', 'label'] as $col) {
                if (so_pre_has_column($pdo, $palletTable, $col)) {
                    $palletParts[] = "pal.`{$col}`";
                }
            }
        }
        if (!$palletParts && $palletCol) {
            $palletParts[] = "CONCAT('PAL-', ppk.`{$palletCol}`)";
        } elseif (!$palletParts) {
            $palletParts[] = "CONCAT('PAL-', ppk.id)";
        }
        $selectRows[] = 'COALESCE(' . implode(', ', $palletParts) . ') AS pallet';

        $sqlRows = 'SELECT ' . implode(', ', $selectRows) . " FROM {$tblPrePick} ppk " . implode(' ', $joins) . ' WHERE ppk.preembarque_id = ? ORDER BY ppk.id ASC';
        $stRows = $pdo->prepare($sqlRows);
        $stRows->execute([$preId]);
        $rows = $stRows->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $totUv = 0;
        $totUc = 0;
        foreach ($rows as &$row) {
            $row['uv_cajas'] = (int) ($row['uv_cajas'] ?? 0);
            $row['uc_unidades'] = (int) ($row['uc_unidades'] ?? 0);
            $totUv += $row['uv_cajas'];
            $totUc += $row['uc_unidades'];

            $row['pallet'] = isset($row['pallet']) && $row['pallet'] !== null && $row['pallet'] !== '' ? (string) $row['pallet'] : ($row['pallet_id'] ? 'PAL-' . $row['pallet_id'] : '');
            $row['from_pos'] = isset($row['from_pos']) && $row['from_pos'] !== null && $row['from_pos'] !== '' ? (string) $row['from_pos'] : ($row['from_pos_id'] ? 'POS-' . $row['from_pos_id'] : '');
            $row['to_pos'] = isset($row['to_pos']) && $row['to_pos'] !== null && $row['to_pos'] !== '' ? (string) $row['to_pos'] : ($row['to_pos_id'] ? 'POS-' . $row['to_pos_id'] : '');
            $row['producto'] = isset($row['producto']) && $row['producto'] !== null && $row['producto'] !== '' ? (string) $row['producto'] : 'Producto';
            $row['lote'] = isset($row['lote']) && $row['lote'] !== null && $row['lote'] !== '' ? (string) $row['lote'] : '';
            $row['vto'] = isset($row['vto']) && $row['vto'] !== null && $row['vto'] !== '' ? (string) $row['vto'] : '';
            if (!isset($row['destinatario_nombre'])) {
                $row['destinatario_nombre'] = '';
            }
        }
        unset($row);

        return [
            'pre' => $preInfo,
            'pedido' => $pedidoInfo,
            'rows' => $rows,
            'totals' => ['uv' => $totUv, 'uc' => $totUc],
        ];
    }
}
