<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Soporte: app/Support/ViewHelpers.php
 *
 * Helpers genéricos reutilizables para vistas y layouts.
 * - No redefine funciones si ya existen (seguros en includes múltiples).
 * - Depende opcionalmente de BASE_PATH y variables de entorno (.env).
 */

// ---------------------------------------------------------------------
// Raíz del proyecto
// ---------------------------------------------------------------------
if (!function_exists('sol_base_path')) {
  /**
   * Retorna la raíz del proyecto.
   * Usa BASE_PATH si está definido; si no, infiere desde app/Support/.
   */
  function sol_base_path(): string
  {
    if (defined('BASE_PATH') && is_string(constant('BASE_PATH'))) {
      return (string) constant('BASE_PATH');
    }
    return dirname(__DIR__, 2); // .../app/Support → raíz
  }
}

// ---------------------------------------------------------------------
// Secciones para el layout (stack de assets, título, etc.)
// ---------------------------------------------------------------------
if (!function_exists('sol_section')) {
  /**
   * Setea / sobreescribe una sección para que el layout la consuma.
   * Ej: sol_section('page_title', 'Productos')
   *     sol_section('page_js', ['/a.js','/b.js'])
   *
   * @param string $name
   * @param mixed  $value
   */
  function sol_section(string $name, $value): void
  {
    $GLOBALS['__sol_sections'][$name] = $value;
  }
}

if (!function_exists('sol_get_section')) {
  /**
   * Obtiene el valor de una sección (o $default si no existe).
   *
   * @param string $name
   * @param mixed  $default
   * @return mixed
   */
  function sol_get_section(string $name, $default = null)
  {
    return $GLOBALS['__sol_sections'][$name] ?? $default;
  }
}

if (!function_exists('sol_append_section')) {
  /**
   * Agrega valores a una sección tipo arreglo, normalizando a array.
   * Útil para ir “pusheando” assets desde distintas partes.
   *
   * @param string          $name
   * @param string|array    $values
   */
  function sol_append_section(string $name, $values): void
  {
    $current = $GLOBALS['__sol_sections'][$name] ?? [];
    if (!is_array($current)) $current = [$current];

    if (!is_array($values)) {
      $values = [$values];
    }

    // concatenación simple, evitando vacíos
    foreach ($values as $v) {
      if ($v === null || $v === '') continue;
      $current[] = $v;
    }

    $GLOBALS['__sol_sections'][$name] = $current;
  }
}

// ---------------------------------------------------------------------
// Rutas de archivos del proyecto
// ---------------------------------------------------------------------
if (!function_exists('project_path')) {
  /**
   * Convierte una ruta relativa a ruta absoluta dentro del proyecto.
   * Ej: project_path('views/layout/_partials/table-def.php')
   */
  function project_path(string $rel): string
  {
    return rtrim(sol_base_path(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($rel, DIRECTORY_SEPARATOR);
  }
}

// ---------------------------------------------------------------------
// Entorno (.env) y URLs
// ---------------------------------------------------------------------
if (!function_exists('env')) {
  /**
   * Lee una variable de entorno:
   * - Primero de $_ENV, luego getenv(), si no existe retorna $default.
   *
   * @param string $key
   * @param mixed  $default
   * @return mixed
   */
  function env(string $key, $default = null)
  {
    return $_ENV[$key] ?? getenv($key) ?: $default;
  }
}

if (!function_exists('url')) {
  /**
   * Construye una URL absoluta basada en BASE_URL (si existe) + $path.
   * - Acepta paths absolutos o relativos; siempre asegura una barra inicial.
   * - Si BASE_URL no está, retorna el path normalizado (relativo a dominio actual).
   *
   * @param string $path
   * @return string
   */
  function url(string $path = '/'): string
  {
    $base = rtrim((string) env('BASE_URL', ''), '/');
    $p    = '/' . ltrim($path, '/');

    // Si BASE_URL está definida, devolvemos absoluta; si no, solo el path.
    if ($base !== '') {
      return $base . $p;
    }
    return $p;
  }
}
