<?php
declare(strict_types=1);

/**
 * DataTables config para jQuery (v1.13.x) + Bootstrap 5 + Buttons + Select + DateTime + Select2 + Editor
 *
 * DEV:   /node_modules     (sirve en desarrollo si expones node_modules por symlink en /public)
 * PROD:  /assets/dist/vendor
 * Editor (comercial): SIEMPRE local y web-accesible debajo de /public, ej: /public/vendor/Editor/{js,css,lib}
 *
 * NOTA IMPORTANTE:
 * - Asegúrate de que los archivos de Editor EXISTAN como rutas web (bajo /public), p. ej.:
 *     /public/vendor/Editor/js/dataTables.editor.min.js
 *     /public/vendor/Editor/js/editor.bootstrap5.min.js
 *     /public/vendor/Editor/css/editor.bootstrap5.css
 *   De lo contrario obtendrás 404. No uses rutas fuera de /public (como vendor/ del back) para scripts CSS/JS.
 */

if (!function_exists('getDataTablesConfig')) {
    function getDataTablesConfig(): array
    {
        $appEnv = $_ENV['APP_ENV'] ?? getenv('APP_ENV') ?? '';
        $isProd = (strtolower((string)$appEnv) === 'production');

        // Asumimos DocumentRoot = .../public
        $docroot  = rtrim((string)($_SERVER['DOCUMENT_ROOT'] ?? ''), DIRECTORY_SEPARATOR);

        // Bases relativas WEB (no filesystem)
        $baseDev  = '/node_modules';
        $baseProd = '/assets/dist/vendor';

        // Helper: verificar existencia en FS usando DOCUMENT_ROOT + ruta web
        $exists = function (string $urlPath) use ($docroot): bool {
            if ($urlPath === '') return false;
            $full = $docroot . str_replace('/', DIRECTORY_SEPARATOR, $urlPath);
            return is_file($full);
        };

        // Elegir path por entorno.
        // En DEV devolvemos /node_modules/... (si no existe igual lo devolvemos para que sea visible en Network)
        // En PROD devolvemos /assets/dist/vendor/...
        $pick = function (string $devPath, string $prodPath) use ($isProd, $baseDev, $baseProd, $exists): string {
            if ($isProd) return $baseProd . $prodPath;
            $devUrl = $baseDev . $devPath;
            return $exists($devUrl) ? $devUrl : $devUrl;
        };

        // ------------------------------------------------------
        // RUTAS de Editor (siempre locales y web-accesibles)
        // Colócalas en /public/vendor/Editor/{js,css,lib}
        // ------------------------------------------------------
        $dtcss = '/vendor/Editor/datables/datatables.min.css';
        $dtjs = '/vendor/Editor/datables/datatables.min.js';
        $editorJsSelect2       = '/vendor/Editor/js/editor.select2.js';
        $editorJsCore       = '/vendor/Editor/js/dataTables.editor.min.js';
        $editorJsBootstrap5 = '/vendor/Editor/js/editor.bootstrap5.min.js';
        $editorCssBootstrap = '/vendor/Editor/css/editor.bootstrap5.css';

        // -------------------
        // JS (en orden)
        // -------------------
        $scripts = [
            // -------------------------
            // Datatables
            // -------------------------
            ['path' => $dtjs,                   'description' => 'DT scripts (local)'],
            ['path' => $editorJsSelect2,        'description' => 'Select2 (local)'],
            // -------------------------
            // Editor (SIEMPRE al final)
            // -------------------------
            ['path' => $editorJsCore,       'description' => 'DT Editor (local)'],
            ['path' => $editorJsBootstrap5, 'description' => 'DT Editor Bootstrap 5 (local)'],
        ];

        // ------------
        // CSS (en orden)
        // ------------
        $styles = [
            // -------------------------
            // Datatables
            // -------------------------
            ['path' => $dtcss, 'description' => 'DT style (local)'],
            // -------------------------
            // Editor CSS (AL FINAL)
            // -------------------------
            ['path' => $editorCssBootstrap, 'description' => 'DT Editor CSS (local)'],
        ];

        return [
            'scripts' => $scripts,
            'styles'  => $styles,
            '_env'    => $isProd ? 'production' : 'development',
        ];
    }
}
