<?php
/**
 * SOL - Sistema de Operaciones Logísticas
 * Punto de configuración global de la app (sin ezSQL).
 * - Carga .env (Dotenv)
 * - Timezone, error reporting, sesiones (con timeout de inactividad)
 * - Constantes globales (BASE_URL, BASE_PATH, etc.)
 * - Helpers: env(), url(), project_path(), generate_breadcrumb(), respond_json()
 * - Conexión PDO disponible vía config/db.php::getPDO()
 */

declare(strict_types=1);

// -----------------------------------------------------------
// Autoload + .env
// -----------------------------------------------------------
$__ROOT = realpath(__DIR__ . '/..');
if ($__ROOT === false) {
    throw new RuntimeException('No se pudo resolver la ruta base del proyecto.');
}

require_once $__ROOT . '/vendor/autoload.php';

if (!isset($_ENV['APP_ENV'])) {
    // Carga segura: no lanza excepción si falta .env, pero respeta valores existentes
    $dotenv = Dotenv\Dotenv::createImmutable($__ROOT);
    $dotenv->safeLoad();
}

// -----------------------------------------------------------
// Utilidad: leer variable de entorno con default
// -----------------------------------------------------------
if (!function_exists('env')) {
    function env(string $key, mixed $default = null): mixed
    {
        // Prioriza $_ENV, luego getenv()
        if (array_key_exists($key, $_ENV)) {
            $val = $_ENV[$key];
        } else {
            $val = getenv($key);
        }
        if ($val === false || $val === null || $val === '') {
            return $default;
        }
        // Normaliza strings "true"/"false"/"null"/"empty"
        $lower = is_string($val) ? strtolower(trim($val)) : $val;
        return match ($lower) {
            'true'  => true,
            'false' => false,
            'null'  => null,
            'empty' => '',
            default => $val,
        };
    }
}

// -----------------------------------------------------------
// Timezone y errores
// -----------------------------------------------------------
date_default_timezone_set((string) env('TIMEZONE', 'UTC'));

$debug = (bool) env('APP_DEBUG', false);
// Build error_reporting level without referencing deprecated constants directly
$level = E_ALL;
if (defined('E_DEPRECATED')) { $level &= ~constant('E_DEPRECATED'); }
if (defined('E_NOTICE'))     { $level &= ~constant('E_NOTICE'); }
error_reporting($level);
ini_set('display_errors', $debug ? '1' : '0');
ini_set('log_errors', $debug ? '1' : '0');

// -----------------------------------------------------------
// Sesión (con timeout de inactividad)
// -----------------------------------------------------------
if (session_status() !== PHP_SESSION_ACTIVE) {
    // Endurece cookies de sesión
    $secure   = !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
    $lifetime = (int) env('SESSION_COOKIE_LIFETIME', 0); // 0 = session cookie
    session_set_cookie_params([
        'lifetime' => $lifetime,
        'path'     => '/',
        'domain'   => '',
        'secure'   => $secure,
        'httponly' => true,
        'samesite' => 'Lax',
    ]);
    session_start();
}

/**
 * Timeout por inactividad (minutos). Por defecto 30.
 * Si se excede, destruye sesión y redirige a login.
 */
$idleMinutes = (int) env('AUTH_IDLE_MINUTES', 30);
if ($idleMinutes > 0) {
    $now = time();
    $last = $_SESSION['__last_activity'] ?? $now;
    if (($now - (int)$last) > ($idleMinutes * 60)) {
        // Cierra sesión y redirige a login (ruta pública)
        $_SESSION = [];
        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000, $params['path'], $params['domain'], $params['secure'], $params['httponly']);
        }
        session_destroy();

        // Calcula URL de login (ajusta si tu router usa otra ruta)
        $baseUrl = rtrim((string) env('BASE_URL', ''), '/');
        $loginPath = '/login';
        // Si tu index maneja rutas en el mismo /public, puedes usar BASE_URL sin /public
        $redirect = $baseUrl !== '' ? $baseUrl . $loginPath : $loginPath;

        header('Location: ' . $redirect);
        exit;
    }
    $_SESSION['__last_activity'] = $now;
}

// -----------------------------------------------------------
// Constantes globales
// -----------------------------------------------------------
if (!defined('BASE_PATH')) {
    define('BASE_PATH', $__ROOT);
}
if (!defined('APP_BASE_PATH')) {
    // Prefijo opcional de rutas (p. ej.: "/sol" o ".")
    define('APP_BASE_PATH', (string) env('APP_BASE_PATH', '.'));
}
if (!defined('BASE_URL')) {
    // URL pública base (http://sol.test, https://midominio.com)
    $bu = rtrim((string) env('BASE_URL', ''), '/');
    define('BASE_URL', $bu);
}

if (!defined('PROYECTO_TITULO')) define('PROYECTO_TITULO', (string) env('APP_NAME', 'SOL'));
if (!defined('PROYECTO_CLIENTE')) define('PROYECTO_CLIENTE', (string) env('APP_CLIENTE', 'Cliente'));
if (!defined('PROYECTO_LEMA')) define('PROYECTO_LEMA', (string) env('APP_LEMA', 'Sistema de Operaciones Logísticas'));
if (!defined('PROYECTO_LOGO_CLIENTE')) define('PROYECTO_LOGO_CLIENTE', (string) env('APP_LOGO', 'default_logo.jpg'));
if (!defined('PROYECTO_LOGO_CLIENTE_DARK')) define('PROYECTO_LOGO_CLIENTE_DARK', (string) env('APP_LOGO_DARK', 'default_logo.jpg'));
if (!defined('IMAGEN_FONDO_INICIO')) define('IMAGEN_FONDO_INICIO', (string) env('INI_BG', '3.jpg'));

// -----------------------------------------------------------
// Helpers de rutas y URLs
// -----------------------------------------------------------
if (!function_exists('project_path')) {
    /**
     * Devuelve una ruta absoluta dentro del proyecto.
     * @throws RuntimeException si no existe.
     */
    function project_path(string $path): string
    {
        $path = ltrim($path, '/\\');
        $abs  = realpath(BASE_PATH . DIRECTORY_SEPARATOR . $path);
        if ($abs === false) {
            throw new RuntimeException("La ruta especificada no existe: {$path}");
        }
        return $abs;
    }
}

if (!function_exists('url')) {
    /**
     * Construye URL absoluta a partir de BASE_URL (+ path relativo).
     * Si BASE_URL está vacío, retorna el path como está (útil en CLI/tests).
     */
    function url(string $path = ''): string
    {
        $path = '/' . ltrim($path, '/');
        return BASE_URL !== '' ? (BASE_URL . $path) : $path;
    }
}

// -----------------------------------------------------------
// Breadcrumb básico para vistas en /vistas/*
// -----------------------------------------------------------
if (!function_exists('generate_breadcrumb')) {
    function generate_breadcrumb(string $filePath): string
    {
        $real = realpath($filePath);
        if ($real === false) return '';
        $norm = str_replace('\\', '/', $real);
        $pos = strpos($norm, '/vistas/');
        if ($pos === false) return '';

        $relative = substr($norm, $pos + strlen('/vistas/'));
        $parts = array_values(array_filter(explode('/', trim($relative, '/'))));
        if (empty($parts)) return '';

        $last = array_pop($parts);
        $pageName = ucwords(str_replace(['.php', '-', '_'], ['', ' ', ' '], $last));

        $html = '<ol class="breadcrumb">';
        $html .= '<li class="breadcrumb-item"><a href="' . htmlspecialchars(url('/')) . '"><i class="iconly-Home icli svg-color"></i></a></li>';
        if (!empty($parts)) {
            foreach ($parts as $segment) {
                $html .= '<li class="breadcrumb-item f-w-400">' . htmlspecialchars(ucwords(str_replace(['-', '_'], [' ', ' '], $segment))) . '</li>';
            }
        }
        $html .= '<li class="breadcrumb-item f-w-400 active">' . htmlspecialchars($pageName) . '</li>';
        $html .= '</ol>';
        return $html;
    }
}

// -----------------------------------------------------------
// Helper JSON para endpoints de backend
// -----------------------------------------------------------
if (!function_exists('respond_json')) {
    function respond_json(mixed $data, int $status = 200): void
    {
        if (!headers_sent()) {
            http_response_code($status);
            header('Content-Type: application/json; charset=utf-8');
        }
        echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }
}

// -----------------------------------------------------------
// Datos de sesión expuestos globalmente (si existen)
// -----------------------------------------------------------
$usuario          = $_SESSION['usuario_id']      ?? null;
$user             = $_SESSION['usuario_login']   ?? null;
$grupoNivel       = $_SESSION['rolecode']        ?? null;
$usuarioN         = $_SESSION['usuario_nombre']  ?? null;
$usuarioRol       = $_SESSION['usuario_rol']     ?? null;
$nombreOperativa  = $_SESSION['nombreOperativa'] ?? null;
$operativa        = $_SESSION['operativa']       ?? null;

// -----------------------------------------------------------
// Conexión PDO (disponible a demanda)
// -----------------------------------------------------------
require_once __DIR__ . '/db.php'; // expone getPDO(): PDO

// Fin de config
