<?php
// config/db.php
// Conexión PDO usando variables de entorno (.env)

require_once __DIR__ . '/../vendor/autoload.php';

// Cargar variables de entorno solo una vez
if (!isset($_ENV['DB_HOST'])) {
    $dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
    $dotenv->load();
}

/**
 * Retorna una conexión PDO a la base de datos (singleton simple)
 *
 * @return PDO
 * @throws PDOException
 */
function getPDO(): PDO
{
    static $pdo = null;
    if ($pdo instanceof PDO) {
        return $pdo;
    }

    $dsn = sprintf(
        "mysql:host=%s;dbname=%s;charset=%s",
        $_ENV['DB_HOST']    ?? 'localhost',
        $_ENV['DB_NAME']    ?? 'sol',
        $_ENV['DB_CHARSET'] ?? 'utf8mb4'
    );

    // Opcionales desde .env
    $persistent = !empty($_ENV['DB_PERSISTENT']); // "1" o "true"
    $timeout    = isset($_ENV['DB_TIMEOUT']) ? (int)$_ENV['DB_TIMEOUT'] : 5;

    $options = [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,           // ← Recomendado
        PDO::ATTR_TIMEOUT            => $timeout,        // opcional (segundos)
    ];

    if ($persistent) {
        $options[PDO::ATTR_PERSISTENT] = true;          // opcional
    }

    $pdo = new PDO(
        $dsn,
        $_ENV['DB_USER'] ?? 'arasa',
        $_ENV['DB_PASS'] ?? 'Chocla1508#@',
        $options
    );

    // Opcional: forzar collation de la sesión si se define en .env (ej.: DB_COLLATION=utf8mb4_0900_ai_ci)
    if (!empty($_ENV['DB_COLLATION'])) {
        $charset  = $_ENV['DB_CHARSET']   ?? 'utf8mb4';
        $collate  = $_ENV['DB_COLLATION'];
        try {
            $pdo->exec(sprintf('SET NAMES %s COLLATE %s', $charset, $collate));
            $pdo->exec("SET collation_connection = '" . str_replace("'", "''", $collate) . "'");
        } catch (Throwable $e) {
            // No romper la app si falla; solo loguear en error_log
            error_log('DB collation session setup failed: ' . $e->getMessage());
        }
    }

    return $pdo;
}

// Alias por conveniencia, si lo usas en otro código
if (!function_exists('get_pdo')) {
    function get_pdo(): PDO {
        return getPDO();
    }
}
