-- 2025-09-30_001_warehouse_core.sql
-- SOL - Sistema de Operaciones Logísticas
-- Paso 1: Núcleo de Depósito (posiciones, pallets, ocupación y movimientos)
-- Notas:
-- - No elimina ni modifica tablas existentes.
-- - FK hacia tablas ya existentes: sys_users (usuarios del sistema) y para_productos.
-- - Todas las tablas usan InnoDB y utf8mb4.

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- =========================================================
-- 1) POSICIONES FÍSICAS DEL DEPÓSITO
--    Modelo 3D: rack, columna, nivel, fondo (cubos).
--    Una posición puede alojar 0..N pallets (capacidad).
-- =========================================================
CREATE TABLE IF NOT EXISTS `wh_positions` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `deposito_id` BIGINT UNSIGNED NOT NULL,
  `rack` INT UNSIGNED NOT NULL,
  `columna` INT UNSIGNED NOT NULL,
  `nivel` INT UNSIGNED NOT NULL,
  `fondo` INT UNSIGNED NOT NULL DEFAULT 1,
  `capacidad_pallets` TINYINT UNSIGNED NOT NULL DEFAULT 1,
  `activo` TINYINT(1) NOT NULL DEFAULT 1,
  `nota` VARCHAR(255) NULL,

  -- Auditoría
  `created_by` BIGINT UNSIGNED NULL,
  `updated_by` BIGINT UNSIGNED NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NULL ON UPDATE CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  UNIQUE KEY `ux_pos_deposito_rack_grid` (`deposito_id`,`rack`,`columna`,`nivel`,`fondo`),
  KEY `ix_pos_deposito` (`deposito_id`),
  KEY `ix_pos_activo` (`activo`),

  CONSTRAINT `fk_pos_created_by` FOREIGN KEY (`created_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_pos_updated_by` FOREIGN KEY (`updated_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =========================================================
-- 2) PALLETS
--    Los pallets son unidades móviles que ocupan posiciones.
--    Un pallet puede contener ítems de un SKU (caso simple) o mixto vía wh_pallet_items.
-- =========================================================
CREATE TABLE IF NOT EXISTS `wh_pallets` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `codigo` VARCHAR(64) NOT NULL,              -- etiqueta/identificador físico
  `estado` ENUM('DISPONIBLE','OCUPADO','CUARENTENA','BLOQUEADO','DAÑADO','BAJA') NOT NULL DEFAULT 'DISPONIBLE',
  `peso_est_kg` DECIMAL(10,3) NULL,           -- opcional, informativo
  `volumen_est_m3` DECIMAL(10,3) NULL,        -- opcional, informativo
  `nota` VARCHAR(255) NULL,

  -- Auditoría
  `created_by` BIGINT UNSIGNED NULL,
  `updated_by` BIGINT UNSIGNED NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NULL ON UPDATE CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  UNIQUE KEY `ux_pallet_codigo` (`codigo`),

  CONSTRAINT `fk_pallet_created_by` FOREIGN KEY (`created_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_pallet_updated_by` FOREIGN KEY (`updated_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =========================================================
-- 3) CONTENIDO DE PALLETS (granularidad por SKU y LOTE)
--    Soporta pallets mixtos. Permite UV (cajas) y UC (unidades).
--    Registra fechas de producción/vencimiento para trazabilidad FEFO.
-- =========================================================
CREATE TABLE IF NOT EXISTS `wh_pallet_items` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `pallet_id` BIGINT UNSIGNED NOT NULL,
  `producto_id` BIGINT UNSIGNED NOT NULL,     -- FK para_productos.id
  `lote` VARCHAR(64) NULL,
  `fecha_produccion` DATE NULL,
  `fecha_vencimiento` DATE NULL,

  `uv_cajas` INT UNSIGNED NOT NULL DEFAULT 0,       -- cantidad de cajas (UV) en el pallet (para este SKU/lote)
  `uc_por_caja` INT UNSIGNED NULL,                  -- si se conoce
  `uc_sueltas` INT UNSIGNED NOT NULL DEFAULT 0,     -- unidades sueltas (UC) fuera de cajas
  `uc_total_cache` INT UNSIGNED AS (
      (IFNULL(`uv_cajas`,0) * IFNULL(`uc_por_caja`,0)) + IFNULL(`uc_sueltas`,0)
  ) STORED,                                         -- cache para consultas rápidas

  `estado` ENUM('OK','CUARENTENA','RESERVADO','DAÑADO') NOT NULL DEFAULT 'OK',
  `nota` VARCHAR(255) NULL,

  -- Auditoría
  `created_by` BIGINT UNSIGNED NULL,
  `updated_by` BIGINT UNSIGNED NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NULL ON UPDATE CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `ix_pallet_items_pallet` (`pallet_id`),
  KEY `ix_pallet_items_producto` (`producto_id`),
  KEY `ix_pallet_items_lote` (`lote`),
  KEY `ix_pallet_items_venc` (`fecha_vencimiento`),

  CONSTRAINT `fk_pi_pallet` FOREIGN KEY (`pallet_id`) REFERENCES `wh_pallets`(`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_pi_producto` FOREIGN KEY (`producto_id`) REFERENCES `para_productos`(`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_pi_created_by` FOREIGN KEY (`created_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_pi_updated_by` FOREIGN KEY (`updated_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =========================================================
-- 4) OCUPACIÓN DE POSICIONES (histórico)
--    Intervalos [desde, hasta). Si hasta es NULL -> ocupado actualmente.
--    Permite N pallets por posición (capacidad_pallets).
-- =========================================================
CREATE TABLE IF NOT EXISTS `wh_position_occupancy` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `position_id` BIGINT UNSIGNED NOT NULL,
  `pallet_id` BIGINT UNSIGNED NOT NULL,

  `desde` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `hasta` DATETIME NULL,

  `mov_ingreso_id` BIGINT UNSIGNED NULL,  -- referencia al movimiento que ingresó
  `mov_salida_id`  BIGINT UNSIGNED NULL,  -- referencia al movimiento que retiró

  -- Auditoría
  `created_by` BIGINT UNSIGNED NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  UNIQUE KEY `ux_posocc_unico_abierto` (`position_id`, `pallet_id`, `hasta`),
  KEY `ix_posocc_position` (`position_id`),
  KEY `ix_posocc_pallet` (`pallet_id`),

  CONSTRAINT `fk_posocc_pos` FOREIGN KEY (`position_id`) REFERENCES `wh_positions`(`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_posocc_pallet` FOREIGN KEY (`pallet_id`) REFERENCES `wh_pallets`(`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_posocc_mov_in` FOREIGN KEY (`mov_ingreso_id`) REFERENCES `wh_moves`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_posocc_mov_out` FOREIGN KEY (`mov_salida_id`)  REFERENCES `wh_moves`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_posocc_created_by` FOREIGN KEY (`created_by`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =========================================================
-- 5) MOVIMIENTOS (TRAZABILIDAD)
--    Registra todos los cambios: ingresos, reubicaciones, ajustes, bajas.
--    Mide tiempo de ejecución (asignado → inicio → fin) y responsables.
-- =========================================================
CREATE TABLE IF NOT EXISTS `wh_moves` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `tipo` ENUM('INGRESO','REUBICACION','AJUSTE','BAJA') NOT NULL,

  `pallet_id` BIGINT UNSIGNED NOT NULL,

  `desde_position_id` BIGINT UNSIGNED NULL,
  `hasta_position_id` BIGINT UNSIGNED NULL,

  -- Cantidades relacionadas al movimiento (por si no se mueve todo el pallet)
  `uv_cajas` INT UNSIGNED NULL,
  `uc_unidades` INT UNSIGNED NULL,

  -- Referencias externas (ej: doc ingreso, orden interna, incidencia)
  `ref_tipo` VARCHAR(40) NULL,
  `ref_id`   VARCHAR(64) NULL,

  `motivo` VARCHAR(255) NULL,

  -- Asignación y tiempos (para medir productividad)
  `asignado_a` BIGINT UNSIGNED NULL,           -- usuario asignado (ej: montacarguista)
  `asignado_at` DATETIME NULL,
  `iniciado_at` DATETIME NULL,
  `finalizado_at` DATETIME NULL,

  -- Calculado (vista/consulta): duración total y efectiva en segundos
  -- (no se define columna calculada aquí para evitar incompatibilidades, se podrá crear una VIEW luego)

  -- Auditoría
  `solicitado_por` BIGINT UNSIGNED NULL,
  `ejecutado_por` BIGINT UNSIGNED NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `ix_moves_tipo` (`tipo`),
  KEY `ix_moves_pallet` (`pallet_id`),
  KEY `ix_moves_desde` (`desde_position_id`),
  KEY `ix_moves_hasta` (`hasta_position_id`),
  KEY `ix_moves_asignado` (`asignado_a`),
  KEY `ix_moves_tiempos` (`asignado_at`,`iniciado_at`,`finalizado_at`),

  CONSTRAINT `fk_moves_pallet` FOREIGN KEY (`pallet_id`) REFERENCES `wh_pallets`(`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_moves_desde_pos` FOREIGN KEY (`desde_position_id`) REFERENCES `wh_positions`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_moves_hasta_pos` FOREIGN KEY (`hasta_position_id`) REFERENCES `wh_positions`(`id`) ON DELETE SET NULL,

  CONSTRAINT `fk_moves_asignado_a` FOREIGN KEY (`asignado_a`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_moves_solicitado_por` FOREIGN KEY (`solicitado_por`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_moves_ejecutado_por` FOREIGN KEY (`ejecutado_por`) REFERENCES `sys_users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =========================================================
-- 6) RELACIÓN MOVIMIENTO ↔ ÍTEMS (detalle granular por SKU/LOTE)
--    Para trazabilidad exacta cuando un movimiento afecta a parte del contenido.
-- =========================================================
CREATE TABLE IF NOT EXISTS `wh_move_items` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `move_id` BIGINT UNSIGNED NOT NULL,
  `pallet_item_id` BIGINT UNSIGNED NOT NULL,

  `uv_cajas` INT UNSIGNED NULL,
  `uc_unidades` INT UNSIGNED NULL,

  PRIMARY KEY (`id`),
  KEY `ix_mvit_move` (`move_id`),
  KEY `ix_mvit_item` (`pallet_item_id`),

  CONSTRAINT `fk_mvit_move` FOREIGN KEY (`move_id`) REFERENCES `wh_moves`(`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_mvit_item` FOREIGN KEY (`pallet_item_id`) REFERENCES `wh_pallet_items`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =========================================================
-- 7) VISTA OPCIONAL (duración en segundos) - comentar si tu versión no soporta
--    Si tu MySQL es 8+, puedes activar esta vista útil para reportes.
-- =========================================================
DROP VIEW IF EXISTS `vw_wh_moves_duracion`;
CREATE VIEW `vw_wh_moves_duracion` AS
SELECT
  m.`id`,
  m.`tipo`,
  m.`pallet_id`,
  m.`desde_position_id`,
  m.`hasta_position_id`,
  m.`asignado_a`,
  m.`asignado_at`,
  m.`iniciado_at`,
  m.`finalizado_at`,
  TIMESTAMPDIFF(SECOND, m.`asignado_at`, m.`finalizado_at`) AS `duracion_total_seg`,
  TIMESTAMPDIFF(SECOND, m.`iniciado_at`,  m.`finalizado_at`) AS `duracion_efectiva_seg`
FROM `wh_moves` m;

SET FOREIGN_KEY_CHECKS = 1;
