-- 2025-10-01_T001_fk_verificacion_movimientos.sql
-- Verificación integral de FK y flujo básico: wh_moves ↔ wh_position_occupancy
-- Crea datos mínimos (si no existen) y prueba INGRESO y REUBICACION, además de ON DELETE SET NULL.

SET NAMES utf8mb4;
SET @db := DATABASE();

-- --------------------------------------------------------------------
-- 0) Helper: obtener un usuario cualquiera (o crear uno de prueba)
-- --------------------------------------------------------------------
-- Intentar recuperar un usuario existente
SET @user_id := (SELECT id FROM sys_users ORDER BY id LIMIT 1);

-- Si no existe ninguno, crear uno “tester” y capturar su ID
INSERT INTO sys_users (username, pass_hash, full_name, role_code, created_at)
SELECT 'tester', '$2y$10$abcdefghijklmnopqrstuv', 'Tester', 'ADMIN', NOW()
WHERE NOT EXISTS (SELECT 1 FROM sys_users);

-- En cualquier caso, refrescar @user_id
SET @user_id := (SELECT id FROM sys_users ORDER BY id LIMIT 1);


-- --------------------------------------------------------------------
-- 1) Pallet de prueba (si no existe)
-- --------------------------------------------------------------------
SET @pallet_id := (SELECT id FROM wh_pallets WHERE codigo='PAL-TEST-0001' LIMIT 1);
IF @pallet_id IS NULL THEN
  INSERT INTO wh_pallets (codigo, estado, created_by, created_at)
  VALUES ('PAL-TEST-0001', 'DISPONIBLE', @user_id, NOW());
  SET @pallet_id := LAST_INSERT_ID();
END IF;

-- --------------------------------------------------------------------
-- 2) Posición A (si no existe) y Posición B (para reubicación)
--    Ajusta deposito_id según tu catálogo de depósitos (usamos 1 por defecto)
-- --------------------------------------------------------------------
SET @posA := (SELECT id FROM wh_positions 
              WHERE deposito_id=1 AND rack=1 AND columna=1 AND nivel=1 AND fondo=1 LIMIT 1);
IF @posA IS NULL THEN
  INSERT INTO wh_positions (deposito_id, rack, columna, nivel, fondo, capacidad_pallets, activo, created_by)
  VALUES (1, 1, 1, 1, 1, 1, 1, @user_id);
  SET @posA := LAST_INSERT_ID();
END IF;

SET @posB := (SELECT id FROM wh_positions 
              WHERE deposito_id=1 AND rack=1 AND columna=2 AND nivel=1 AND fondo=1 LIMIT 1);
IF @posB IS NULL THEN
  INSERT INTO wh_positions (deposito_id, rack, columna, nivel, fondo, capacidad_pallets, activo, created_by)
  VALUES (1, 1, 2, 1, 1, 1, 1, @user_id);
  SET @posB := LAST_INSERT_ID();
END IF;

-- --------------------------------------------------------------------
-- 3) Movimiento de INGRESO (crea wh_moves y abre ocupación en Posición A)
-- --------------------------------------------------------------------
INSERT INTO wh_moves (tipo, pallet_id, hasta_position_id, motivo, asignado_a, asignado_at, iniciado_at, finalizado_at, solicitado_por, ejecutado_por, created_at)
VALUES ('INGRESO', @pallet_id, @posA, 'Ingreso inicial de prueba', @user_id, NOW(), NOW(), DATE_ADD(NOW(), INTERVAL 2 MINUTE), @user_id, @user_id, NOW());
SET @mov_ing := LAST_INSERT_ID();

-- Abrir ocupación en A
INSERT INTO wh_position_occupancy (position_id, pallet_id, desde, mov_ingreso_id, created_by, created_at)
VALUES (@posA, @pallet_id, NOW(), @mov_ing, @user_id, NOW());
SET @occA := LAST_INSERT_ID();

-- --------------------------------------------------------------------
-- 4) Reubicación A -> B (crea wh_moves, cierra ocupación A y abre en B)
-- --------------------------------------------------------------------
INSERT INTO wh_moves (tipo, pallet_id, desde_position_id, hasta_position_id, motivo, asignado_a, asignado_at, iniciado_at, finalizado_at, solicitado_por, ejecutado_por, created_at)
VALUES ('REUBICACION', @pallet_id, @posA, @posB, 'Mover a Posición B', @user_id, NOW(), NOW(), DATE_ADD(NOW(), INTERVAL 3 MINUTE), @user_id, @user_id, NOW());
SET @mov_reub := LAST_INSERT_ID();

-- Cerrar ocupación en A con referencia al movimiento de salida
UPDATE wh_position_occupancy
SET hasta = NOW(), mov_salida_id = @mov_reub
WHERE id = @occA;

-- Abrir ocupación en B con referencia al movimiento de ingreso (=reubicación)
INSERT INTO wh_position_occupancy (position_id, pallet_id, desde, mov_ingreso_id, created_by, created_at)
VALUES (@posB, @pallet_id, NOW(), @mov_reub, @user_id, NOW());
SET @occB := LAST_INSERT_ID();

-- --------------------------------------------------------------------
-- 5) Verificaciones de integridad y joins
-- --------------------------------------------------------------------
-- 5.1 Listar FKs esperados en wh_position_occupancy → wh_moves
SELECT 
  k.CONSTRAINT_NAME, k.TABLE_NAME, k.COLUMN_NAME, 
  k.REFERENCED_TABLE_NAME, k.REFERENCED_COLUMN_NAME
FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE k
WHERE k.TABLE_SCHEMA = @db
  AND k.TABLE_NAME = 'wh_position_occupancy'
  AND k.REFERENCED_TABLE_NAME = 'wh_moves';

-- 5.2 Verificar vínculo mov_ingreso_id (debería existir para @occA y @occB)
SELECT o.id AS occ_id, o.position_id, o.pallet_id, o.desde, o.hasta,
       o.mov_ingreso_id, mi.tipo AS mov_ingreso_tipo
FROM wh_position_occupancy o
LEFT JOIN wh_moves mi ON mi.id = o.mov_ingreso_id
WHERE o.id IN (@occA, @occB);

-- 5.3 Verificar vínculo mov_salida_id (debería existir para @occA)
SELECT o.id AS occ_id, o.mov_salida_id, ms.tipo AS mov_salida_tipo
FROM wh_position_occupancy o
LEFT JOIN wh_moves ms ON ms.id = o.mov_salida_id
WHERE o.id = @occA;

-- --------------------------------------------------------------------
-- 6) Probar ON DELETE SET NULL en mov_salida_id:
--    Borramos el movimiento de reubicación y validamos que mov_salida_id quede NULL
--    (La ocupación @occA NO se borra, solo pierde la referencia)
-- --------------------------------------------------------------------
DELETE FROM wh_moves WHERE id = @mov_reub;

SELECT o.id AS occ_id, o.position_id, o.pallet_id, o.desde, o.hasta,
       o.mov_ingreso_id, o.mov_salida_id
FROM wh_position_occupancy o
WHERE o.id = @occA;

-- Observación: la ocupación @occB quedó con mov_ingreso_id apuntando al movimiento borrado.
-- Según la FK definida (ON DELETE SET NULL), ahora debe estar NULL también si lo referenciaba.
SELECT o.id AS occ_id, o.mov_ingreso_id
FROM wh_position_occupancy o
WHERE o.id = @occB;

-- --------------------------------------------------------------------
-- 7) Limpieza opcional (comentado por defecto)
-- --------------------------------------------------------------------
-- DELETE FROM wh_position_occupancy WHERE id IN (@occA, @occB);
-- DELETE FROM wh_moves WHERE id = @mov_ing;
-- DELETE FROM wh_pallets WHERE id = @pallet_id AND codigo='PAL-TEST-0001';
