/* =============================================================
 * Update sp_so_preparar_auto: prioriza PICKING y repone pallets
 * Si no hay suficiente en PICKING, mueve pallet entero desde
 * ambientes operativos (excluye PREP/CUARENTENA) hacia PICKING,
 * y luego hace picking desde PICKING a PREP.
 *
 * Compatibilidad pma: usar delimitador UI $$
 * ============================================================= */

SET NAMES utf8mb4 $$

DROP PROCEDURE IF EXISTS sp_so_preparar_auto $$
CREATE PROCEDURE sp_so_preparar_auto(
  IN p_pedido_codigo   VARCHAR(64),
  IN p_deposito_code   VARCHAR(32),
  IN p_pos_prep_code   VARCHAR(64)  -- NULL = seleccionar PREP menos ocupada
)
BEGIN
  DECLARE v_pedido_id    BIGINT UNSIGNED;
  DECLARE v_dep_id       INT UNSIGNED;
  DECLARE v_pos_prep_id  BIGINT UNSIGNED;
  DECLARE v_pos_pick_id  BIGINT UNSIGNED;
  DECLARE v_done         INT DEFAULT 0;

  DECLARE v_item_id      BIGINT UNSIGNED;
  DECLARE v_prod_id      BIGINT UNSIGNED;
  DECLARE v_lote_code    VARCHAR(64);
  DECLARE v_dest_id      BIGINT UNSIGNED;
  DECLARE v_need_uv      INT;
  DECLARE v_need_uc      INT;

  DECLARE v_pos_from     BIGINT UNSIGNED;
  DECLARE v_lote_id      BIGINT UNSIGNED;
  DECLARE v_pallet_id    BIGINT UNSIGNED;
  DECLARE v_take_uv      INT;
  DECLARE v_take_uc      INT;
  DECLARE v_move_id      BIGINT UNSIGNED;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK; RESIGNAL;
  END;

  IF p_pedido_codigo IS NULL OR p_pedido_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_pedido_codigo es requerido.';
  END IF;

  SELECT id INTO v_pedido_id FROM so_pedido WHERE codigo=p_pedido_codigo LIMIT 1;
  IF v_pedido_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Pedido no encontrado.';
  END IF;

  SELECT id INTO v_dep_id FROM wh_deposito WHERE code=p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Depósito no encontrado.';
  END IF;

  /* PREP position */
  IF p_pos_prep_code IS NOT NULL AND p_pos_prep_code <> '' THEN
    SELECT id INTO v_pos_prep_id
      FROM wh_posicion
     WHERE deposito_id=v_dep_id
       AND (code=p_pos_prep_code OR code_full=p_pos_prep_code OR pos_code=p_pos_prep_code OR pos_code_full=p_pos_prep_code)
     LIMIT 1;
    IF v_pos_prep_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Posición PREP indicada no existe en el depósito.';
    END IF;
  ELSE
    SELECT p.id INTO v_pos_prep_id
      FROM wh_posicion p
      JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP'
      LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=v_dep_id
     WHERE p.deposito_id=v_dep_id AND p.activo=1
     GROUP BY p.id
     ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
     LIMIT 1;
    IF v_pos_prep_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='No hay posiciones PREP en el depósito.';
    END IF;
  END IF;

  /* PICKING position (menos ocupada) */
  SELECT p.id INTO v_pos_pick_id
    FROM wh_posicion p
    JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
    LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=v_dep_id
   WHERE p.deposito_id=v_dep_id AND p.activo=1
   GROUP BY p.id
   ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
   LIMIT 1;
  IF v_pos_pick_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='No hay posiciones PICKING en el depósito.';
  END IF;

  START TRANSACTION;

  /* Ítems pendientes */
  DROP TEMPORARY TABLE IF EXISTS _pend;
  CREATE TEMPORARY TABLE _pend (
    pedido_dest_item_id BIGINT UNSIGNED,
    pedido_dest_id      BIGINT UNSIGNED,
    destinatario_id     BIGINT UNSIGNED,
    producto_id         BIGINT UNSIGNED,
    lote_codigo         VARCHAR(64),
    need_uv             INT,
    need_uc             INT
  ) ENGINE=InnoDB;

  INSERT INTO _pend
  SELECT i.id, d.id, d.destinatario_id, i.producto_id, i.lote_codigo,
         GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv,
         GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc
  FROM so_pedido_dest_item i
  JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
  WHERE d.pedido_id = v_pedido_id
    AND (GREATEST(i.expected_uv - i.prepared_uv,0) > 0
      OR GREATEST(i.expected_uc - i.prepared_uc,0) > 0);

  /* Cursor por item */
  BEGIN
    DECLARE cur_it CURSOR FOR
      SELECT pedido_dest_item_id, producto_id, lote_codigo, destinatario_id, need_uv, need_uc
      FROM _pend;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done=1;

    SET v_done=0;
    OPEN cur_it;
    it_loop: LOOP
      FETCH cur_it INTO v_item_id, v_prod_id, v_lote_code, v_dest_id, v_need_uv, v_need_uc;
      IF v_done=1 THEN LEAVE it_loop; END IF;

      /* 1) Consumir primero desde PICKING (FEFO) */
      DROP TEMPORARY TABLE IF EXISTS _cand_pick;
      CREATE TEMPORARY TABLE _cand_pick (
        posicion_id BIGINT UNSIGNED,
        lote_id     BIGINT UNSIGNED,
        pallet_id   BIGINT UNSIGNED,
        qty_uv      INT,
        qty_uc      INT,
        venc        DATE
      ) ENGINE=InnoDB;

      INSERT INTO _cand_pick (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
      SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
      FROM wh_stock s
      JOIN wh_posicion p   ON p.id=s.posicion_id
      JOIN wh_ambiente a   ON a.id=p.ambiente_id AND a.code='PICKING'
      LEFT JOIN wh_lote l  ON l.id=s.lote_id
      WHERE s.deposito_id = v_dep_id
        AND s.producto_id = v_prod_id
        AND (v_lote_code IS NULL OR v_lote_code='' OR EXISTS(
              SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=v_lote_code))
        AND (s.qty_uv > 0 OR s.qty_uc > 0)
      ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;

      WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
        SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc
          INTO v_pos_from, v_lote_id, v_pallet_id, v_take_uv, v_take_uc
        FROM _cand_pick ORDER BY COALESCE(venc,'9999-12-31') ASC LIMIT 1;

        IF v_pos_from IS NULL THEN LEAVE; END IF;

        SET v_take_uv = LEAST(COALESCE(v_take_uv,0), v_need_uv);
        SET v_take_uc = LEAST(COALESCE(v_take_uc,0), v_need_uc);

        IF v_take_uv=0 AND v_take_uc=0 THEN
          DELETE FROM _cand_pick LIMIT 1; ITERATE;
        END IF;

        INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
        VALUES (v_dep_id, 'MOVE', 'PREPARACION', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_prep_id, -v_take_uv, -v_take_uc, CONCAT('PRE-', v_pedido_id));
        SET v_move_id = LAST_INSERT_ID();

        INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por)
        VALUES ((SELECT id FROM so_preembarque WHERE codigo=CONCAT('PRE-', p_pedido_codigo) LIMIT 1), v_item_id, v_pos_from, v_pos_prep_id, v_pallet_id, v_lote_id, v_take_uv, v_take_uc, NULL);

        SET v_need_uv = v_need_uv - v_take_uv;
        SET v_need_uc = v_need_uc - v_take_uc;

        UPDATE _cand_pick SET qty_uv=qty_uv - v_take_uv, qty_uc=qty_uc - v_take_uc
         WHERE posicion_id=v_pos_from AND lote_id=v_lote_id AND COALESCE(pallet_id,0)=COALESCE(v_pallet_id,0);
        DELETE FROM _cand_pick WHERE qty_uv<=0 AND qty_uc<=0 LIMIT 1;
      END WHILE;
      DROP TEMPORARY TABLE IF EXISTS _cand_pick;

      /* 2) Si falta, reponer pallet entero de otros ambientes hacia PICKING y volver a intentar */
      WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
        DROP TEMPORARY TABLE IF EXISTS _src_pal;
        CREATE TEMPORARY TABLE _src_pal (
          from_pos_id BIGINT UNSIGNED,
          pallet_id   BIGINT UNSIGNED,
          lote_id     BIGINT UNSIGNED,
          venc        DATE
        ) ENGINE=InnoDB;

        INSERT INTO _src_pal (from_pos_id, pallet_id, lote_id, venc)
        SELECT s.posicion_id, s.pallet_id, s.lote_id, l.fecha_vencimiento
          FROM wh_stock s
          JOIN wh_posicion p ON p.id=s.posicion_id
          JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING')
          LEFT JOIN wh_lote l ON l.id=s.lote_id
         WHERE s.deposito_id=v_dep_id AND s.producto_id=v_prod_id
           AND (v_lote_code IS NULL OR v_lote_code='' OR EXISTS(
                 SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=v_lote_code))
           AND s.pallet_id IS NOT NULL
         GROUP BY s.pallet_id
         ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;

        SELECT from_pos_id, pallet_id, lote_id INTO v_pos_from, v_pallet_id, v_lote_id FROM _src_pal LIMIT 1;
        DROP TEMPORARY TABLE IF EXISTS _src_pal;

        IF v_pallet_id IS NULL THEN
          LEAVE it_loop; -- no hay más pallets para reponer
        END IF;

        /* Mover pallet entero a PICKING */
        INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
        VALUES (v_dep_id, 'MOVE', 'REPOSICION_PICKING', v_pallet_id, NULL, NULL, v_pos_from, v_pos_pick_id, 0, 0, CONCAT('PREP-REP-', v_pedido_id));
        SET v_move_id = LAST_INSERT_ID();

        /* Intentar nuevamente consumir desde PICKING */
        DROP TEMPORARY TABLE IF EXISTS _cand_pick2;
        CREATE TEMPORARY TABLE _cand_pick2 (
          posicion_id BIGINT UNSIGNED,
          lote_id     BIGINT UNSIGNED,
          pallet_id   BIGINT UNSIGNED,
          qty_uv      INT,
          qty_uc      INT,
          venc        DATE
        ) ENGINE=InnoDB;

        INSERT INTO _cand_pick2 (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
        SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
          FROM wh_stock s
          JOIN wh_posicion p   ON p.id=s.posicion_id
          JOIN wh_ambiente a   ON a.id=p.ambiente_id AND a.code='PICKING'
          LEFT JOIN wh_lote l  ON l.id=s.lote_id
         WHERE s.deposito_id = v_dep_id
           AND s.producto_id = v_prod_id
           AND (v_lote_code IS NULL OR v_lote_code='' OR EXISTS(
                 SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=v_lote_code))
           AND (s.qty_uv > 0 OR s.qty_uc > 0)
         ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;

        WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
          SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc
            INTO v_pos_from, v_lote_id, v_pallet_id, v_take_uv, v_take_uc
          FROM _cand_pick2 ORDER BY COALESCE(venc,'9999-12-31') ASC LIMIT 1;

          IF v_pos_from IS NULL THEN LEAVE; END IF;

          SET v_take_uv = LEAST(COALESCE(v_take_uv,0), v_need_uv);
          SET v_take_uc = LEAST(COALESCE(v_take_uc,0), v_need_uc);

          IF v_take_uv=0 AND v_take_uc=0 THEN
            DELETE FROM _cand_pick2 LIMIT 1; ITERATE;
          END IF;

          INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
          VALUES (v_dep_id, 'MOVE', 'PREPARACION', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_prep_id, -v_take_uv, -v_take_uc, CONCAT('PRE-', v_pedido_id));
          SET v_move_id = LAST_INSERT_ID();

          INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por)
          VALUES ((SELECT id FROM so_preembarque WHERE codigo=CONCAT('PRE-', p_pedido_codigo) LIMIT 1), v_item_id, v_pos_from, v_pos_prep_id, v_pallet_id, v_lote_id, v_take_uv, v_take_uc, NULL);

          SET v_need_uv = v_need_uv - v_take_uv;
          SET v_need_uc = v_need_uc - v_take_uc;

          UPDATE _cand_pick2 SET qty_uv=qty_uv - v_take_uv, qty_uc=qty_uc - v_take_uc LIMIT 1;
          DELETE FROM _cand_pick2 WHERE qty_uv<=0 AND qty_uc<=0 LIMIT 1;
        END WHILE;
        DROP TEMPORARY TABLE IF EXISTS _cand_pick2;

        /* Si aún hay faltante, el while exterior continuará reponiendo más pallets */
        IF NOT (v_need_uv > 0 OR v_need_uc > 0) THEN LEAVE; END IF;
      END WHILE; /* reposiciones */
    END LOOP;
    CLOSE cur_it;
  END; /* cursor */

  COMMIT;

  /* Resultado */
  SELECT d.pedido_id, i.id AS pedido_dest_item_id, i.producto_id, i.lote_codigo,
         i.expected_uv, i.prepared_uv, (i.expected_uv - i.prepared_uv) AS pend_uv,
         i.expected_uc, i.prepared_uc, (i.expected_uc - i.prepared_uc) AS pend_uc
  FROM so_pedido_dest_item i
  JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
  WHERE d.pedido_id=v_pedido_id;
END $$
