/**
 * SOL - Sistema de Operaciones Logísticas
 * Migración: 20251004_sys_hardening.sql
 *
 * Objetivo:
 * - Unificar collation/charset a utf8mb4_unicode_ci en tablas sys_*.
 * - Agregar claves únicas en columnas naturales (username, email, role.name, operativas.nombre).
 * - Asegurar unicidad de permisos por (role_id, menu_id).
 * - Asegurar integridad jerárquica de sys_menu vía FK parent_id → sys_menu(id).
 * - Crear índices de apoyo para consultas frecuentes.
 *
 * Modo de uso (phpMyAdmin):
 * 1) Opcional: ejecute primero SOLO el bloque "CHEQUEOS PREVIOS" para identificar duplicados.
 * 2) Corrija duplicados si aparecen resultados.
 * 3) Ejecute el resto del archivo completo.
 *
 * Notas:
 * - Requiere MySQL 8.0+.
 * - CREATE INDEX IF NOT EXISTS está disponible en MySQL 8.0.13+.
 * - Esta migración NO agrega columnas nuevas, solo endurece integridad/consistencia.
 */

-- =========================================================
-- 0) CHEQUEOS PREVIOS (identifique duplicados antes de UNIQUE)
--    Si estas consultas devuelven filas, resuélvalas antes de seguir.
-- =========================================================

-- Usuarios: username duplicado
SELECT 'DUP_USERNAME' AS check_name, username, COUNT(*) AS cnt
FROM sys_users
WHERE deleted_at IS NULL
GROUP BY username
HAVING COUNT(*) > 1;

-- Usuarios: email duplicado (si utiliza unicidad de correo)
SELECT 'DUP_EMAIL' AS check_name, email, COUNT(*) AS cnt
FROM sys_users
WHERE email IS NOT NULL AND email <> '' AND deleted_at IS NULL
GROUP BY email
HAVING COUNT(*) > 1;

-- Roles: name duplicado
SELECT 'DUP_ROLE_NAME' AS check_name, name, COUNT(*) AS cnt
FROM sys_role
GROUP BY name
HAVING COUNT(*) > 1;

-- Operativas: nombre duplicado
SELECT 'DUP_OPERATIVA' AS check_name, nombre, COUNT(*) AS cnt
FROM sys_operativas
WHERE nombre IS NOT NULL AND nombre <> ''
GROUP BY nombre
HAVING COUNT(*) > 1;

-- Permisos: (role_id, menu_id) duplicado
SELECT 'DUP_ROLE_MENU' AS check_name, role_id, menu_id, COUNT(*) AS cnt
FROM sys_role_rights
GROUP BY role_id, menu_id
HAVING COUNT(*) > 1;

-- =========================================================
-- 1) CONFIGURACIÓN GLOBAL SEGURA PARA DDL
-- =========================================================
SET @OLD_UNIQUE_CHECKS=@@UNIQUE_CHECKS, UNIQUE_CHECKS=0;
SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0;
SET @OLD_SQL_NOTES=@@SQL_NOTES, SQL_NOTES=0;

-- =========================================================
-- 2) UNIFICACIÓN DE COLLATION / CHARSET (sys_*)
-- =========================================================
ALTER TABLE sys_users        CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
ALTER TABLE sys_role         CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
ALTER TABLE sys_role_rights  CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
ALTER TABLE sys_menu         CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
ALTER TABLE sys_operativas   CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- =========================================================
-- 3) ÍNDICES Y CLAVES ÚNICAS (evitan duplicados)
--    Nota: si existen índices/constraints con los mismos nombres,
--    puede omitir el DROP. Si falla por duplicados, resuelva y reintente.
-- =========================================================

-- sys_users: UNIQUE(username), UNIQUE(email opcional)
-- Intente eliminar índices previos con nombres conocidos (si existieran)
DROP INDEX uq_sys_users_username ON sys_users;
DROP INDEX uq_sys_users_email ON sys_users;

-- Cree los UNIQUE (si su MySQL es >= 8.0.13 puede usar IF NOT EXISTS)
ALTER TABLE sys_users
  ADD CONSTRAINT uq_sys_users_username UNIQUE (username),
  ADD CONSTRAINT uq_sys_users_email    UNIQUE (email);

-- sys_role: UNIQUE(name)
DROP INDEX uq_sys_role_name ON sys_role;
ALTER TABLE sys_role
  ADD CONSTRAINT uq_sys_role_name UNIQUE (name);

-- sys_operativas: UNIQUE(nombre) (aplica si desea evitar nombres repetidos)
DROP INDEX uq_sys_operativas_nombre ON sys_operativas;
ALTER TABLE sys_operativas
  ADD CONSTRAINT uq_sys_operativas_nombre UNIQUE (nombre);

-- sys_role_rights: UNIQUE(role_id, menu_id)
DROP INDEX uq_sys_role_rights_role_menu ON sys_role_rights;
ALTER TABLE sys_role_rights
  ADD CONSTRAINT uq_sys_role_rights_role_menu UNIQUE (role_id, menu_id);

-- =========================================================
-- 4) ÍNDICES DE APOYO A CONSULTAS
-- =========================================================

-- sys_users: acceso habitual por username y filtros por operativa
CREATE INDEX IF NOT EXISTS idx_sys_users_username     ON sys_users(username);
CREATE INDEX IF NOT EXISTS idx_sys_users_operativa_id ON sys_users(operativa_id);

-- sys_role_rights: búsquedas por role_id y menu_id
CREATE INDEX IF NOT EXISTS idx_sys_role_rights_role   ON sys_role_rights(role_id);
CREATE INDEX IF NOT EXISTS idx_sys_role_rights_menu   ON sys_role_rights(menu_id);

-- sys_menu: árbol y orden
CREATE INDEX IF NOT EXISTS idx_sys_menu_parent        ON sys_menu(parent_id);
CREATE INDEX IF NOT EXISTS idx_sys_menu_active_ord    ON sys_menu(activo, parent_id, orden);

-- =========================================================
-- 5) INTEGRIDAD JERÁRQUICA DE sys_menu (FK parent_id → sys_menu.id)
--    Primero saneamos posibles huérfanos para no fallar al crear la FK.
-- =========================================================

-- Colocar en NULL cualquier parent_id que no exista en sys_menu.id
UPDATE sys_menu m
LEFT JOIN sys_menu p ON p.id = m.parent_id
SET m.parent_id = NULL
WHERE m.parent_id IS NOT NULL AND p.id IS NULL;

-- Si había una FK anterior con otro nombre, elimínela (intento seguro)
ALTER TABLE sys_menu DROP FOREIGN KEY fk_sys_menu_parent;

-- Ahora sí, crear la FK con borrado seguro (SET NULL) y actualización en cascada
ALTER TABLE sys_menu
  ADD CONSTRAINT fk_sys_menu_parent
  FOREIGN KEY (parent_id) REFERENCES sys_menu(id)
  ON DELETE SET NULL
  ON UPDATE CASCADE;

-- =========================================================
-- 6) RESTAURAR PARÁMETROS
-- =========================================================
SET FOREIGN_KEY_CHECKS=@OLD_FOREIGN_KEY_CHECKS;
SET UNIQUE_CHECKS=@OLD_UNIQUE_CHECKS;
SET SQL_NOTES=@OLD_SQL_NOTES;

-- FIN DE MIGRACIÓN
SQL