-- database/migrations/20251007_pl_staging_sync_all.sql
-- Unificación del staging PL:
--  - Fuente operativa (lo que usa el SP): pl_import_batch + pl_import_row (raw JSON)
--  - Compat: triggers que reflejan inserciones/actualizaciones/borrados desde
--    pl_import_batches / pl_import_rows hacia las tablas singulares.

SET NAMES utf8mb4;
SET @OLD_SQL_MODE := @@SQL_MODE;
SET SQL_MODE = 'STRICT_ALL_TABLES,NO_ZERO_IN_DATE,NO_ZERO_DATE,ERROR_FOR_DIVISION_BY_ZERO,NO_ENGINE_SUBSTITUTION';

DELIMITER $$

/* ==========================================================
   0) Helper: siguiente rownum dentro del batch (si fila_nro viene NULL)
   ========================================================== */
DROP FUNCTION IF EXISTS _pl_next_rownum $$
CREATE FUNCTION _pl_next_rownum(p_batch BIGINT UNSIGNED)
RETURNS INT
DETERMINISTIC
BEGIN
  DECLARE v_next INT;
  SELECT COALESCE(MAX(rownum),0) + 1 INTO v_next
    FROM pl_import_row
   WHERE batch_id = p_batch;
  RETURN COALESCE(v_next,1);
END $$


/* ==========================================================
   1) TRIGGERS DE BATCHES: pl_import_batches → pl_import_batch
   ========================================================== */

-- INSERT (AI)
DROP TRIGGER IF EXISTS trg_pl_import_batches_ai $$
CREATE TRIGGER trg_pl_import_batches_ai
AFTER INSERT ON pl_import_batches
FOR EACH ROW
BEGIN
  /* Inserta/actualiza el batch singular con el MISMO id del plural.
     Campos mapeados conservadores que existen en tu esquema. */
  INSERT INTO pl_import_batch (
    id,              -- sincronizamos el id
    tipo,            -- tipificamos
    filename,        -- nombre de archivo
    sheet_name,      -- desconocido aquí
    imported_at,     -- fecha creación del plural
    user_id,         -- si existe en plural, úsalo; si no, va NULL
    rows_total,      -- se recalcula en SP; aquí NULL
    rows_ok,
    rows_error,
    log              -- info útil (path/estado/obs)
  )
  VALUES (
    NEW.id,
    'PACKINGLIST',
    NEW.archivo_nombre,
    NULL,
    COALESCE(NEW.created_at, NOW()),
    NULL,
    NULL,
    NULLIF(NEW.rows_ok, 0),
    NULLIF(NEW.rows_error, 0),
    CONCAT('path=', COALESCE(NEW.archivo_path,''), '; estado=', COALESCE(NEW.estado,''), '; obs=', COALESCE(NEW.observacion,''))
  )
  ON DUPLICATE KEY UPDATE
    tipo        = VALUES(tipo),
    filename    = VALUES(filename),
    sheet_name  = VALUES(sheet_name),
    imported_at = VALUES(imported_at),
    user_id     = VALUES(user_id),
    rows_ok     = VALUES(rows_ok),
    rows_error  = VALUES(rows_error),
    log         = VALUES(log);
END $$

-- UPDATE (AU)
DROP TRIGGER IF EXISTS trg_pl_import_batches_au $$
CREATE TRIGGER trg_pl_import_batches_au
AFTER UPDATE ON pl_import_batches
FOR EACH ROW
BEGIN
  UPDATE pl_import_batch
     SET filename    = COALESCE(NEW.archivo_nombre, filename),
         imported_at = COALESCE(NEW.created_at, imported_at),
         rows_ok     = NULLIF(NEW.rows_ok, 0),
         rows_error  = NULLIF(NEW.rows_error, 0),
         log         = CONCAT('path=', COALESCE(NEW.archivo_path,''), '; estado=', COALESCE(NEW.estado,''), '; obs=', COALESCE(NEW.observacion,''))
   WHERE id = NEW.id;
END $$

-- DELETE (AD)
DROP TRIGGER IF EXISTS trg_pl_import_batches_ad $$
CREATE TRIGGER trg_pl_import_batches_ad
AFTER DELETE ON pl_import_batches
FOR EACH ROW
BEGIN
  -- Limpieza en singular (batch y sus filas)
  DELETE FROM pl_import_row   WHERE batch_id = OLD.id;
  DELETE FROM pl_import_batch WHERE id       = OLD.id;
END $$


/* ==========================================================
   2) TRIGGERS DE ROWS: pl_import_rows → pl_import_row (raw JSON)
   ========================================================== */

-- INSERT (AI)
DROP TRIGGER IF EXISTS trg_pl_import_rows_ai $$
CREATE TRIGGER trg_pl_import_rows_ai
AFTER INSERT ON pl_import_rows
FOR EACH ROW
BEGIN
  /* Calcular expected_uc = (uv_cajas * uc_por_caja) + uc_sueltas */
  SET @exp_uc := COALESCE(NEW.uv_cajas, 0) * COALESCE(NEW.uc_por_caja, 0) + COALESCE(NEW.uc_sueltas, 0);
  SET @rownum := COALESCE(NEW.fila_nro, _pl_next_rownum(NEW.batch_id));

  /* Construir JSON crudo como espera el SP */
  SET @raw := JSON_OBJECT(
    'sku_cliente',        NEW.sku,
    'descripcion',        NEW.denominacion,
    'lote',               NEW.lote,
    'fecha_produccion',   DATE_FORMAT(NEW.fecha_produccion,   '%Y-%m-%d'),
    'fecha_vencimiento',  DATE_FORMAT(NEW.fecha_vencimiento,  '%Y-%m-%d'),
    'expected_uv',        COALESCE(NEW.uv_cajas, 0),
    'expected_uc',        COALESCE(@exp_uc, 0)
  );

  /* Insertar en staging canónico:
     - status NULL (el SP setea OK/ERROR)
     - id autoincrement independiente del plural
  */
  INSERT INTO pl_import_row (batch_id, rownum, raw, status)
  VALUES (NEW.batch_id, @rownum, @raw, NULL);
END $$

-- UPDATE (AU)
DROP TRIGGER IF EXISTS trg_pl_import_rows_au $$
CREATE TRIGGER trg_pl_import_rows_au
AFTER UPDATE ON pl_import_rows
FOR EACH ROW
BEGIN
  /* Re-espejar cambios relevantes (idempotente; afecta la fila con mismo rownum si existe) */
  SET @exp_uc := COALESCE(NEW.uv_cajas, 0) * COALESCE(NEW.uc_por_caja, 0) + COALESCE(NEW.uc_sueltas, 0);
  SET @rownum := COALESCE(NEW.fila_nro, _pl_next_rownum(NEW.batch_id));
  SET @raw := JSON_OBJECT(
    'sku_cliente',        NEW.sku,
    'descripcion',        NEW.denominacion,
    'lote',               NEW.lote,
    'fecha_produccion',   DATE_FORMAT(NEW.fecha_produccion,   '%Y-%m-%d'),
    'fecha_vencimiento',  DATE_FORMAT(NEW.fecha_vencimiento,  '%Y-%m-%d'),
    'expected_uv',        COALESCE(NEW.uv_cajas, 0),
    'expected_uc',        COALESCE(@exp_uc, 0)
  );

  /* Si existe la fila con ese rownum: actualizar; si no, insertar */
  IF EXISTS (
    SELECT 1 FROM pl_import_row
     WHERE batch_id = NEW.batch_id AND rownum = @rownum
  ) THEN
    UPDATE pl_import_row
       SET raw = @raw
     WHERE batch_id = NEW.batch_id AND rownum = @rownum;
  ELSE
    INSERT INTO pl_import_row (batch_id, rownum, raw, status)
    VALUES (NEW.batch_id, @rownum, @raw, NULL);
  END IF;
END $$

-- DELETE (AD)  **(versión robusta con DELIMITER, evita 1093)**
DROP TRIGGER IF EXISTS trg_pl_import_rows_ad $$
CREATE TRIGGER trg_pl_import_rows_ad
AFTER DELETE ON pl_import_rows
FOR EACH ROW
BEGIN
  DECLARE v_rownum INT;
  DECLARE v_del_id BIGINT UNSIGNED;

  SET v_rownum := OLD.fila_nro;

  IF v_rownum IS NOT NULL THEN
    -- Borrado directo por rownum dentro del mismo batch
    DELETE FROM pl_import_row
     WHERE batch_id = OLD.batch_id
       AND rownum   = v_rownum
     LIMIT 1;
  ELSE
    -- Fallback: localizar una fila candidata por sku (y lote si existe) y eliminar por id (evita error 1093)
    SELECT pir.id
      INTO v_del_id
      FROM pl_import_row pir
     WHERE pir.batch_id = OLD.batch_id
       AND JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.sku_cliente')) = OLD.sku
       AND (JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.lote')) <=> OLD.lote)
     ORDER BY pir.id
     LIMIT 1;

    IF v_del_id IS NOT NULL THEN
      DELETE FROM pl_import_row WHERE id = v_del_id;
    END IF;
  END IF;
END $$

DELIMITER ;

-- (Opcional pero recomendado) Asegurá que status permita NULL en el staging singular:
-- ALTER TABLE pl_import_row MODIFY COLUMN status ENUM('OK','ERROR') NULL;

-- Restaurar SQL_MODE
SET SQL_MODE := @OLD_SQL_MODE;
