/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 10: SPs de Salida (picking → OUT → retorno/devoluciones)
 * Archivo: database/step10_so_outbound_sps.sql
 *
 * Compatibilidad phpMyAdmin:
 * - Usar delimitador UI: $$
 * - Evitar DEFAULT numéricos en JSON_TABLE; castear con CAST()+COALESCE()
 * - Tablas TEMPORARY → ENGINE=InnoDB (acepta LONGTEXT)
 * ===================================================================== */

SET NAMES utf8mb4 $$

/* ==============================================================
   0) Triggers: acumulan prepared_* y shipped_*
   ============================================================== */
DROP TRIGGER IF EXISTS trg_so_pre_pick_ai $$
CREATE TRIGGER trg_so_pre_pick_ai
AFTER INSERT ON so_pre_pick
FOR EACH ROW
BEGIN
  UPDATE so_pedido_dest_item
     SET prepared_uv = LEAST(expected_uv, prepared_uv + COALESCE(NEW.uv_cajas,0)),
         prepared_uc = LEAST(expected_uc, prepared_uc + COALESCE(NEW.uc_unidades,0))
   WHERE id = NEW.pedido_dest_item_id;
END $$

DROP TRIGGER IF EXISTS trg_so_ship_link_ai $$
CREATE TRIGGER trg_so_ship_link_ai
AFTER INSERT ON so_ship_link
FOR EACH ROW
BEGIN
  UPDATE so_pedido_dest_item
     SET shipped_uv = LEAST(expected_uv, shipped_uv + COALESCE(NEW.uv_cajas,0)),
         shipped_uc = LEAST(expected_uc, shipped_uc + COALESCE(NEW.uc_unidades,0))
   WHERE id = NEW.pedido_dest_item_id;
END $$


/* ==============================================================
   1) SP: Preparación automática (FEFO) hacia zona PREP
   ============================================================== */
DROP PROCEDURE IF EXISTS sp_so_preparar_auto $$
CREATE PROCEDURE sp_so_preparar_auto(
  IN p_pedido_codigo   VARCHAR(64),
  IN p_deposito_code   VARCHAR(32),
  IN p_pos_prep_code   VARCHAR(64)  -- NULL = seleccionar PREP menos ocupada
)
BEGIN
  /* ---------- DECLARE ---------- */
  DECLARE v_pedido_id    BIGINT UNSIGNED;
  DECLARE v_dep_id       INT UNSIGNED;
  DECLARE v_pre_id       BIGINT UNSIGNED;
  DECLARE v_pre_code     VARCHAR(64);
  DECLARE v_pre_est_id   INT UNSIGNED;
  DECLARE v_pos_prep_id  BIGINT UNSIGNED;
  DECLARE v_done         INT DEFAULT 0;

  -- Iteración por ítems pendientes
  DECLARE v_item_id      BIGINT UNSIGNED;
  DECLARE v_prod_id      BIGINT UNSIGNED;
  DECLARE v_lote_code    VARCHAR(64);
  DECLARE v_dest_id      BIGINT UNSIGNED;
  DECLARE v_need_uv      INT;
  DECLARE v_need_uc      INT;

  -- Candidato de stock
  DECLARE v_pos_from     BIGINT UNSIGNED;
  DECLARE v_lote_id      BIGINT UNSIGNED;
  DECLARE v_pallet_id    BIGINT UNSIGNED;
  DECLARE v_take_uv      INT;
  DECLARE v_take_uc      INT;
  DECLARE v_move_id      BIGINT UNSIGNED;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  /* ---------- Validaciones ---------- */
  IF p_pedido_codecOde IS NULL OR p_pedido_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_pedido_codigo es requerido.';
  END IF;

  SELECT id INTO v_pedido_id FROM so_pedido WHERE codigo=p_pedido_codigo LIMIT 1;
  IF v_pedido_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Pedido no encontrado.';
  END IF;

  SELECT id INTO v_dep_id FROM wh_deposito WHERE code=p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Depósito no encontrado.';
  END IF;

  /* ---------- Determinar posición PREP ---------- */
  IF p_pos_prep_code IS NOT NULL AND p_pos_prep_code <> '' THEN
    SELECT id INTO v_pos_prep_id
      FROM wh_posicion
     WHERE deposito_id=v_dep_id
       AND (code=p_pos_prep_code OR code_full=p_pos_prep_code OR pos_code=p_pos_prep_code OR pos_code_full=p_pos_prep_code)
     LIMIT 1;
    IF v_pos_prep_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Posición PREP indicada no existe en el depósito.';
    END IF;
  ELSE
    SELECT p.id INTO v_pos_prep_id
      FROM wh_posicion p
      JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP'
      LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=v_dep_id
     WHERE p.deposito_id=v_dep_id AND p.activo=1
     GROUP BY p.id
     ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
     LIMIT 1;
    IF v_pos_prep_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='No hay posiciones PREP en el depósito.';
    END IF;
  END IF;

  /* ---------- Crear/obtener pre-embarque ---------- */
  SELECT id INTO v_pre_est_id FROM so_preembarque_estado WHERE code='PENDIENTE' LIMIT 1;

  SET v_pre_code = CONCAT('PRE-', p_pedido_codigo);
  SELECT id INTO v_pre_id FROM so_preembarQUE WHERE codigo=v_pre_code LIMIT 1;

  IF v_pre_id IS NULL THEN
    INSERT INTO so_preembarque (codigo, pedido_id, deposito_id, estado_id, zona_posicion_id, asignado_at, inicio_at)
    VALUES (v_pre_code, v_pedido_id, v_dep_id, v_pre_est_id, v_pos_prep_id, NOW(), NOW());
    SET v_pre_id = LAST_INSERT_ID();
  ELSE
    UPDATE so_preembarque SET zona_posicion_id=v_pos_prep_id WHERE id=v_pre_id AND zona_posicion_id IS NULL;
  END IF;

  -- Marcar pedido en EN_PREPARACION si procede
  UPDATE so_pedido p
     JOIN so_pedido_estado pe ON pe.id=p.estado_id
     SET p.estado_id = (SELECT id FROM so_pedido_estado WHERE code='EN_PREPARACION' LIMIT 1)
   WHERE p.id=v_pedido_id AND pe.code IN ('RECIBIDO','PARCIAL');

  START TRANSACTION;

  /* ---------- Ítems pendientes a preparar ---------- */
  DROP TEMPORARY TABLE IF EXISTS _pend;
  CREATE TEMPORARY TABLE _pend (
    pedido_dest_item_id BIGINT UNSIGNED,
    pedido_dest_id      BIGINT UNSIGNED,
    destinatario_id     BIGINT UNSIGNED,
    producto_id         BIGINT UNSIGNED,
    lote_codigo         VARCHAR(64),
    need_uv             INT,
    need_uc             INT
  ) ENGINE=InnoDB;

  INSERT INTO _pend
  SELECT i.id, d.id, d.destinatario_id, i.producto_id, i.lote_codigo,
         GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv,
         GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc
  FROM so_pedido_dest_item i
  JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
  WHERE d.pedido_id = v_pedido_id
    AND (GREATEST(i.expected_uv - i.prepared_uv,0) > 0
      OR GREATEST(i.expected_uc - i.prepared_uc,0) > 0);

  /* Cursor por ítem pendiente */
  BEGIN
    DECLARE cur_it CURSOR FOR
      SELECT pedido_dest_item_id, producto_id, lote_codigo, destinatario_id, need_uv, need_uc
      FROM _pend;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done=1;

    SET v_done=0;
    OPEN cur_it;
    it_loop: LOOP
      FETCH cur_it INTO v_item_id, v_prod_id, v_lote_code, v_dest_id, v_need_uv, v_need_uc;
      IF v_done=1 THEN LEAVE it_loop; END IF;

      /* Candidatos de stock (no PREP/CUARENTENA), FEFO */
      DROP TEMPORARY TABLE IF EXISTS _cand;
      CREATE TEMPORARY TABLE _cand (
        posicion_id BIGINT UNSIGNED,
        lote_id     BIGINT UNSIGNED,
        pallet_id   BIGINT UNSIGNED,
        qty_uv      INT,
        qty_uc      INT,
        venc        DATE
      ) ENGINE=InnoDB;

      INSERT INTO _cand (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
      SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
      FROM wh_stock s
      JOIN wh_posicion p   ON p.id=s.posicion_id
      JOIN wh_ambiente a   ON a.id=p.ambiente_id
      LEFT JOIN wh_lote l  ON l.id=s.lote_id
      WHERE s.deposito_id = v_dep_id
        AND s.producto_id = v_prod_id
        AND (v_lote_code IS NULL OR v_lote_code='' OR EXISTS(
              SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=v_lote_code))
        AND a.code NOT IN ('PREP','CUARENTENA')   -- solo desde posiciones operativas
        AND (s.qty_uv > 0 OR s.qty_uc > 0)
      ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;

      /* Iterar candidatos hasta cubrir necesidad */
      WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
        SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc
          INTO v_pos_from, v_lote_id, v_pallet_id, v_take_uv, v_take_uc
        FROM _cand ORDER BY COALESCE(venc,'9999-12-31') ASC LIMIT 1;

        IF v_pos_from IS NULL THEN
          LEAVE it_loop; -- no hay más stock; deja pendientes
        END IF;

        SET v_take_uv = LEAST(COALESCE(v_take_uv,0), v_need_uv);
        SET v_take_uc = LEAST(COALESCE(v_take_uc,0), v_need_uc);

        IF v_take_uv=0 AND v_take_uc=0 THEN
          DELETE FROM _cand LIMIT 1;
          ITERATE it_loop;
        END IF;

        /* Registrar move MOVE (from → PREP) */
        INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
        VALUES (v_dep_id, 'MOVE', 'PREPARACION', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_prep_id, -v_take_uv, -v_take_uc, CONCAT('PRE-', v_pre_id));
        SET v_move_id = LAST_INSERT_ID();

        /* Registrar pick */
        INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por)
        VALUES (v_pre_id, v_item_id, v_pos_from, v_pos_prep_id, v_pallet_id, v_lote_id, v_take_uv, v_take_uc, NULL);

        /* Actualizar necesidad */
        SET v_need_uv = v_need_uv - v_take_uv;
        SET v_need_uc = v_need_uc - v_take_uc;

        /* Descontar candidato ya usado */
        UPDATE _cand SET qty_uv = qty_uv - v_take_uv, qty_uc = qty_uc - v_take_uc
        WHERE posicion_id=v_pos_from AND lote_id=v_lote_id AND COALESCE(pallet_id,0)=COALESCE(v_pallet_id,0);

        DELETE FROM _cand
         WHERE qty_uv <= 0 AND qty_uc <= 0
         LIMIT 1;
      END WHILE;
      DROP TEMPORARY TABLE IF EXISTS _cand;
    END LOOP;
    CLOSE cur_it;
  END;

  COMMIT;

  /* Resultado: items preparados vs pendientes */
  SELECT d.pedido_id, i.id AS pedido_dest_item_id, i.producto_id, i.lote_codigo,
         i.expected_uv, i.prepared_uv, (i.expected_uv - i.prepared_uv) AS pend_uv,
         i.expected_uc, i.prepared_uc, (i.expected_uc - i.prepared_uc) AS pend_uc
  FROM so_pedido_dest_item i
  JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
  WHERE d.pedido_id=v_pedido_id;
END $$


/* ==============================================================
   2) SP: Despachar (OUT) un embarque desde PREP
   ============================================================== */
DROP PROCEDURE IF EXISTS sp_so_despachar_embarque $$
CREATE PROCEDURE sp_so_despachar_embarque(
  IN p_embarque_codigo VARCHAR(64)
)
BEGIN
  /* ---------- DECLARE ---------- */
  DECLARE v_emb_id     BIGINT UNSIGNED;
  DECLARE v_dep_id     INT UNSIGNED;
  DECLARE v_done       INT DEFAULT 0;
  DECLARE v_move_id    BIGINT UNSIGNED;

  -- Ítem a despachar
  DECLARE v_item_id    BIGINT UNSIGNED;
  DECLARE v_pd_id      BIGINT UNSIGNED;
  DECLARE v_dest_id    BIGINT UNSIGNED;
  DECLARE v_prod_id    BIGINT UNSIGNED;
  DECLARE v_lote_code  VARCHAR(64);
  DECLARE v_need_uv    INT;
  DECLARE v_need_uc    INT;

  -- Stock en PREP
  DECLARE v_pos_id     BIGINT UNSIGNED;
  DECLARE v_lote_id    BIGINT UNSIGNED;
  DECLARE v_pallet_id  BIGINT UNSIGNED;
  DECLARE v_take_uv    INT;
  DECLARE v_take_uc    INT;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK; RESIGNAL;
  END;

  SELECT id, deposito_id INTO v_emb_id, v_dep_id FROM so_embarque WHERE codigo=p_embarque_codigo LIMIT 1;
  IF v_emb_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Embarque no encontrado.';
  END IF;

  START TRANSACTION;

  /* Pasar a EN_CARGA si corresponde e inicializar inicio */
  UPDATE so_embarque e
     JOIN so_embarque_estado ee ON ee.id=e.estado_id
     SET e.estado_id = (SELECT id FROM so_embarque_estado WHERE code='EN_CARGA' LIMIT 1),
         e.carga_inicio_at = COALESCE(e.carga_inicio_at, NOW())
   WHERE e.id=v_emb_id AND ee.code IN ('EN_COLA');

  /* Ítems pendientes por embarque (pedidos vinculados) */
  DROP TEMPORARY TABLE IF EXISTS _ship_pend;
  CREATE TEMPORARY TABLE _ship_pend (
    item_id BIGINT UNSIGNED,
    pedido_dest_id BIGINT UNSIGNED,
    destinatario_id BIGINT UNSIGNED,
    producto_id BIGINT UNSIGNED,
    lote_codigo VARCHAR(64),
    need_uv INT, need_uc INT
  ) ENGINE=InnoDB;

  INSERT INTO _ship_pend
  SELECT i.id, d.id, d.destinatario_id, i.producto_id, i.lote_codigo,
         GREATEST(LEAST(i.expected_uv, i.prepared_uv) - i.shipped_uv, 0) AS need_uv,
         GREATEST(LEAST(i.expected_uc, i.prepared_uc) - i.shipped_uc, 0) AS need_uc
  FROM so_embarque_pre ep
  JOIN so_preembarque pre ON pre.id=ep.preembarque_id
  JOIN so_pedido_dest d   ON d.pedido_id=pre.pedido_id
  JOIN so_pedido_dest_item i ON i.pedido_dest_id=d.id
  WHERE ep.embarque_id=v_emb_id
    AND (GREATEST(LEAST(i.expected_uv, i.prepared_uv) - i.shipped_uv, 0) > 0
      OR GREATEST(LEAST(i.expected_uc, i.prepared_uc) - i.shipped_uc, 0) > 0);

  /* Cursor por ítem */
  BEGIN
    DECLARE cur_it CURSOR FOR
      SELECT item_id, pedido_dest_id, destinatario_id, producto_id, lote_codigo, need_uv, need_uc
      FROM _ship_pend;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done=1;

    SET v_done=0; OPEN cur_it;
    ship_loop: LOOP
      FETCH cur_it INTO v_item_id, v_pd_id, v_dest_id, v_prod_id, v_lote_code, v_need_uv, v_need_uc;
      IF v_done=1 THEN LEAVE ship_loop; END IF;

      /* Stock en PREP (todas las posiciones PREP del depósito) */
      DROP TEMPORARY TABLE IF EXISTS _prep_stock;
      CREATE TEMPORARY TABLE _prep_stock (
        pos_id   BIGINT UNSIGNED,
        lote_id  BIGINT UNSIGNED,
        pallet_id BIGINT UNSIGNED,
        qty_uv   INT, qty_uc INT
      ) ENGINE=InnoDB;

      INSERT INTO _prep_stock (pos_id,lote_id,pallet_id,qty_uv,qty_uc)
      SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc
      FROM wh_stock s
      JOIN wh_posicion p ON p.id=s.posicion_id
      JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP'
      WHERE s.deposito_id=v_dep_id
        AND s.producto_id=v_prod_id
        AND (v_lote_code IS NULL OR v_lote_code='' OR EXISTS(
              SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=v_lote_code))
        AND (s.qty_uv>0 OR s.qty_uc>0);

      WHILE (v_need_uv>0 OR v_need_uc>0) DO
        SELECT pos_id, lote_id, pallet_id, qty_uv, qty_uc
          INTO v_pos_id, v_lote_id, v_pallet_id, v_take_uv, v_take_uc
        FROM _prep_stock LIMIT 1;

        IF v_pos_id IS NULL THEN
          LEAVE ship_loop; -- no hay más en PREP; queda pendiente
        END IF;

        SET v_take_uv = LEAST(COALESCE(v_take_uv,0), v_need_uv);
        SET v_take_uc = LEAST(COALESCE(v_take_uc,0), v_need_uc);

        IF v_take_uv=0 AND v_take_uc=0 THEN
          DELETE FROM _prep_stock LIMIT 1;
          ITERATE ship_loop;
        END IF;

        /* OUT desde PREP (to_pos_id NULL) */
        INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
        VALUES (v_dep_id, 'OUT', 'DESPACHO', v_pallet_id, v_prod_id, v_lote_id, v_pos_id, NULL, -v_take_uv, -v_take_uc, CONCAT('EMB-', v_emb_id));
        SET v_move_id = LAST_INSERT_ID();

        /* Link a pedido y ruteo (parada por destinatario) */
        INSERT INTO so_ship_link (pedido_dest_item_id, producto_id, lote_id, pallet_id, move_id, embarque_id,
                                  parada_id, uv_cajas, uc_unidades)
        VALUES (
          v_item_id, v_prod_id, v_lote_id, v_pallet_id, v_move_id, v_emb_id,
          (SELECT id FROM so_embarque_parada WHERE embarque_id=v_emb_id AND destinatario_id=v_dest_id ORDER BY orden LIMIT 1),
          v_take_uv, v_take_uc
        );

        -- Actualizar necesidades locales
        SET v_need_uv = v_need_uv - v_take_uv;
        SET v_need_uc = v_need_uc - v_take_uc;

        UPDATE _prep_stock SET qty_uv=qty_uv - v_take_uv, qty_uc=qty_uc - v_take_uc LIMIT 1;
        DELETE FROM _prep_stock WHERE qty_uv<=0 AND qty_uc<=0 LIMIT 1;
      END WHILE;

      DROP TEMPORARY TABLE IF EXISTS _prep_stock;
    END LOOP;
    CLOSE cur_it;
  END;

  /* ¿Quedó todo despachado? → marcar embarque CARGADO */
  IF NOT EXISTS (
    SELECT 1
    FROM so_embarque_pre ep
    JOIN so_preembarque pre ON pre.id=ep.preembarque_id
    JOIN so_pedido_dest d   ON d.pedido_id=pre.pedido_id
    JOIN so_pedido_dest_item i ON i.pedido_dest_id=d.id
    WHERE ep.embarque_id=v_emb_id
      AND (GREATEST(LEAST(i.expected_uv, i.prepared_uv) - i.shipped_uv, 0) > 0
        OR GREATEST(LEAST(i.expected_uc, i.prepared_uc) - i.shipped_uc, 0) > 0)
  ) THEN
    UPDATE so_embarque
       SET estado_id = (SELECT id FROM so_embarque_estado WHERE code='CARGADO' LIMIT 1),
           carga_fin_at = COALESCE(carga_fin_at, NOW())
     WHERE id=v_emb_id;
  END IF;

  COMMIT;

  /* Salida: resumen por pedido-dest */
  SELECT * FROM v_so_pedido_resumen
  WHERE pedido_id IN (
    SELECT pre.pedido_id
    FROM so_embarque_pre ep JOIN so_preembarque pre ON pre.id=ep.preembarque_id
    WHERE ep.embarque_id=v_emb_id
  );
END $$


/* ==============================================================
   3) SP: Retorno + Devoluciones (IN a CUARENTENA)
   ============================================================== */
DROP PROCEDURE IF EXISTS sp_so_retorno_y_devoluciones $$

CREATE PROCEDURE sp_so_retorno_y_devoluciones(
  IN p_embarque_codigo VARCHAR(64),
  IN p_llegada_at      DATETIME,
  IN p_devol_json      JSON
)
BEGIN
  /* ---------- DECLARE ---------- */
  DECLARE v_emb_id   BIGINT UNSIGNED;
  DECLARE v_dep_id   INT UNSIGNED;
  DECLARE v_ret_id   BIGINT UNSIGNED;
  DECLARE v_cuar_pos BIGINT UNSIGNED;
  DECLARE v_pal_code VARCHAR(64);
  DECLARE v_pal_id   BIGINT UNSIGNED;
  DECLARE v_move_id  BIGINT UNSIGNED;
  DECLARE v_seq      INT DEFAULT 0;
  DECLARE v_est_cuar INT UNSIGNED;

  -- fila json
  DECLARE v_dest_id  BIGINT UNSIGNED;
  DECLARE v_doc_t    VARCHAR(20);
  DECLARE v_doc_n    VARCHAR(64);
  DECLARE v_mot_code VARCHAR(32);
  DECLARE v_mot_id   INT UNSIGNED;
  DECLARE v_prod_id  BIGINT UNSIGNED;
  DECLARE v_lote_c   VARCHAR(64);
  DECLARE v_lote_id  BIGINT UNSIGNED;
  DECLARE v_uv       INT;
  DECLARE v_uc       INT;

  DECLARE v_done INT DEFAULT 0;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  /* ---------- Validaciones ---------- */
  IF p_embarque_codigo IS NULL OR p_embarque_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_embarque_codigo es requerido.';
  END IF;

  SELECT id, deposito_id INTO v_emb_id, v_dep_id
  FROM so_embarque WHERE codigo = p_embarque_codigo LIMIT 1;

  IF v_emb_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Embarque no encontrado.';
  END IF;

  /* ---------- Retorno ---------- */
  START TRANSACTION;

  INSERT INTO so_retorno (embarque_id, llegada_at, observacion)
  VALUES (v_emb_id, COALESCE(p_llegada_at, NOW()), 'Retorno registrado por SP');
  SET v_ret_id = LAST_INSERT_ID();

  /* Posición CUARENTENA y estado */
  SELECT p.id INTO v_cuar_pos
  FROM wh_posicion p
  JOIN wh_ambiente a ON a.id = p.ambiente_id AND a.code = 'CUARENTENA'
  WHERE p.deposito_id = v_dep_id
  ORDER BY p.id
  LIMIT 1;

  IF v_cuar_pos IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'No hay posición de CUARENTENA en el depósito.';
  END IF;

  SELECT id INTO v_est_cuar FROM wh_pallet_estado WHERE code = 'CUARENTENA' LIMIT 1;

  /* ---------- Devoluciones ---------- */
  DROP TEMPORARY TABLE IF EXISTS _dev;
  CREATE TEMPORARY TABLE _dev (
    destinatario_id BIGINT UNSIGNED,
    doc_tipo        VARCHAR(20),
    doc_numero      VARCHAR(64),
    motivo_code     VARCHAR(32),
    producto_id     BIGINT UNSIGNED,
    lote_codigo     VARCHAR(64),
    uv              INT,
    uc              INT
  ) ENGINE=InnoDB;

  IF p_devol_json IS NOT NULL AND JSON_LENGTH(p_devol_json) > 0 THEN
    INSERT INTO _dev (destinatario_id,doc_tipo,doc_numero,motivo_code,producto_id,lote_codigo,uv,uc)
    SELECT
      jt.destinatario_id,
      jt.doc_tipo,
      jt.doc_numero,
      jt.motivo_code,
      jt.producto_id,
      jt.lote_codigo,
      COALESCE(CAST(jt.uv AS UNSIGNED),0),
      COALESCE(CAST(jt.uc AS UNSIGNED),0)
    FROM JSON_TABLE(
           p_devol_json, '$[*]'
           COLUMNS(
             destinatario_id BIGINT      PATH '$.destinatario_id' NULL ON EMPTY,
             doc_tipo       VARCHAR(20)  PATH '$.doc_tipo'        NULL ON EMPTY,
             doc_numero     VARCHAR(64)  PATH '$.doc_numero'      NULL ON EMPTY,
             motivo_code    VARCHAR(32)  PATH '$.motivo_code',
             producto_id    BIGINT       PATH '$.producto_id',
             lote_codigo    VARCHAR(64)  PATH '$.lote_codigo'     NULL ON EMPTY,
             uv             VARCHAR(32)  PATH '$.uv'              NULL ON EMPTY,
             uc             VARCHAR(32)  PATH '$.uc'              NULL ON EMPTY
           )
         ) AS jt;
  END IF;

  /* Loop de _dev → pallets RET-... y movimientos IN a CUARENTENA */
  BEGIN
    DECLARE cur CURSOR FOR
      SELECT destinatario_id, doc_tipo, doc_numero, motivo_code, producto_id, lote_codigo, uv, uc
      FROM _dev;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done = 1;

    SET v_done = 0;
    OPEN cur;
    dev_loop: LOOP
      FETCH cur INTO v_dest_id, v_doc_t, v_doc_n, v_mot_code, v_prod_id, v_lote_c, v_uv, v_uc;
      IF v_done = 1 THEN LEAVE dev_loop; END IF;

      /* Motivo parametrizado (fallback a OTRO) */
      SELECT id INTO v_mot_id FROM para_devolucion_motivo WHERE code = v_mot_code LIMIT 1;
      IF v_mot_id IS NULL THEN
        SELECT id INTO v_mot_id FROM para_devolucion_motivo WHERE code = 'OTRO' LIMIT 1;
        IF v_mot_id IS NULL THEN
          INSERT INTO para_devolucion_motivo (code, nombre, activo) VALUES ('OTRO','Otro',1)
          ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), activo=1;
          SELECT id INTO v_mot_id FROM para_devolucion_motivo WHERE code='OTRO' LIMIT 1;
        END IF;
      END IF;

      /* Lote (crear si no existe) */
      IF v_lote_c IS NOT NULL AND v_lote_c <> '' THEN
        SELECT id INTO v_lote_id FROM wh_lote WHERE producto_id = v_prod_id AND codigo = v_lote_c LIMIT 1;
        IF v_lote_id IS NULL THEN
          INSERT INTO wh_lote (producto_id, codigo) VALUES (v_prod_id, v_lote_c);
          SET v_lote_id = LAST_INSERT_ID();
        END IF;
      ELSE
        SET v_lote_id = NULL;
      END IF;

      /* Pallet RET-... en CUARENTENA */
      SET v_seq = v_seq + 1;
      SET v_pal_code = CONCAT('RET-', p_embarque_codigo, '-', LPAD(v_seq,3,'0'));

      INSERT INTO wh_pallet (codigo, deposito_id, posicion_id, estado_id, pickeado, reservado, observacion)
      VALUES (v_pal_code, v_dep_id, v_cuar_pos, v_est_cuar, 1, 0, 'Devolución de ruta');
      SET v_pal_id = LAST_INSERT_ID();

      INSERT INTO wh_pallet_item (pallet_id, producto_id, lote_id, uv_cajas, uc_unidades)
      VALUES (v_pal_id, v_prod_id, v_lote_id, COALESCE(v_uv,0), COALESCE(v_uc,0))
      ON DUPLICATE KEY UPDATE
        uv_cajas    = wh_pallet_item.uv_cajas + VALUES(uv_cajas),
        uc_unidades = wh_pallet_item.uc_unidades + VALUES(uc_unidades);

      /* Movimiento IN (traza) */
      INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
      VALUES (v_dep_id, 'IN', 'DEVOLUCION', v_pal_id, v_prod_id, v_lote_id, NULL, v_cuar_pos, 0, 0, CONCAT('RET-', p_embarque_codigo));
      SET v_move_id = LAST_INSERT_ID();

      /* Registrar devolución */
      INSERT INTO so_devolucion (retorno_id, destinatario_id, doc_tipo, doc_numero, motivo_id, producto_id, lote_id,
                                 qty_pallets, qty_uv, qty_uc, responsable_user_id)
      VALUES (v_ret_id, v_dest_id, v_doc_t, v_doc_n, v_mot_id, v_prod_id, v_lote_id,
              1, COALESCE(v_uv,0), COALESCE(v_uc,0), NULL);
    END LOOP;
    CLOSE cur;
  END;

  /* Finalizar embarque (si aplica) */
  UPDATE so_embarque
     SET estado_id = (SELECT id FROM so_embarque_estado WHERE code = 'FINALIZADO' LIMIT 1)
   WHERE id = v_emb_id;

  COMMIT;

  /* Salida */
  SELECT * FROM v_so_devoluciones WHERE embarque = p_embarque_codigo;
END $$
