/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 2: Esquema Warehouse (tablas wh_*)
 * Archivo: database/step2_wh.sql
 *
 * Cambios vs versión previa:
 * - Fix en wh_stock: NO usar funciones en índices. Se agregan columnas
 *   generadas persistidas (posicion_id_norm, pallet_id_norm) y la UNIQUE
 *   se define sobre ellas.
 * ===================================================================== */

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- ======================================================================
-- 1) Catálogos base
-- ======================================================================

DROP TABLE IF EXISTS wh_ambiente;
CREATE TABLE wh_ambiente (
  id           INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code         VARCHAR(32)  NOT NULL,     -- p.ej.: CUARENTENA, SECO, FRIO, PICKING
  nombre       VARCHAR(100) NOT NULL,
  descripcion  VARCHAR(255) NULL,
  activo       TINYINT(1)   NOT NULL DEFAULT 1,
  created_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at   DATETIME NULL,
  UNIQUE KEY uk_ambiente_code (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

INSERT INTO wh_ambiente (code, nombre, descripcion) VALUES
('CUARENTENA','Cuarentena','Área de inspección previa a ubicación'),
('SECO','Seco','Temperatura ambiente'),
('FRIO','Frío','Cámara o zona refrigerada'),
('PICKING','Picking','Zona de fraccionamiento'),
('PISO','Piso','Almacenamiento a piso')
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), descripcion=VALUES(descripcion);

DROP TABLE IF EXISTS wh_pallet_estado;
CREATE TABLE wh_pallet_estado (
  id           INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code         VARCHAR(32)  NOT NULL, -- RECIBIDO, CUARENTENA, POS_ENTERO, POS_PICKEADO, RESERVADO, REMANEJO, AVERIADO, EMBARCADO
  nombre       VARCHAR(100) NOT NULL,
  orden        TINYINT UNSIGNED NOT NULL DEFAULT 1,
  activo       TINYINT(1)   NOT NULL DEFAULT 1,
  created_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at   DATETIME NULL,
  UNIQUE KEY uk_pallet_estado_code (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

INSERT INTO wh_pallet_estado (code, nombre, orden) VALUES
('RECIBIDO','Recibido',1),
('CUARENTENA','En cuarentena',2),
('POS_ENTERO','En posición - entero',3),
('POS_PICKEADO','En posición - pickeado',4),
('RESERVADO','Reservado',5),
('REMANEJO','Remanejo',6),
('AVERIADO','Averiado',7),
('EMBARCADO','Embarcado/Despachado',8)
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden);

-- ======================================================================
-- 2) Layout físico: Depósitos y Posiciones
-- ======================================================================

DROP TABLE IF EXISTS wh_deposito;
CREATE TABLE wh_deposito (
  id           INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code         VARCHAR(32)  NOT NULL,     -- DEP1, DEP2...
  nombre       VARCHAR(100) NOT NULL,
  descripcion  VARCHAR(255) NULL,
  direccion    VARCHAR(255) NULL,
  activo       TINYINT(1)   NOT NULL DEFAULT 1,
  created_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at   DATETIME NULL,
  UNIQUE KEY uk_deposito_code (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS wh_posicion;
CREATE TABLE wh_posicion (
  id             BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  deposito_id    INT UNSIGNED NOT NULL,
  rack           INT UNSIGNED NOT NULL,
  columna        INT UNSIGNED NOT NULL,
  nivel          INT UNSIGNED NOT NULL,
  fondo          TINYINT UNSIGNED NOT NULL DEFAULT 1,
  lado           ENUM('A','B') NOT NULL DEFAULT 'A',   -- cara del rack
  orientacion    ENUM('N','S','E','W') NOT NULL DEFAULT 'N',
  ambiente_id    INT UNSIGNED NOT NULL,
  capacidad_pallets SMALLINT UNSIGNED NOT NULL DEFAULT 1,
  ocupado        TINYINT(1) NOT NULL DEFAULT 0,
  picked         TINYINT(1) NOT NULL DEFAULT 0,
  color_hex      CHAR(7) NULL,
  title          VARCHAR(100) NULL,
  -- Códigos
  code           VARCHAR(32)  NOT NULL,          -- C01-N01-F1
  code_full      VARCHAR(64)  NOT NULL,          -- DEP1-R01-C01-N01-F1
  pos_code       VARCHAR(32)  NOT NULL,          -- alias corto
  pos_code_full  VARCHAR(64)  NOT NULL,          -- alias largo
  activo         TINYINT(1)   NOT NULL DEFAULT 1,
  created_at     DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at     DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at     DATETIME NULL,
  CONSTRAINT fk_pos_deposito  FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_pos_ambiente  FOREIGN KEY (ambiente_id) REFERENCES wh_ambiente(id),
  UNIQUE KEY uk_pos_geo (deposito_id, rack, columna, nivel, fondo, lado),
  UNIQUE KEY uk_pos_codes (deposito_id, code, code_full, pos_code, pos_code_full)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ======================================================================
-- 3) Lotes y Pallets (contenedores), Items dentro del pallet
-- ======================================================================

DROP TABLE IF EXISTS wh_lote;
CREATE TABLE wh_lote (
  id                 BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  producto_id        BIGINT UNSIGNED NOT NULL,   -- FK a para_productos(id)
  codigo             VARCHAR(64)  NOT NULL,
  fecha_produccion   DATE NULL,
  fecha_vencimiento  DATE NULL,
  observacion        VARCHAR(255) NULL,
  UNIQUE KEY uk_lote_prod_codigo (producto_id, codigo),
  created_at         DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at         DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at         DATETIME NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS wh_pallet;
CREATE TABLE wh_pallet (
  id               BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  codigo           VARCHAR(64)  NOT NULL,         -- etiqueta interna (EAN128/SSCC, etc.)
  deposito_id      INT UNSIGNED NOT NULL,
  posicion_id      BIGINT UNSIGNED NULL,          -- NULL si en tránsito
  estado_id        INT UNSIGNED NOT NULL,         -- wh_pallet_estado
  pickeado         TINYINT(1) NOT NULL DEFAULT 0, -- depalletizado/parcial
  reservado        TINYINT(1) NOT NULL DEFAULT 0,
  peso_bruto_kg    DECIMAL(10,3) NULL,
  volumen_m3       DECIMAL(10,3) NULL,
  observacion      VARCHAR(255) NULL,
  created_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at       DATETIME NULL,
  CONSTRAINT fk_pal_deposito   FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_pal_posicion   FOREIGN KEY (posicion_id) REFERENCES wh_posicion(id),
  CONSTRAINT fk_pal_estado     FOREIGN KEY (estado_id) REFERENCES wh_pallet_estado(id),
  UNIQUE KEY uk_pallet_codigo (codigo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS wh_pallet_item;
CREATE TABLE wh_pallet_item (
  id            BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  pallet_id     BIGINT UNSIGNED NOT NULL,
  producto_id   BIGINT UNSIGNED NOT NULL,
  lote_id       BIGINT UNSIGNED NOT NULL,
  uv_cajas      INT UNSIGNED NOT NULL DEFAULT 0,  -- UV: cajas/bolsas (unidad de venta)
  uc_unidades   INT UNSIGNED NOT NULL DEFAULT 0,  -- UC: unidades sueltas (consumo)
  peso_bruto_kg DECIMAL(10,3) NULL,
  created_at    DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at    DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at    DATETIME NULL,
  CONSTRAINT fk_pi_pallet    FOREIGN KEY (pallet_id) REFERENCES wh_pallet(id) ON DELETE CASCADE,
  CONSTRAINT fk_pi_lote      FOREIGN KEY (lote_id)   REFERENCES wh_lote(id),
  UNIQUE KEY uk_pi_unique (pallet_id, producto_id, lote_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ======================================================================
-- 4) Movimientos & Stock resumido
-- ======================================================================

DROP TABLE IF EXISTS wh_move;
CREATE TABLE wh_move (
  id              BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  deposito_id     INT UNSIGNED NOT NULL,
  tipo            ENUM('IN','MOVE','OUT') NOT NULL,
  motivo          VARCHAR(64) NULL, -- RECEPCION, UBICACION, REMANEO, PICKING, DESPACHO, AJUSTE, etc.
  pallet_id       BIGINT UNSIGNED NULL,     -- movimiento por pallet completo
  producto_id     BIGINT UNSIGNED NULL,        -- o parcial por producto
  lote_id         BIGINT UNSIGNED NULL,
  from_pos_id     BIGINT UNSIGNED NULL,
  to_pos_id       BIGINT UNSIGNED NULL,
  delta_uv        INT  NOT NULL DEFAULT 0,  -- + entrada, - salida
  delta_uc        INT  NOT NULL DEFAULT 0,
  referencia      VARCHAR(64) NULL,         -- p.ej. #ingreso, #pedido
  user_id         INT UNSIGNED NULL,        -- futuro: sys_users
  created_at      DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_mv_dep     FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_mv_pal     FOREIGN KEY (pallet_id)   REFERENCES wh_pallet(id),
  CONSTRAINT fk_mv_lote    FOREIGN KEY (lote_id)     REFERENCES wh_lote(id),
  CONSTRAINT fk_mv_from    FOREIGN KEY (from_pos_id) REFERENCES wh_posicion(id),
  CONSTRAINT fk_mv_to      FOREIGN KEY (to_pos_id)   REFERENCES wh_posicion(id),
  INDEX idx_mv_busq (deposito_id, pallet_id, producto_id, lote_id, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Stock resumido por POSICIÓN + PRODUCTO + LOTE (+ opcional PALLET)
-- Se usan columnas generadas persistidas para normalizar NULL→0 y permitir UNIQUE.
DROP TABLE IF EXISTS wh_stock;
CREATE TABLE wh_stock (
  id               BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  deposito_id      INT UNSIGNED NOT NULL,
  posicion_id      BIGINT UNSIGNED NULL,      -- NULL si en tránsito
  producto_id      BIGINT UNSIGNED NOT NULL,
  lote_id          BIGINT UNSIGNED NOT NULL,
  pallet_id        BIGINT UNSIGNED NULL,      -- saldo por pallet (si corresponde)
  qty_uv           INT  NOT NULL DEFAULT 0,   -- cajas
  qty_uc           INT  NOT NULL DEFAULT 0,   -- unidades
  pickeado         TINYINT(1) NOT NULL DEFAULT 0,
  updated_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  -- Normalizaciones para índices únicos (no se pueden usar funciones en índices)
  posicion_id_norm BIGINT UNSIGNED AS (IFNULL(posicion_id, 0)) STORED,
  pallet_id_norm   BIGINT UNSIGNED AS (IFNULL(pallet_id, 0)) STORED,

  UNIQUE KEY uk_stk_key (deposito_id, posicion_id_norm, producto_id, lote_id, pallet_id_norm),
  INDEX idx_stk_pos (deposito_id, posicion_id),
  INDEX idx_stk_prod (producto_id, lote_id),
  INDEX idx_stk_fast_pick (deposito_id, producto_id, lote_id, posicion_id, pallet_id),

  CONSTRAINT fk_stk_dep   FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_stk_pos   FOREIGN KEY (posicion_id) REFERENCES wh_posicion(id),
  CONSTRAINT fk_stk_lote  FOREIGN KEY (lote_id)     REFERENCES wh_lote(id),
  CONSTRAINT fk_stk_pal   FOREIGN KEY (pallet_id)   REFERENCES wh_pallet(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ======================================================================
-- 5) FKs hacia para_productos (si existe)
-- ======================================================================

-- Si para_productos existe, estas FKs funcionarán; si no, ejecuta luego.
ALTER TABLE wh_lote
  ADD CONSTRAINT fk_lote_producto
  FOREIGN KEY (producto_id) REFERENCES para_productos(id)
  ON UPDATE RESTRICT ON DELETE RESTRICT;

ALTER TABLE wh_pallet_item
  ADD CONSTRAINT fk_pi_producto
  FOREIGN KEY (producto_id) REFERENCES para_productos(id)
  ON UPDATE RESTRICT ON DELETE RESTRICT;

-- ======================================================================
-- 6) Datos mínimos para probar
-- ======================================================================

INSERT INTO wh_deposito (code, nombre) VALUES ('DEP1','Depósito Principal')
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre);

-- Posición SECO
INSERT INTO wh_posicion
(deposito_id, rack, columna, nivel, fondo, lado, orientacion, ambiente_id, capacidad_pallets, code, code_full, pos_code, pos_code_full, color_hex, title)
SELECT d.id, 1, 1, 1, 1, 'A', 'N', a.id, 1,
       'C01-N01-F1', CONCAT('DEP1-','R01-','C01-N01-F1'), 'C01-N01-F1', CONCAT('DEP1-','R01-','C01-N01-F1'),
       '#0d6efd', 'Rack1 Col1 Niv1 F1'
FROM wh_deposito d, wh_ambiente a
WHERE d.code='DEP1' AND a.code='SECO'
LIMIT 1;

-- Posición CUARENTENA
INSERT INTO wh_posicion
(deposito_id, rack, columna, nivel, fondo, lado, orientacion, ambiente_id, capacidad_pallets, code, code_full, pos_code, pos_code_full, color_hex, title)
SELECT d.id, 99, 1, 1, 1, 'A', 'N', a.id, 10,
       'CUAR-01', 'DEP1-R99-CUAR-01', 'CUAR-01', 'DEP1-R99-CUAR-01',
       '#6f42c1', 'Área de Cuarentena'
FROM wh_deposito d, wh_ambiente a
WHERE d.code='DEP1' AND a.code='CUARENTENA'
LIMIT 1;

SET FOREIGN_KEY_CHECKS = 1;
