/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 2C: Vistas Warehouse (wh_*)
 * Archivo: database/step2c_wh_views.sql
 *
 * Crea vistas para:
 * 1) Posiciones existentes (DEP1)
 * 2) Pallets en CUARENTENA (por posición)
 * 3) Movimientos mínimos (últimos movimientos - ordenar al consultar)
 * 4) Stock por posición (DEP1)
 *
 * Notas:
 * - Las vistas no garantizan orden. Usa ORDER BY/LIMIT al consultar.
 * - Requiere tablas del Paso 2 (wh_*).
 * ===================================================================== */

SET NAMES utf8mb4;
SET SQL_SAFE_UPDATES = 0;

-- ---------------------------------------------------------------------
-- Limpieza previa
-- ---------------------------------------------------------------------
DROP VIEW IF EXISTS v_wh_posiciones;
DROP VIEW IF EXISTS v_wh_posiciones_dep1;
DROP VIEW IF EXISTS v_wh_pallets_cuarentena;
DROP VIEW IF EXISTS v_wh_moves_min;
DROP VIEW IF EXISTS v_wh_stock;
DROP VIEW IF EXISTS v_wh_stock_dep1;

-- ---------------------------------------------------------------------
-- 1) Posiciones existentes (todas) + helper filtrado DEP1
-- ---------------------------------------------------------------------
CREATE OR REPLACE VIEW v_wh_posiciones AS
SELECT
  p.id,
  p.deposito_id,
  d.code        AS deposito_code,
  p.rack,
  p.columna,
  p.nivel,
  p.fondo,
  p.lado,
  p.code_full,
  p.ambiente_id,
  a.code        AS ambiente_code,
  a.nombre      AS ambiente_nombre
FROM wh_posicion p
JOIN wh_deposito d  ON d.id  = p.deposito_id
JOIN wh_ambiente a  ON a.id  = p.ambiente_id;

-- Versión filtrada equivalente al SQL "Posiciones existentes (DEP1)"
CREATE OR REPLACE VIEW v_wh_posiciones_dep1 AS
SELECT
  id, deposito_id, rack, columna, nivel, fondo, lado, code_full, ambiente_id
FROM v_wh_posiciones
WHERE deposito_code = 'DEP1';

-- ---------------------------------------------------------------------
-- 2) Pallets en CUARENTENA (por posición)
-- ---------------------------------------------------------------------
CREATE OR REPLACE VIEW v_wh_pallets_cuarentena AS
SELECT
  p.codigo          AS pallet,
  pe.code           AS estado,
  d.code            AS deposito_code,
  p.posicion_id,
  pos.code_full     AS posicion
FROM wh_pallet p
JOIN wh_pallet_estado pe ON pe.id = p.estado_id
JOIN wh_deposito d       ON d.id  = p.deposito_id
LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
WHERE pe.code = 'CUARENTENA';

-- ---------------------------------------------------------------------
-- 3) Movimientos mínimos (para "últimos movimientos")
--    Ordenar/limitar al consultar: ORDER BY id DESC LIMIT 50
-- ---------------------------------------------------------------------
CREATE OR REPLACE VIEW v_wh_moves_min AS
SELECT
  m.id,
  m.tipo,
  m.motivo,
  m.deposito_id,
  d.code      AS deposito_code,
  m.pallet_id,
  m.producto_id,
  m.lote_id,
  m.from_pos_id,
  m.to_pos_id,
  m.delta_uv,
  m.delta_uc,
  m.created_at
FROM wh_move m
JOIN wh_deposito d ON d.id = m.deposito_id;

-- ---------------------------------------------------------------------
-- 4) Stock por posición (todas) + helper filtrado DEP1
-- ---------------------------------------------------------------------
CREATE OR REPLACE VIEW v_wh_stock AS
SELECT
  s.id,
  s.deposito_id,
  d.code        AS deposito_code,
  s.posicion_id,
  s.producto_id,
  s.lote_id,
  s.pallet_id,
  s.qty_uv,
  s.qty_uc,
  s.pickeado,
  s.updated_at
FROM wh_stock s
JOIN wh_deposito d ON d.id = s.deposito_id;

-- Versión filtrada equivalente al SQL "Stock por posición (DEP1)"
CREATE OR REPLACE VIEW v_wh_stock_dep1 AS
SELECT
  deposito_id, posicion_id, producto_id, lote_id, pallet_id, qty_uv, qty_uc, pickeado, updated_at
FROM v_wh_stock
WHERE deposito_code = 'DEP1';
