/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 3 (phpMyAdmin v3): Rutinas + Trigger para Warehouse
 * - Usar el DELIMITADOR de la UI en: $$
 * - No incluye líneas DELIMITER dentro del script.
 * - Comentarios sólo con /* ... * / o -- (con espacio).
 * ===================================================================== */

/* Limpieza previa (idempotente) */
DROP TRIGGER IF EXISTS trg_wh_move_ai $$
DROP PROCEDURE IF EXISTS sp_wh_stock_apply $$
DROP PROCEDURE IF EXISTS sp_wh_pos_refresh_flags $$
DROP PROCEDURE IF EXISTS sp_wh_move_after_insert $$

/* -----------------------------------------------------------
   SP: Upsert en wh_stock + validaciones + refresco de flags
----------------------------------------------------------- */
CREATE PROCEDURE sp_wh_stock_apply(
  IN p_deposito_id   INT UNSIGNED,
  IN p_posicion_id   BIGINT UNSIGNED,   -- puede ser NULL
  IN p_producto_id   BIGINT UNSIGNED,
  IN p_lote_id       BIGINT UNSIGNED,
  IN p_pallet_id     BIGINT UNSIGNED,   -- puede ser NULL
  IN p_delta_uv      INT,
  IN p_delta_uc      INT,
  IN p_mark_pickeado TINYINT
)
BEGIN
  DECLARE v_id BIGINT UNSIGNED DEFAULT NULL;
  DECLARE v_qty_uv INT DEFAULT 0;
  DECLARE v_qty_uc INT DEFAULT 0;
  DECLARE v_pickeado TINYINT DEFAULT 0;

  /* Buscar fila existente (NULL-safe con <=>) */
  SELECT id, qty_uv, qty_uc, pickeado
    INTO v_id, v_qty_uv, v_qty_uc, v_pickeado
  FROM wh_stock
  WHERE deposito_id = p_deposito_id
    AND (posicion_id <=> p_posicion_id)
    AND producto_id = p_producto_id
    AND lote_id = p_lote_id
    AND (pallet_id <=> p_pallet_id)
  FOR UPDATE;

  IF v_id IS NULL THEN
    IF (p_delta_uv < 0 OR p_delta_uc < 0) THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'Stock insuficiente o inexistente para decrementar.';
    END IF;

    INSERT INTO wh_stock(
      deposito_id, posicion_id, producto_id, lote_id, pallet_id,
      qty_uv, qty_uc, pickeado
    ) VALUES (
      p_deposito_id, p_posicion_id, p_producto_id, p_lote_id, p_pallet_id,
      p_delta_uv, p_delta_uc, IF(p_mark_pickeado=1 OR p_delta_uc<>0, 1, 0)
    );
  ELSE
    SET v_qty_uv = v_qty_uv + p_delta_uv;
    SET v_qty_uc = v_qty_uc + p_delta_uc;

    IF v_qty_uv < 0 OR v_qty_uc < 0 THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'La operación dejaría el stock en negativo.';
    END IF;

    IF v_qty_uv = 0 AND v_qty_uc = 0 THEN
      DELETE FROM wh_stock WHERE id = v_id;
    ELSE
      UPDATE wh_stock
         SET qty_uv  = v_qty_uv,
             qty_uc  = v_qty_uc,
             pickeado = CASE
                          WHEN p_mark_pickeado=1 OR p_delta_uc<>0 THEN 1
                          ELSE pickeado
                        END
       WHERE id = v_id;
    END IF;
  END IF;

  /* Refrescar flags de la posición */
  IF p_posicion_id IS NOT NULL THEN
    CALL sp_wh_pos_refresh_flags(p_posicion_id);
  END IF;
END $$

/* -----------------------------------------------------------
   SP: Recalcular ocupado/picked de una posición
----------------------------------------------------------- */
CREATE PROCEDURE sp_wh_pos_refresh_flags(
  IN p_posicion_id BIGINT UNSIGNED
)
BEGIN
  DECLARE v_ocupa INT DEFAULT 0;
  DECLARE v_pick INT DEFAULT 0;

  SELECT EXISTS(
           SELECT 1 FROM wh_stock
            WHERE posicion_id = p_posicion_id
              AND (qty_uv > 0 OR qty_uc > 0)
         ),
         EXISTS(
           SELECT 1 FROM wh_stock
            WHERE posicion_id = p_posicion_id
              AND (pickeado = 1 OR qty_uc > 0)
         )
    INTO v_ocupa, v_pick;

  UPDATE wh_posicion
     SET ocupado = IFNULL(v_ocupa,0),
         picked  = IFNULL(v_pick,0)
   WHERE id = p_posicion_id;
END $$

/* -----------------------------------------------------------
   SP: Aplicar efectos de un movimiento insertado (wh_move.NEW.id)
----------------------------------------------------------- */
CREATE PROCEDURE sp_wh_move_after_insert(IN p_move_id BIGINT UNSIGNED)
BEGIN
  DECLARE v_tipo VARCHAR(10);
  DECLARE v_deposito INT UNSIGNED;
  DECLARE v_pallet BIGINT UNSIGNED;
  DECLARE v_producto BIGINT UNSIGNED;
  DECLARE v_lote BIGINT UNSIGNED;
  DECLARE v_from BIGINT UNSIGNED;
  DECLARE v_to   BIGINT UNSIGNED;
  DECLARE v_duv INT;
  DECLARE v_duc INT;

  SELECT tipo, deposito_id, pallet_id, producto_id, lote_id,
         from_pos_id, to_pos_id, delta_uv, delta_uc
    INTO v_tipo, v_deposito, v_pallet, v_producto, v_lote,
         v_from, v_to, v_duv, v_duc
  FROM wh_move
  WHERE id = p_move_id
  FOR UPDATE;

  /* Validaciones mínimas */
  IF v_tipo = 'IN' AND v_to IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'IN requiere to_pos_id.';
  END IF;
  IF v_tipo = 'MOVE' AND (v_from IS NULL OR v_to IS NULL) THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'MOVE requiere from_pos_id y to_pos_id.';
  END IF;
  IF v_tipo = 'OUT' AND v_from IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'OUT requiere from_pos_id.';
  END IF;

  /* Caso A: Pallet completo (deltas=0, sin producto/lote) */
  IF v_pallet IS NOT NULL AND v_producto IS NULL AND v_lote IS NULL
     AND v_duv = 0 AND v_duc = 0 THEN

    BEGIN
      DECLARE done INT DEFAULT 0;
      DECLARE c_prod BIGINT UNSIGNED;
      DECLARE c_lote BIGINT UNSIGNED;
      DECLARE c_uv INT;
      DECLARE c_uc INT;

      DECLARE cur CURSOR FOR
        SELECT producto_id, lote_id, uv_cajas, uc_unidades
          FROM wh_pallet_item
         WHERE pallet_id = v_pallet;
      DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = 1;

      OPEN cur;
      read_loop: LOOP
        FETCH cur INTO c_prod, c_lote, c_uv, c_uc;
        IF done = 1 THEN LEAVE read_loop; END IF;

        IF v_tipo = 'IN' THEN
          CALL sp_wh_stock_apply(v_deposito, v_to,   c_prod, c_lote, v_pallet, +c_uv, +c_uc, 0);
        ELSEIF v_tipo = 'MOVE' THEN
          CALL sp_wh_stock_apply(v_deposito, v_from, c_prod, c_lote, v_pallet, -c_uv, -c_uc, 0);
          CALL sp_wh_stock_apply(v_deposito, v_to,   c_prod, c_lote, v_pallet, +c_uv, +c_uc, 0);
        ELSEIF v_tipo = 'OUT' THEN
          CALL sp_wh_stock_apply(v_deposito, v_from, c_prod, c_lote, v_pallet, -c_uv, -c_uc, 0);
        END IF;
      END LOOP;
      CLOSE cur;
    END;

  /* Caso B: Parcial por producto/lote */
  ELSE
    IF v_producto IS NULL OR v_lote IS NULL THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'Movimiento parcial requiere producto_id y lote_id.';
    END IF;

    IF v_tipo = 'IN' THEN
      CALL sp_wh_stock_apply(v_deposito, v_to,   v_producto, v_lote, v_pallet, +v_duv, +v_duc, IF(v_duc<>0,1,0));
    ELSEIF v_tipo = 'MOVE' THEN
      CALL sp_wh_stock_apply(v_deposito, v_from, v_producto, v_lote, v_pallet, -v_duv, -v_duc, 0);
      CALL sp_wh_stock_apply(v_deposito, v_to,   v_producto, v_lote, v_pallet, +v_duv, +v_duc, IF(v_duc<>0,1,0));
    ELSEIF v_tipo = 'OUT' THEN
      CALL sp_wh_stock_apply(v_deposito, v_from, v_producto, v_lote, v_pallet, -v_duv, -v_duc, 0);
    END IF;
  END IF;

  /* Actualizar posición del pallet (si corresponde) */
  IF v_pallet IS NOT NULL THEN
    IF v_tipo IN ('IN','MOVE') THEN
      UPDATE wh_pallet SET posicion_id = v_to WHERE id = v_pallet;
    ELSEIF v_tipo = 'OUT' THEN
      IF v_producto IS NULL AND v_lote IS NULL AND v_duv = 0 AND v_duc = 0 THEN
        UPDATE wh_pallet SET posicion_id = NULL WHERE id = v_pallet;
      END IF;
    END IF;

    IF v_duc <> 0 THEN
      UPDATE wh_pallet SET pickeado = 1 WHERE id = v_pallet;
    END IF;
  END IF;

  /* Refrescar flags de posiciones */
  IF v_from IS NOT NULL THEN CALL sp_wh_pos_refresh_flags(v_from); END IF;
  IF v_to   IS NOT NULL THEN CALL sp_wh_pos_refresh_flags(v_to);   END IF;
END $$

/* -----------------------------------------------------------
   TRIGGER: AFTER INSERT en wh_move
----------------------------------------------------------- */
CREATE TRIGGER trg_wh_move_ai
AFTER INSERT ON wh_move
FOR EACH ROW
BEGIN
  CALL sp_wh_move_after_insert(NEW.id);
END $$
