/* =====================================================================
 * FIX #1442: Evitar leer wh_move dentro del trigger que inserta en wh_move
 * Estrategia: el trigger pasa los valores NEW.* por parámetros al SP,
 *             y el SP no consulta wh_move.
 * Requisitos previos: existen sp_wh_stock_apply y sp_wh_pos_refresh_flags (Paso 3).
 * Uso: establecer delimitador de la UI en $$ antes de ejecutar.
 * ===================================================================== */

DROP TRIGGER IF EXISTS trg_wh_move_ai $$
DROP PROCEDURE IF EXISTS sp_wh_move_after_insert $$   -- eliminamos la antigua
DROP PROCEDURE IF EXISTS sp_wh_move_apply $$

/* ---------------------------------------------------------------------
   SP nuevo: aplica un movimiento (sin leer wh_move)
--------------------------------------------------------------------- */
CREATE PROCEDURE sp_wh_move_apply(
  IN p_tipo         VARCHAR(10),          -- 'IN' | 'MOVE' | 'OUT'
  IN p_deposito_id  INT UNSIGNED,
  IN p_pallet_id    BIGINT UNSIGNED,      -- puede ser NULL
  IN p_producto_id  BIGINT UNSIGNED,      -- puede ser NULL
  IN p_lote_id      BIGINT UNSIGNED,      -- puede ser NULL
  IN p_from_pos_id  BIGINT UNSIGNED,      -- puede ser NULL
  IN p_to_pos_id    BIGINT UNSIGNED,      -- puede ser NULL
  IN p_delta_uv     INT,
  IN p_delta_uc     INT
)
BEGIN
  /* Validaciones mínimas por tipo */
  IF p_tipo = 'IN' AND p_to_pos_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'IN requiere to_pos_id.';
  END IF;
  IF p_tipo = 'MOVE' AND (p_from_pos_id IS NULL OR p_to_pos_id IS NULL) THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'MOVE requiere from_pos_id y to_pos_id.';
  END IF;
  IF p_tipo = 'OUT' AND p_from_pos_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'OUT requiere from_pos_id.';
  END IF;

  /* Caso A: PALLET COMPLETO (sin producto/lote y deltas = 0) */
  IF p_pallet_id IS NOT NULL AND p_producto_id IS NULL AND p_lote_id IS NULL
     AND p_delta_uv = 0 AND p_delta_uc = 0 THEN

    BEGIN
      DECLARE done INT DEFAULT 0;
      DECLARE c_prod BIGINT UNSIGNED;
      DECLARE c_lote BIGINT UNSIGNED;
      DECLARE c_uv INT;
      DECLARE c_uc INT;

      DECLARE cur CURSOR FOR
        SELECT producto_id, lote_id, uv_cajas, uc_unidades
          FROM wh_pallet_item
         WHERE pallet_id = p_pallet_id;

      DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = 1;

      OPEN cur;
      read_loop: LOOP
        FETCH cur INTO c_prod, c_lote, c_uv, c_uc;
        IF done = 1 THEN LEAVE read_loop; END IF;

        IF p_tipo = 'IN' THEN
          CALL sp_wh_stock_apply(p_deposito_id, p_to_pos_id,   c_prod, c_lote, p_pallet_id, +c_uv, +c_uc, 0);
        ELSEIF p_tipo = 'MOVE' THEN
          CALL sp_wh_stock_apply(p_deposito_id, p_from_pos_id, c_prod, c_lote, p_pallet_id, -c_uv, -c_uc, 0);
          CALL sp_wh_stock_apply(p_deposito_id, p_to_pos_id,   c_prod, c_lote, p_pallet_id, +c_uv, +c_uc, 0);
        ELSEIF p_tipo = 'OUT' THEN
          CALL sp_wh_stock_apply(p_deposito_id, p_from_pos_id, c_prod, c_lote, p_pallet_id, -c_uv, -c_uc, 0);
        END IF;
      END LOOP;
      CLOSE cur;
    END;

  /* Caso B: PARCIAL (producto/lote con deltas) */
  ELSE
    IF p_producto_id IS NULL OR p_lote_id IS NULL THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'Movimiento parcial requiere producto_id y lote_id.';
    END IF;

    IF p_tipo = 'IN' THEN
      CALL sp_wh_stock_apply(p_deposito_id, p_to_pos_id,   p_producto_id, p_lote_id, p_pallet_id, +p_delta_uv, +p_delta_uc, IF(p_delta_uc<>0,1,0));
    ELSEIF p_tipo = 'MOVE' THEN
      CALL sp_wh_stock_apply(p_deposito_id, p_from_pos_id, p_producto_id, p_lote_id, p_pallet_id, -p_delta_uv, -p_delta_uc, 0);
      CALL sp_wh_stock_apply(p_deposito_id, p_to_pos_id,   p_producto_id, p_lote_id, p_pallet_id, +p_delta_uv, +p_delta_uc, IF(p_delta_uc<>0,1,0));
    ELSEIF p_tipo = 'OUT' THEN
      CALL sp_wh_stock_apply(p_deposito_id, p_from_pos_id, p_producto_id, p_lote_id, p_pallet_id, -p_delta_uv, -p_delta_uc, 0);
    END IF;
  END IF;

  /* Actualizar ubicación/flags del pallet si corresponde */
  IF p_pallet_id IS NOT NULL THEN
    IF p_tipo IN ('IN','MOVE') THEN
      UPDATE wh_pallet SET posicion_id = p_to_pos_id WHERE id = p_pallet_id;
    ELSEIF p_tipo = 'OUT' THEN
      IF p_producto_id IS NULL AND p_lote_id IS NULL AND p_delta_uv = 0 AND p_delta_uc = 0 THEN
        UPDATE wh_pallet SET posicion_id = NULL WHERE id = p_pallet_id;
      END IF;
    END IF;

    IF p_delta_uc <> 0 THEN
      UPDATE wh_pallet SET pickeado = 1 WHERE id = p_pallet_id;
    END IF;
  END IF;

  /* Refrescar flags de posiciones (extra) */
  IF p_from_pos_id IS NOT NULL THEN CALL sp_wh_pos_refresh_flags(p_from_pos_id); END IF;
  IF p_to_pos_id   IS NOT NULL THEN CALL sp_wh_pos_refresh_flags(p_to_pos_id);   END IF;
END $$

/* ---------------------------------------------------------------------
   Trigger nuevo: pasa NEW.* al SP (no lee wh_move)
--------------------------------------------------------------------- */
CREATE TRIGGER trg_wh_move_ai
AFTER INSERT ON wh_move
FOR EACH ROW
BEGIN
  CALL sp_wh_move_apply(
    NEW.tipo,
    NEW.deposito_id,
    NEW.pallet_id,
    NEW.producto_id,
    NEW.lote_id,
    NEW.from_pos_id,
    NEW.to_pos_id,
    NEW.delta_uv,
    NEW.delta_uc
  );
END $$
