/* STEP 4 (compat v2): Alta de Ingreso (SP) — JSON_TABLE sin DEFAULT numéricos
   - Usar el Delimitador de la UI en: $$
   - Extraemos TODO como texto desde JSON_TABLE y casteamos fuera.
*/

DROP PROCEDURE IF EXISTS sp_wh_ingreso_create $$
CREATE PROCEDURE sp_wh_ingreso_create(
  IN p_deposito_code        VARCHAR(32),
  IN p_pallet_codigo        VARCHAR(64),
  IN p_pos_code             VARCHAR(64),   -- opcional
  IN p_estado_code          VARCHAR(32),   -- opcional (default 'CUARENTENA')
  IN p_items_json           JSON,          -- [{producto_id,lote_codigo,fecha_produccion?,fecha_vencimiento?,uv_cajas,uc_unidades}]
  IN p_overwrite_if_exists  TINYINT        -- 0=error si existe, 1=rehusar/limpiar
)
BEGIN
  /* ---------- DECLARACIONES (deben ir primero) ---------- */
  DECLARE v_dep_id INT UNSIGNED;
  DECLARE v_pos_id BIGINT UNSIGNED;
  DECLARE v_pal_id BIGINT UNSIGNED;
  DECLARE v_est_id INT UNSIGNED;
  DECLARE v_est_emb INT UNSIGNED;
  DECLARE v_move_id BIGINT UNSIGNED;
  DECLARE v_estado_code VARCHAR(32);

  /* Handler de errores: rollback y propaga */
  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  /* ---------- INICIALIZACIONES ---------- */
  SET v_estado_code = IFNULL(p_estado_code, 'CUARENTENA');

  /* ---------- VALIDACIONES ---------- */
  IF p_deposito_code IS NULL OR p_deposito_code = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_deposito_code es requerido.';
  END IF;

  IF p_pallet_codigo IS NULL OR p_pallet_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_pallet_codigo es requerido.';
  END IF;

  IF p_items_json IS NULL OR JSON_TYPE(p_items_json) <> 'ARRAY' OR JSON_LENGTH(p_items_json) = 0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_items_json debe ser un ARRAY JSON con al menos un ítem.';
  END IF;

  /* ---------- TRANSACCIÓN ---------- */
  START TRANSACTION;

  /* Depósito */
  SELECT id INTO v_dep_id
    FROM wh_deposito
   WHERE code = p_deposito_code
   LIMIT 1;
  IF v_dep_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Depósito no encontrado.';
  END IF;

  /* Estado pallet */
  SELECT id INTO v_est_id
    FROM wh_pallet_estado
   WHERE code = v_estado_code
   LIMIT 1;
  IF v_est_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Estado de pallet no válido.';
  END IF;

  /* Estado EMBARCADO (para excluirlo de ocupación) */
  SELECT id INTO v_est_emb
    FROM wh_pallet_estado
   WHERE code = 'EMBARCADO'
   LIMIT 1;

  /* Posición destino: la indicada o una de CUARENTENA con menor ocupación */
  IF p_pos_code IS NOT NULL AND p_pos_code <> '' THEN
    SELECT p.id INTO v_pos_id
      FROM wh_posicion p
     WHERE p.deposito_id = v_dep_id
       AND (p.code = p_pos_code OR p.code_full = p_pos_code OR p.pos_code = p_pos_code OR p.pos_code_full = p_pos_code)
     LIMIT 1;
    IF v_pos_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Posición indicada no existe en el depósito.';
    END IF;
  ELSE
    SELECT p.id INTO v_pos_id
      FROM wh_posicion p
      JOIN wh_ambiente a ON a.id = p.ambiente_id AND a.code = 'CUARENTENA'
      LEFT JOIN wh_pallet pal
             ON pal.posicion_id = p.id
            AND pal.deleted_at IS NULL
            AND (v_est_emb IS NULL OR pal.estado_id <> v_est_emb)
     WHERE p.deposito_id = v_dep_id
       AND p.activo = 1
     GROUP BY p.id
     ORDER BY COUNT(pal.id) ASC, p.id ASC
     LIMIT 1;
    IF v_pos_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'No hay posición de CUARENTENA disponible en el depósito.';
    END IF;
  END IF;

  /* Tabla temporal con ítems del JSON (tipos finales) */
  DROP TEMPORARY TABLE IF EXISTS _ing_items;
  CREATE TEMPORARY TABLE _ing_items (
    producto_id      BIGINT UNSIGNED NOT NULL,
    lote_codigo      VARCHAR(64) NOT NULL,
    fecha_produccion DATE NULL,
    fecha_vencimiento DATE NULL,
    uv_cajas         INT UNSIGNED NOT NULL DEFAULT 0,
    uc_unidades      INT UNSIGNED NOT NULL DEFAULT 0
  ) ENGINE=MEMORY;

  /* JSON_TABLE: leemos como cadenas y casteamos en el SELECT de inserción */
  INSERT INTO _ing_items (producto_id, lote_codigo, fecha_produccion, fecha_vencimiento, uv_cajas, uc_unidades)
  SELECT
    CAST(jt.producto_id_str AS UNSIGNED)                                                AS producto_id,
    jt.lote_codigo                                                                      AS lote_codigo,
    IFNULL(NULLIF(jt.fecha_produccion_str,''), NULL)                                     AS fecha_produccion,
    IFNULL(NULLIF(jt.fecha_vencimiento_str,''), NULL)                                    AS fecha_vencimiento,
    COALESCE(NULLIF(jt.uv_cajas_str,''), '0') + 0                                        AS uv_cajas,
    COALESCE(NULLIF(jt.uc_unidades_str,''), '0') + 0                                     AS uc_unidades
  FROM JSON_TABLE(
         p_items_json, '$[*]'
         COLUMNS (
           producto_id_str       VARCHAR(30) PATH '$.producto_id'       NULL ON EMPTY,
           lote_codigo           VARCHAR(64) PATH '$.lote_codigo',
           fecha_produccion_str  VARCHAR(10) PATH '$.fecha_produccion'  NULL ON EMPTY,
           fecha_vencimiento_str VARCHAR(10) PATH '$.fecha_vencimiento' NULL ON EMPTY,
           uv_cajas_str          VARCHAR(30) PATH '$.uv_cajas'          NULL ON EMPTY,
           uc_unidades_str       VARCHAR(30) PATH '$.uc_unidades'       NULL ON EMPTY
         )
       ) AS jt;

  /* Validaciones de staging */
  IF (SELECT COUNT(*) FROM _ing_items WHERE producto_id IS NULL OR producto_id=0) > 0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Al menos un ítem carece de producto_id válido.';
  END IF;

  IF (SELECT COUNT(*) FROM _ing_items WHERE lote_codigo IS NULL OR lote_codigo='') > 0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Al menos un ítem carece de lote_codigo.';
  END IF;

  IF (SELECT COUNT(*) FROM _ing_items WHERE COALESCE(uv_cajas,0)=0 AND COALESCE(uc_unidades,0)=0) > 0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Hay ítems con cantidades nulas (uv_cajas=0 y uc_unidades=0).';
  END IF;

  /* Upsert de LOTES (prefiere nuevos valores no nulos) */
  INSERT INTO wh_lote (producto_id, codigo, fecha_produccion, fecha_vencimiento)
  SELECT i.producto_id, i.lote_codigo, i.fecha_produccion, i.fecha_vencimiento
    FROM _ing_items i
  ON DUPLICATE KEY UPDATE
    wh_lote.fecha_produccion  = COALESCE(VALUES(fecha_produccion),  wh_lote.fecha_produccion),
    wh_lote.fecha_vencimiento = COALESCE(VALUES(fecha_vencimiento), wh_lote.fecha_vencimiento);

  /* Crear / reutilizar PALLET */
  SELECT id INTO v_pal_id FROM wh_pallet WHERE codigo = p_pallet_codigo LIMIT 1;

  IF v_pal_id IS NULL THEN
    INSERT INTO wh_pallet (codigo, deposito_id, posicion_id, estado_id, pickeado, reservado, observacion)
    VALUES (p_pallet_codigo, v_dep_id, v_pos_id, v_est_id, 0, 0, 'Ingreso automático');
    SET v_pal_id = LAST_INSERT_ID();
  ELSE
    IF p_overwrite_if_exists = 1 THEN
      UPDATE wh_pallet
         SET deposito_id = v_dep_id,
             posicion_id = v_pos_id,
             estado_id   = v_est_id,
             pickeado    = 0,
             reservado   = 0
       WHERE id = v_pal_id;
      DELETE FROM wh_pallet_item WHERE pallet_id = v_pal_id;
    ELSE
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'El pallet ya existe. Use p_overwrite_if_exists=1 para reutilizar.';
    END IF;
  END IF;

  /* Ítems del pallet (acumula si ya existe fila) */
  INSERT INTO wh_pallet_item (pallet_id, producto_id, lote_id, uv_cajas, uc_unidades)
  SELECT v_pal_id, i.producto_id, l.id, i.uv_cajas, i.uc_unidades
    FROM _ing_items i
    JOIN wh_lote l
      ON l.producto_id = i.producto_id
     AND l.codigo      = i.lote_codigo
  ON DUPLICATE KEY UPDATE
    uv_cajas    = wh_pallet_item.uv_cajas    + VALUES(uv_cajas),
    uc_unidades = wh_pallet_item.uc_unidades + VALUES(uc_unidades);

  /* Movimiento IN (dispara trigger del Paso 3 → actualiza wh_stock) */
  INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
  VALUES (v_dep_id, 'IN', 'RECEPCION', v_pal_id, NULL, NULL, NULL, v_pos_id, 0, 0, CONCAT('ING-', p_pallet_codigo));
  SET v_move_id = LAST_INSERT_ID();

  COMMIT;

  /* Resultado */
  SELECT
    v_pal_id        AS pallet_id,
    p_pallet_codigo AS pallet_codigo,
    v_dep_id        AS deposito_id,
    v_pos_id        AS posicion_id,
    (SELECT code_full FROM wh_posicion WHERE id=v_pos_id) AS posicion_code_full,
    v_est_id        AS estado_id,
    v_move_id       AS move_id;
END $$
