/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 5: Esquema Packing List (pl_*) + Arribo/Descarga + Cotejo
 * Archivo: database/step5_pl_schema.sql
 *
 * Incluye:
 * - pl_packinglist (cabecera) y pl_packinglist_item (líneas esperadas)
 * - pl_producto_alias (mapa SKU cliente → para_productos.id)
 * - pl_import_batch + pl_import_row (staging de importación Excel)
 * - pl_movil / pl_chofer (catálogos mínimos)
 * - pl_ingreso (arribo + datos de descarga / documento respaldo)
 * - pl_rcv_link (vínculo de lo recibido a pallets/movimientos)
 * - Vistas de conciliación (resúmenes por PL y por ítem)
 * - Triggers: mantener cantidades recibidas por ítem al registrar pl_rcv_link
 *
 * NOTA: la creación automática de productos se hará en un SP del Paso 6
 *       usando pl_producto_alias. Este paso modela el esquema.
 * ===================================================================== */

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- =====================================================================
-- 0) Catálogos de transporte
-- =====================================================================

DROP TABLE IF EXISTS pl_movil;
CREATE TABLE pl_movil (
  id           INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  patente      VARCHAR(20) NOT NULL,
  empresa      VARCHAR(100) NULL,
  activo       TINYINT(1) NOT NULL DEFAULT 1,
  created_at   DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at   DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uk_movil_patente (patente)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS pl_chofer;
CREATE TABLE pl_chofer (
  id           INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  nombre       VARCHAR(100) NOT NULL,
  doc_numero   VARCHAR(50) NULL,   -- DNI/CI
  licencia     VARCHAR(50) NULL,
  empresa      VARCHAR(100) NULL,  -- transportista
  activo       TINYINT(1) NOT NULL DEFAULT 1,
  created_at   DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at   DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 1) Packing List: cabecera + líneas
-- =====================================================================

DROP TABLE IF EXISTS pl_packinglist;
CREATE TABLE pl_packinglist (
  id              BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  codigo          VARCHAR(64) NOT NULL,         -- número/identificador del PL del cliente
  cliente_ref     VARCHAR(64) NULL,             -- código de cliente (si manejamos multi-cliente)
  fecha           DATE NULL,
  estado          ENUM('CREADO','IMPORTADO','EN_DESCARGA','DESCARGADO','VALIDADO','CERRADO')
                  NOT NULL DEFAULT 'CREADO',
  checked_all     TINYINT(1) NOT NULL DEFAULT 0,  -- check general de conformidad
  obs             VARCHAR(255) NULL,
  import_batch_id BIGINT UNSIGNED NULL,         -- FK a staging (opcional)
  created_at      DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at      DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uk_pl_codigo (codigo),
  INDEX idx_pl_estado (estado)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS pl_packinglist_item;
CREATE TABLE pl_packinglist_item (
  id               BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  packinglist_id   BIGINT UNSIGNED NOT NULL,
  -- Identificación del producto (del cliente y nuestra)
  sku_cliente      VARCHAR(64) NOT NULL,              -- SKU provisto por el cliente
  producto_id      BIGINT UNSIGNED NULL,              -- para_productos.id (se autocrea en Paso 6 si falta)
  descripcion      VARCHAR(255) NULL,                 -- descripción que viene del PL
  -- Lote (opcional en PL)
  lote_codigo      VARCHAR(64) NULL,
  fecha_produccion DATE NULL,
  fecha_vencimiento DATE NULL,
  -- Cantidades esperadas y recibidas
  expected_uv      INT UNSIGNED NOT NULL DEFAULT 0,   -- cajas (UV)
  expected_uc      INT UNSIGNED NOT NULL DEFAULT 0,   -- unidades (UC)
  received_uv      INT UNSIGNED NOT NULL DEFAULT 0,
  received_uc      INT UNSIGNED NOT NULL DEFAULT 0,
  -- Diferencias calculadas (generadas)
  diff_uv          INT AS (CAST(expected_uv AS SIGNED) - CAST(received_uv AS SIGNED)) VIRTUAL,
  diff_uc          INT AS (CAST(expected_uc AS SIGNED) - CAST(received_uc AS SIGNED)) VIRTUAL,
  checked_item     TINYINT(1) NOT NULL DEFAULT 0,     -- check individual
  obs              VARCHAR(255) NULL,
  created_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_plitem_pl  FOREIGN KEY (packinglist_id) REFERENCES pl_packinglist(id) ON DELETE CASCADE,
  INDEX idx_plitem_prod (producto_id),
  INDEX idx_plitem_sku (sku_cliente),
  INDEX idx_plitem_pl (packinglist_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 2) Mapeo SKU cliente → producto interno
-- =====================================================================

DROP TABLE IF EXISTS pl_producto_alias;
CREATE TABLE pl_producto_alias (
  id             BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  cliente_ref    VARCHAR(64) NOT NULL,
  sku_cliente    VARCHAR(64) NOT NULL,
  producto_id    BIGINT UNSIGNED NOT NULL,      -- para_productos.id
  producto_desc  VARCHAR(255) NULL,
  created_at     DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at     DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uk_alias (cliente_ref, sku_cliente),
  INDEX idx_alias_prod (producto_id),
  CONSTRAINT fk_alias_prod FOREIGN KEY (producto_id) REFERENCES para_productos(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 3) Staging de importación (Excel → DB)
-- =====================================================================

DROP TABLE IF EXISTS pl_import_batch;
CREATE TABLE pl_import_batch (
  id            BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  tipo          ENUM('PACKINGLIST') NOT NULL DEFAULT 'PACKINGLIST',
  filename      VARCHAR(255) NULL,
  sheet_name    VARCHAR(128) NULL,
  imported_at   DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  user_id       INT UNSIGNED NULL,
  rows_total    INT UNSIGNED NULL,
  rows_ok       INT UNSIGNED NULL,
  rows_error    INT UNSIGNED NULL,
  log           TEXT NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS pl_import_row;
CREATE TABLE pl_import_row (
  id              BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  batch_id        BIGINT UNSIGNED NOT NULL,
  rownum          INT UNSIGNED NOT NULL,
  raw             JSON NULL,                      -- fila cruda del Excel
  status          ENUM('OK','ERROR','SKIPPED') NOT NULL DEFAULT 'OK',
  error_msg       VARCHAR(255) NULL,
  packinglist_id  BIGINT UNSIGNED NULL,
  item_id         BIGINT UNSIGNED NULL,
  created_at      DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_improw_batch FOREIGN KEY (batch_id) REFERENCES pl_import_batch(id) ON DELETE CASCADE,
  INDEX idx_improw_pl (packinglist_id, item_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 4) Arribo y datos de descarga (camión/móvil)
-- =====================================================================

DROP TABLE IF EXISTS pl_ingreso;
CREATE TABLE pl_ingreso (
  id                 BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  packinglist_id     BIGINT UNSIGNED NOT NULL,
  deposito_id        INT UNSIGNED NOT NULL,         -- depósito donde se recibe
  movil_id           INT UNSIGNED NULL,
  chofer_id          INT UNSIGNED NULL,
  movil_desc         VARCHAR(100) NULL,             -- texto libre si no hay catálogo
  chofer_desc        VARCHAR(100) NULL,
  fecha_ingreso      DATE NOT NULL,                 -- fecha del arribo
  llegada_at         DATETIME NULL,                 -- timestamp de llegada a playa
  descarga_inicio_at DATETIME NULL,
  descarga_fin_at    DATETIME NULL,
  operarios_cant     SMALLINT UNSIGNED NULL,
  doc_tipo           ENUM('FACTURA','REMITO','OTRO') NULL,
  doc_numero         VARCHAR(64) NULL,
  doc_fecha          DATE NULL,
  observacion        VARCHAR(255) NULL,
  created_at         DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at         DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_ing_pl   FOREIGN KEY (packinglist_id) REFERENCES pl_packinglist(id) ON DELETE CASCADE,
  CONSTRAINT fk_ing_dep  FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_ing_movil FOREIGN KEY (movil_id) REFERENCES pl_movil(id),
  CONSTRAINT fk_ing_chofer FOREIGN KEY (chofer_id) REFERENCES pl_chofer(id),
  INDEX idx_ing_tiempos (fecha_ingreso, descarga_inicio_at, descarga_fin_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 5) Vínculo de lo recibido a pallets/movimientos (cotejo en cuarentena)
-- =====================================================================

/*
 * Cada registro refleja que parte de la línea del PL (pl_packinglist_item)
 * se recibió en un pallet específico (wh_pallet) y (opcionalmente) en un
 * movimiento particular (wh_move 'IN' a CUARENTENA). Permite pallets mixtos.
 */
DROP TABLE IF EXISTS pl_rcv_link;
CREATE TABLE pl_rcv_link (
  id                 BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  packinglist_id     BIGINT UNSIGNED NOT NULL,
  pl_item_id         BIGINT UNSIGNED NOT NULL,
  producto_id        BIGINT UNSIGNED NOT NULL,
  lote_id            BIGINT UNSIGNED NULL,         -- si se conoce el lote al recibir
  pallet_id          BIGINT UNSIGNED NOT NULL,     -- pallet físico (wh_pallet.id)
  move_id            BIGINT UNSIGNED NULL,         -- wh_move.id (IN a CUARENTENA)
  uv_cajas           INT UNSIGNED NOT NULL DEFAULT 0,
  uc_unidades        INT UNSIGNED NOT NULL DEFAULT 0,
  created_at         DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_rcv_pl       FOREIGN KEY (packinglist_id) REFERENCES pl_packinglist(id) ON DELETE CASCADE,
  CONSTRAINT fk_rcv_plitem   FOREIGN KEY (pl_item_id) REFERENCES pl_packinglist_item(id) ON DELETE CASCADE,
  CONSTRAINT fk_rcv_prod     FOREIGN KEY (producto_id) REFERENCES para_productos(id),
  CONSTRAINT fk_rcv_lote     FOREIGN KEY (lote_id) REFERENCES wh_lote(id),
  CONSTRAINT fk_rcv_pallet   FOREIGN KEY (pallet_id) REFERENCES wh_pallet(id),
  CONSTRAINT fk_rcv_move     FOREIGN KEY (move_id) REFERENCES wh_move(id),
  INDEX idx_rcv_pl (packinglist_id, pl_item_id),
  INDEX idx_rcv_pallet (pallet_id),
  INDEX idx_rcv_prod (producto_id, lote_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 6) Vistas de conciliación
-- =====================================================================

DROP VIEW IF EXISTS v_pl_item_resumen;
CREATE VIEW v_pl_item_resumen AS
SELECT
  i.id                 AS pl_item_id,
  i.packinglist_id,
  pl.codigo            AS packinglist_codigo,
  i.sku_cliente,
  i.producto_id,
  i.descripcion,
  i.lote_codigo,
  i.expected_uv,
  i.expected_uc,
  i.received_uv,
  i.received_uc,
  i.diff_uv,
  i.diff_uc,
  i.checked_item,
  SUM(COALESCE(l.uv_cajas,0))   AS rcv_agg_uv,
  SUM(COALESCE(l.uc_unidades,0))AS rcv_agg_uc
FROM pl_packinglist_item i
JOIN pl_packinglist pl ON pl.id = i.packinglist_id
LEFT JOIN pl_rcv_link l ON l.pl_item_id = i.id
GROUP BY
  i.id, i.packinglist_id, pl.codigo, i.sku_cliente, i.producto_id, i.descripcion,
  i.lote_codigo, i.expected_uv, i.expected_uc, i.received_uv, i.received_uc,
  i.diff_uv, i.diff_uc, i.checked_item;

DROP VIEW IF EXISTS v_pl_resumen;
CREATE VIEW v_pl_resumen AS
SELECT
  pl.id                    AS packinglist_id,
  pl.codigo                AS packinglist_codigo,
  pl.cliente_ref,
  pl.fecha,
  pl.estado,
  pl.checked_all,
  COUNT(i.id)              AS items_total,
  SUM(i.expected_uv)       AS exp_uv_total,
  SUM(i.expected_uc)       AS exp_uc_total,
  SUM(i.received_uv)       AS rcv_uv_total,
  SUM(i.received_uc)       AS rcv_uc_total,
  SUM(i.expected_uv - i.received_uv) AS diff_uv_total,
  SUM(i.expected_uc - i.received_uc) AS diff_uc_total,
  SUM(CASE WHEN i.checked_item=1 THEN 1 ELSE 0 END) AS items_checked
FROM pl_packinglist pl
LEFT JOIN pl_packinglist_item i ON i.packinglist_id = pl.id
GROUP BY pl.id, pl.codigo, pl.cliente_ref, pl.fecha, pl.estado, pl.checked_all;

-- =====================================================================
-- 7) Triggers: mantener received_uv/uc al registrar pl_rcv_link
-- =====================================================================

/*
 * NOTA: phpMyAdmin suele manejar el delimitador automáticamente al ejecutar
 * este archivo completo; si da error por delimitador, ejecuta estos triggers
 * aparte cambiando el "Delimitador" de la UI a $$.
 */

DROP TRIGGER IF EXISTS trg_pl_rcv_link_ai;
DELIMITER $$
CREATE TRIGGER trg_pl_rcv_link_ai
AFTER INSERT ON pl_rcv_link
FOR EACH ROW
BEGIN
  UPDATE pl_packinglist_item i
     SET i.received_uv = (
           SELECT COALESCE(SUM(uv_cajas),0)
             FROM pl_rcv_link l
            WHERE l.pl_item_id = NEW.pl_item_id
         ),
         i.received_uc = (
           SELECT COALESCE(SUM(uc_unidades),0)
             FROM pl_rcv_link l
            WHERE l.pl_item_id = NEW.pl_item_id
         )
   WHERE i.id = NEW.pl_item_id;
END $$
DELIMITER ;

DROP TRIGGER IF EXISTS trg_pl_rcv_link_ad;
DELIMITER $$
CREATE TRIGGER trg_pl_rcv_link_ad
AFTER DELETE ON pl_rcv_link
FOR EACH ROW
BEGIN
  UPDATE pl_packinglist_item i
     SET i.received_uv = (
           SELECT COALESCE(SUM(uv_cajas),0)
             FROM pl_rcv_link l
            WHERE l.pl_item_id = OLD.pl_item_id
         ),
         i.received_uc = (
           SELECT COALESCE(SUM(uc_unidades),0)
             FROM pl_rcv_link l
            WHERE l.pl_item_id = OLD.pl_item_id
         )
   WHERE i.id = OLD.pl_item_id;
END $$
DELIMITER ;

DROP TRIGGER IF EXISTS trg_pl_rcv_link_au;
DELIMITER $$
CREATE TRIGGER trg_pl_rcv_link_au
AFTER UPDATE ON pl_rcv_link
FOR EACH ROW
BEGIN
  -- Recalcular para el ítem afectado (puede cambiar pl_item_id en un UPDATE)
  UPDATE pl_packinglist_item i
     SET i.received_uv = (
           SELECT COALESCE(SUM(uv_cajas),0)
             FROM pl_rcv_link l
            WHERE l.pl_item_id = NEW.pl_item_id
         ),
         i.received_uc = (
           SELECT COALESCE(SUM(uc_unidades),0)
             FROM pl_rcv_link l
            WHERE l.pl_item_id = NEW.pl_item_id
         )
   WHERE i.id = NEW.pl_item_id;

  IF OLD.pl_item_id <> NEW.pl_item_id THEN
    UPDATE pl_packinglist_item i
       SET i.received_uv = (
             SELECT COALESCE(SUM(uv_cajas),0)
               FROM pl_rcv_link l
              WHERE l.pl_item_id = OLD.pl_item_id
           ),
           i.received_uc = (
             SELECT COALESCE(SUM(uc_unidades),0)
               FROM pl_rcv_link l
              WHERE l.pl_item_id = OLD.pl_item_id
           )
     WHERE i.id = OLD.pl_item_id;
  END IF;
END $$
DELIMITER ;

SET FOREIGN_KEY_CHECKS = 1;
