/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 5: Reposición y Salida (procedimientos)
 * Archivo: database/step5_wh_sp_move_out.sql
 *
 * Requisitos:
 * - Paso 2 (tablas wh_*)
 * - Paso 3b (trigger fix #1442: trg_wh_move_ai → sp_wh_move_apply)
 *
 * Procedimientos:
 * 1) sp_wh_reposicion(p_deposito_code, p_pallet_codigo, p_to_pos_code, p_motivo?, p_estado_dest?)
 * 2) sp_wh_salida_total(p_deposito_code, p_pallet_codigo, p_referencia?)
 * 3) sp_wh_salida_parcial(p_deposito_code, p_pallet_codigo, p_producto_id, p_lote_codigo, p_delta_uv, p_delta_uc, p_referencia?)
 *
 * Notas:
 * - p_motivo/p_referencia y p_estado_dest pueden ser NULL (se aplican defaults).
 * - Validamos existencia de depósito, pallet y posiciones.
 * - Capacidad: no permite mover si la posición destino superaría capacidad_pallets.
 * - Estados: reposición → 'POS_ENTERO' por defecto (o 'POS_PICKEADO' si el pallet ya está pickeado);
 *            salida total → 'EMBARCADO';
 *            salida parcial → 'POS_PICKEADO'.
 * ===================================================================== */

SET NAMES utf8mb4 $$

/* Idempotencia */
DROP PROCEDURE IF EXISTS sp_wh_reposicion $$
DROP PROCEDURE IF EXISTS sp_wh_salida_total $$
DROP PROCEDURE IF EXISTS sp_wh_salida_parcial $$

/* ---------------------------------------------------------------------
   1) Reposición: mover pallet a otra posición (MOVE)
--------------------------------------------------------------------- */
CREATE PROCEDURE sp_wh_reposicion(
  IN p_deposito_code   VARCHAR(32),
  IN p_pallet_codigo   VARCHAR(64),
  IN p_to_pos_code     VARCHAR(64),    -- code / code_full / pos_code / pos_code_full
  IN p_motivo          VARCHAR(64),    -- NULL => 'UBICACION'
  IN p_estado_dest     VARCHAR(32)     -- NULL => 'POS_ENTERO' o 'POS_PICKEADO' si pallet ya está pickeado
)
BEGIN
  /* Declaraciones */
  DECLARE v_dep_id INT UNSIGNED;
  DECLARE v_pal_id BIGINT UNSIGNED;
  DECLARE v_from_pos BIGINT UNSIGNED;
  DECLARE v_to_pos BIGINT UNSIGNED;
  DECLARE v_move_id BIGINT UNSIGNED;
  DECLARE v_motivo VARCHAR(64);
  DECLARE v_estado_dest VARCHAR(32);
  DECLARE v_estado_dest_id INT UNSIGNED;
  DECLARE v_est_emb INT UNSIGNED;
  DECLARE v_pickeado TINYINT;
  DECLARE v_cap SMALLINT UNSIGNED;
  DECLARE v_ocup INT;

  /* Manejo de errores */
  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  IF p_deposito_code IS NULL OR p_deposito_code='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_deposito_code es requerido.';
  END IF;
  IF p_pallet_codigo IS NULL OR p_pallet_codigo='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_pallet_codigo es requerido.';
  END IF;
  IF p_to_pos_code IS NULL OR p_to_pos_code='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_to_pos_code es requerido.';
  END IF;

  SET v_motivo = IFNULL(p_motivo,'UBICACION');

  START TRANSACTION;

  /* Resolver depósito y pallet */
  SELECT id INTO v_dep_id FROM wh_deposito WHERE code=p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Depósito no encontrado.'; END IF;

  SELECT id, posicion_id, pickeado
    INTO v_pal_id, v_from_pos, v_pickeado
  FROM wh_pallet
  WHERE codigo = p_pallet_codigo AND deposito_id = v_dep_id
  LIMIT 1;

  IF v_pal_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Pallet no encontrado en el depósito.';
  END IF;

  /* Resolver posición destino dentro del mismo depósito */
  SELECT p.id, p.capacidad_pallets
    INTO v_to_pos, v_cap
  FROM wh_posicion p
  WHERE p.deposito_id = v_dep_id
    AND (p.code = p_to_pos_code OR p.code_full = p_to_pos_code
      OR p.pos_code = p_to_pos_code OR p.pos_code_full = p_to_pos_code)
  LIMIT 1;

  IF v_to_pos IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Posición destino no existe en el depósito.';
  END IF;

  IF v_from_pos <=> v_to_pos THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='La posición destino es igual a la actual.';
  END IF;

  /* Capacidad: pallets no embarcados en destino */
  SELECT id INTO v_est_emb FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1;

  SELECT COUNT(*) INTO v_ocup
  FROM wh_pallet pal
  WHERE pal.posicion_id = v_to_pos
    AND pal.deleted_at IS NULL
    AND (v_est_emb IS NULL OR pal.estado_id <> v_est_emb);

  IF v_cap IS NOT NULL AND v_ocup >= v_cap THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Capacidad de la posición destino completa.';
  END IF;

  /* Estado destino por defecto: POS_ENTERO o POS_PICKEADO si el pallet ya está pickeado */
  SET v_estado_dest = IFNULL(p_estado_dest, IF(v_pickeado=1,'POS_PICKEADO','POS_ENTERO'));
  SELECT id INTO v_estado_dest_id FROM wh_pallet_estado WHERE code=v_estado_dest LIMIT 1;
  IF v_estado_dest_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Estado destino inválido.';
  END IF;

  /* Insertar movimiento MOVE (deltas 0 => pallet completo) */
  INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
  VALUES (v_dep_id, 'MOVE', v_motivo, v_pal_id, v_from_pos, v_to_pos, 0, 0, CONCAT('MOVE-', p_pallet_codigo));
  SET v_move_id = LAST_INSERT_ID();

  /* Actualizar estado del pallet (ubicado) */
  UPDATE wh_pallet SET estado_id = v_estado_dest_id WHERE id = v_pal_id;

  COMMIT;

  /* Resultado */
  SELECT
    v_move_id                 AS move_id,
    p_pallet_codigo           AS pallet_codigo,
    (SELECT code_full FROM wh_posicion WHERE id=v_from_pos) AS from_pos_code_full,
    (SELECT code_full FROM wh_posicion WHERE id=v_to_pos)   AS to_pos_code_full,
    v_estado_dest             AS estado_destino;
END $$


/* ---------------------------------------------------------------------
   2) Salida total: OUT de pallet completo (despacho)
--------------------------------------------------------------------- */
CREATE PROCEDURE sp_wh_salida_total(
  IN p_deposito_code VARCHAR(32),
  IN p_pallet_codigo VARCHAR(64),
  IN p_referencia    VARCHAR(64)   -- NULL => 'DESPACHO'
)
BEGIN
  DECLARE v_dep_id INT UNSIGNED;
  DECLARE v_pal_id BIGINT UNSIGNED;
  DECLARE v_from_pos BIGINT UNSIGNED;
  DECLARE v_move_id BIGINT UNSIGNED;
  DECLARE v_ref VARCHAR(64);
  DECLARE v_est_emb INT UNSIGNED;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK; RESIGNAL;
  END;

  IF p_deposito_code IS NULL OR p_deposito_code='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_deposito_code es requerido.';
  END IF;
  IF p_pallet_codigo IS NULL OR p_pallet_codigo='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_pallet_codigo es requerido.';
  END IF;

  SET v_ref = IFNULL(p_referencia,'DESPACHO');

  START TRANSACTION;

  SELECT id INTO v_dep_id FROM wh_deposito WHERE code=p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Depósito no encontrado.'; END IF;

  SELECT id, posicion_id INTO v_pal_id, v_from_pos
  FROM wh_pallet
  WHERE codigo=p_pallet_codigo AND deposito_id=v_dep_id
  LIMIT 1;
  IF v_pal_id IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Pallet no encontrado.'; END IF;
  IF v_from_pos IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='El pallet no está posicionado (no puede salir).'; END IF;

  /* OUT de pallet completo (deltas 0) */
  INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
  VALUES (v_dep_id, 'OUT', v_ref, v_pal_id, v_from_pos, NULL, 0, 0, CONCAT('OUT-', p_pallet_codigo));
  SET v_move_id = LAST_INSERT_ID();

  /* Estado EMBARCADO */
  SELECT id INTO v_est_emb FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1;
  IF v_est_emb IS NOT NULL THEN
    UPDATE wh_pallet SET estado_id = v_est_emb WHERE id = v_pal_id;
  END IF;

  COMMIT;

  SELECT v_move_id AS move_id, p_pallet_codigo AS pallet_codigo, v_ref AS motivo;
END $$


/* ---------------------------------------------------------------------
   3) Salida parcial: OUT por SKU/lote y cantidades (UV/UC)
--------------------------------------------------------------------- */
CREATE PROCEDURE sp_wh_salida_parcial(
  IN p_deposito_code VARCHAR(32),
  IN p_pallet_codigo VARCHAR(64),
  IN p_producto_id   BIGINT UNSIGNED,
  IN p_lote_codigo   VARCHAR(64),
  IN p_delta_uv      INT,           -- cajas a extraer (>=0)
  IN p_delta_uc      INT,           -- unidades sueltas a extraer (>=0)
  IN p_referencia    VARCHAR(64)    -- NULL => 'PICKING'
)
BEGIN
  DECLARE v_dep_id INT UNSIGNED;
  DECLARE v_pal_id BIGINT UNSIGNED;
  DECLARE v_from_pos BIGINT UNSIGNED;
  DECLARE v_lote_id BIGINT UNSIGNED;
  DECLARE v_move_id BIGINT UNSIGNED;
  DECLARE v_ref VARCHAR(64);
  DECLARE v_est_pick INT UNSIGNED;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK; RESIGNAL;
  END;

  IF p_deposito_code IS NULL OR p_deposito_code='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_deposito_code es requerido.';
  END IF;
  IF p_pallet_codigo IS NULL OR p_pallet_codigo='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_pallet_codigo es requerido.';
  END IF;
  IF p_producto_id IS NULL OR p_producto_id=0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_producto_id es requerido.';
  END IF;
  IF p_lote_codigo IS NULL OR p_lote_codigo='' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_lote_codigo es requerido.';
  END IF;
  IF (COALESCE(p_delta_uv,0)=0 AND COALESCE(p_delta_uc,0)=0) OR p_delta_uv<0 OR p_delta_uc<0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Debe indicar cantidades no negativas y al menos una > 0.';
  END IF;

  SET v_ref = IFNULL(p_referencia,'PICKING');

  START TRANSACTION;

  SELECT id INTO v_dep_id FROM wh_deposito WHERE code=p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Depósito no encontrado.'; END IF;

  SELECT id, posicion_id INTO v_pal_id, v_from_pos
  FROM wh_pallet
  WHERE codigo=p_pallet_codigo AND deposito_id=v_dep_id
  LIMIT 1;
  IF v_pal_id IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Pallet no encontrado.'; END IF;
  IF v_from_pos IS NULL THEN SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='El pallet no está posicionado (no puede salir).'; END IF;

  /* Resolver lote */
  SELECT id INTO v_lote_id
  FROM wh_lote
  WHERE producto_id = p_producto_id AND codigo = p_lote_codigo
  LIMIT 1;
  IF v_lote_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Lote no encontrado para el producto.';
  END IF;

  /* OUT parcial (deltas positivos → trigger descontará) */
  INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
  VALUES (v_dep_id, 'OUT', v_ref, v_pal_id, p_producto_id, v_lote_id, v_from_pos, NULL, p_delta_uv, p_delta_uc, CONCAT('PICK-', p_pallet_codigo));
  SET v_move_id = LAST_INSERT_ID();

  /* Estado pickeado */
  SELECT id INTO v_est_pick FROM wh_pallet_estado WHERE code='POS_PICKEADO' LIMIT 1;
  IF v_est_pick IS NOT NULL THEN
    UPDATE wh_pallet SET estado_id = v_est_pick, pickeado = 1 WHERE id = v_pal_id;
  ELSE
    UPDATE wh_pallet SET pickeado = 1 WHERE id = v_pal_id;
  END IF;

  COMMIT;

  SELECT v_move_id AS move_id, p_pallet_codigo AS pallet_codigo, v_ref AS motivo, p_delta_uv AS uv_cajas, p_delta_uc AS uc_unidades;
END $$
