/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 6: SP de importación de Packing List (desde staging Excel)
 * Archivo: database/step6_pl_import_sp.sql
 *
 * Procedimiento principal:
 *   sp_pl_import_packinglist_from_batch(
 *     IN  p_batch_id           BIGINT,
 *     IN  p_cliente_ref        VARCHAR(64),
 *     IN  p_packinglist_codigo VARCHAR(64),
 *     IN  p_fecha              DATE,
 *     IN  p_overwrite_if_exists TINYINT,   -- 0:error si existe; 1:limpia y reimporta
 *     OUT o_packinglist_id     BIGINT
 *   )
 *
 * Comportamiento:
 * - Lee filas de pl_import_row (batch_id = p_batch_id), columna RAW (JSON).
 * - Para cada fila obtiene: sku_cliente, descripcion, expected_uv/uc,
 *   lote_codigo?, fecha_produccion?, fecha_vencimiento?.
 * - Si no existe alias (pl_producto_alias) para (cliente_ref, sku_cliente),
 *   crea producto en para_productos (columna 'denominacion') y guarda alias.
 * - Crea/actualiza pl_packinglist + pl_packinglist_item (agrega cantidades si
 *   el ítem ya existe por (sku_cliente, producto_id, lote_codigo)).
 * - Completa punteros en pl_import_row (packinglist_id, item_id) y status.
 *
 * Supuestos mínimos:
 * - para_productos(id PK, denominacion VARCHAR).
 * - pl_import_row.raw almacena un JSON por fila del Excel con llaves estándar:
 *   {
 *     "sku_cliente": "SKU-ACME-001",
 *     "descripcion": "Producto A",
 *     "expected_uv": 10,
 *     "expected_uc": 0,
 *     "lote_codigo": "L-001",                (opcional)
 *     "fecha_produccion": "2025-09-01",      (opcional; admite "dd/mm/yyyy")
 *     "fecha_vencimiento": "2026-09-01"      (opcional; admite "dd/mm/yyyy")
 *   }
 * - El procedimiento es tolerante y probará llaves alternativas comunes:
 *   sku / SKU, description / nombre, uv / cajas, uc / unidades, etc.
 * ===================================================================== */

/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 6 (v2): SP de importación de Packing List (desde staging Excel)
 * Archivo: database/step6_pl_import_sp_v2.sql
 *
 * Corrige: mueve TODOS los DECLARE (variables, cursor y handlers)
 * al comienzo del bloque BEGIN, como exige MySQL.
 * ===================================================================== */

SET NAMES utf8mb4 $$

DROP PROCEDURE IF EXISTS sp_pl_import_packinglist_from_batch $$

CREATE PROCEDURE sp_pl_import_packinglist_from_batch(
  IN  p_batch_id            BIGINT,
  IN  p_cliente_ref         VARCHAR(64),
  IN  p_cliente_id          INT DEFAULT NULL,
  IN  p_packinglist_codigo  VARCHAR(64),
  IN  p_fecha               DATE,
  IN  p_overwrite_if_exists TINYINT,
  OUT o_packinglist_id      BIGINT
)
BEGIN
  /* ---------- DECLARACIONES (deben ir primero) ---------- */
  -- Variables generales
  DECLARE v_pl_id      BIGINT UNSIGNED DEFAULT NULL;
  DECLARE v_row_id     BIGINT UNSIGNED;
  DECLARE v_raw        JSON;

  -- Variables de fila
  DECLARE v_sku        VARCHAR(64);
  DECLARE v_desc       VARCHAR(255);
  DECLARE v_exp_uv     INT UNSIGNED;
  DECLARE v_exp_uc     INT UNSIGNED;
  DECLARE v_lote       VARCHAR(64);
  DECLARE v_fprod_str  VARCHAR(20);
  DECLARE v_fvenc_str  VARCHAR(20);
  DECLARE v_fprod      DATE;
  DECLARE v_fvenc      DATE;

  DECLARE v_prod_id    BIGINT UNSIGNED;
  DECLARE v_item_id    BIGINT UNSIGNED;

  -- Cliente helpers
  DECLARE v_cliente_ref VARCHAR(64);
  DECLARE v_cli_razon   VARCHAR(255);
  DECLARE v_cli_ruc     VARCHAR(64);

  DECLARE v_has_error  INT DEFAULT 0;
  DECLARE done         INT DEFAULT 0;

  -- Cursor sobre filas del batch
  DECLARE cur CURSOR FOR
    SELECT id, raw
      FROM pl_import_row
     WHERE batch_id = p_batch_id
     ORDER BY rownum;

  -- Handler para fin de cursor
  DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = 1;

  -- Handler de errores: marca fila como ERROR y continúa
  DECLARE CONTINUE HANDLER FOR SQLEXCEPTION
  BEGIN
    SET v_has_error = 1;
    IF v_row_id IS NOT NULL THEN
      UPDATE pl_import_row
         SET status = 'ERROR',
             error_msg = 'Error procesando fila (verificar datos).'
       WHERE id = v_row_id;
    END IF;
    SET v_row_id = NULL; -- limpiar contexto para continuar
  END;

  /* ---------- Validaciones iniciales ---------- */
  IF p_batch_id IS NULL OR p_batch_id = 0 THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_batch_id es requerido.';
  END IF;
  IF p_packinglist_codigo IS NULL OR p_packinglist_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_packinglist_codigo es requerido.';
  END IF;
  -- Resolver cliente_ref: si no viene p_cliente_ref, intentar usar p_cliente_id para construir uno
  SET v_cliente_ref = p_cliente_ref;
  IF v_cliente_ref IS NULL OR v_cliente_ref = '' THEN
    IF p_cliente_id IS NULL OR p_cliente_id = 0 THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_cliente_ref o p_cliente_id es requerido.';
    ELSE
      -- Intentar obtener datos del cliente
      SELECT razon_social, ruc INTO v_cli_razon, v_cli_ruc
        FROM para_clientes
       WHERE id = p_cliente_id
       LIMIT 1;
      IF v_cli_ruc IS NOT NULL AND v_cli_ruc <> '' THEN
        SET v_cliente_ref = v_cli_ruc;
      ELSE
        SET v_cliente_ref = CONCAT('CLI-', LEFT(REPLACE(UPPER(COALESCE(v_cli_razon,'')), ' ', ''), 20));
      END IF;
    END IF;
  END IF;

  /* ---------- Construir/limpiar Packing List ---------- */
  SELECT id INTO v_pl_id
    FROM pl_packinglist
   WHERE codigo = p_packinglist_codigo
   LIMIT 1;

  IF v_pl_id IS NOT NULL THEN
    IF p_overwrite_if_exists = 1 THEN
      DELETE FROM pl_packinglist_item WHERE packinglist_id = v_pl_id;
      UPDATE pl_packinglist
         SET cliente_ref = v_cliente_ref,
             cliente_id  = p_cliente_id,
             fecha       = p_fecha,
             estado      = 'IMPORTADO',
             import_batch_id = p_batch_id,
             updated_at  = NOW()
       WHERE id = v_pl_id;
    ELSE
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'El Packing List ya existe. Use p_overwrite_if_exists=1 para reimportar.';
    END IF;
  ELSE
    INSERT INTO pl_packinglist (codigo, cliente_ref, cliente_id, fecha, estado, import_batch_id)
    VALUES (p_packinglist_codigo, v_cliente_ref, p_cliente_id, p_fecha, 'IMPORTADO', p_batch_id);
    SET v_pl_id = LAST_INSERT_ID();
  END IF;

  /* Vincular conteos al batch */
  UPDATE pl_import_batch
     SET rows_total = (SELECT COUNT(*) FROM pl_import_row WHERE batch_id=p_batch_id)
   WHERE id = p_batch_id;

  /* ---------- Bucle de filas ---------- */
  OPEN cur;
  read_loop: LOOP
    FETCH cur INTO v_row_id, v_raw;
    IF done = 1 THEN LEAVE read_loop; END IF;

    SET v_has_error = 0;

    /* --------- Extracción tolerante de campos --------- */
    -- SKU (obligatorio)
    SET v_sku =
      COALESCE(
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.sku_cliente')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.sku')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.SKU'))
      );
    IF v_sku IS NULL OR v_sku = '' THEN
      UPDATE pl_import_row SET status='ERROR', error_msg='Falta sku_cliente' WHERE id=v_row_id;
      ITERATE read_loop;
    END IF;

    -- Descripción (opcional)
    SET v_desc =
      COALESCE(
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.descripcion')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.description')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.nombre')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.product_name'))
      );

    -- Cantidades esperadas (por defecto 0)
    SET v_exp_uv =
      COALESCE(
        CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.expected_uv')) AS UNSIGNED),
        CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.uv')) AS UNSIGNED),
        CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.cajas')) AS UNSIGNED),
        0
      );
    SET v_exp_uc =
      COALESCE(
        CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.expected_uc')) AS UNSIGNED),
        CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.uc')) AS UNSIGNED),
        CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.unidades')) AS UNSIGNED),
        0
      );

    -- Lote y fechas (opcionales, soporta YYYY-MM-DD o DD/MM/YYYY)
    SET v_lote =
      COALESCE(
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.lote_codigo')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.lote'))
      );
    SET v_fprod_str =
      COALESCE(
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.fecha_produccion')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.fprod'))
      );
    SET v_fvenc_str =
      COALESCE(
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.fecha_vencimiento')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.fvenc')),
        JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.vencimiento'))
      );
    SET v_fprod = COALESCE(STR_TO_DATE(v_fprod_str, '%Y-%m-%d'), STR_TO_DATE(v_fprod_str, '%d/%m/%Y'));
    SET v_fvenc = COALESCE(STR_TO_DATE(v_fvenc_str, '%Y-%m-%d'), STR_TO_DATE(v_fvenc_str, '%d/%m/%Y'));

    /* --------- Resolver/crear producto + alias --------- */
    SET v_prod_id = NULL;

    SELECT producto_id INTO v_prod_id
      FROM pl_producto_alias
     WHERE cliente_ref = v_cliente_ref
       AND sku_cliente = v_sku
     LIMIT 1;

    IF v_prod_id IS NULL THEN
      -- ¿producto_id en JSON?
      SET v_prod_id = CAST(JSON_UNQUOTE(JSON_EXTRACT(v_raw,'$.producto_id')) AS UNSIGNED);

      IF v_prod_id IS NULL OR v_prod_id = 0 THEN
        -- Crear producto mínimo en para_productos (denominacion)
        INSERT INTO para_productos (denominacion)
        VALUES (COALESCE(v_desc, CONCAT('SKU ', v_sku)));
        SET v_prod_id = LAST_INSERT_ID();
      END IF;

      -- Crear/actualizar alias
      INSERT INTO pl_producto_alias (cliente_ref, sku_cliente, producto_id, producto_desc)
      VALUES (v_cliente_ref, v_sku, v_prod_id, v_desc)
      ON DUPLICATE KEY UPDATE
        producto_id  = VALUES(producto_id),
        producto_desc= VALUES(producto_desc);
    END IF;

    /* --------- Upsert de lote (si vino) --------- */
    IF v_lote IS NOT NULL AND v_lote <> '' THEN
      INSERT INTO wh_lote (producto_id, codigo, fecha_produccion, fecha_vencimiento)
      VALUES (v_prod_id, v_lote, v_fprod, v_fvenc)
      ON DUPLICATE KEY UPDATE
        fecha_produccion  = COALESCE(wh_lote.fecha_produccion, VALUES(fecha_produccion)),
        fecha_vencimiento = COALESCE(wh_lote.fecha_vencimiento, VALUES(fecha_vencimiento));
    END IF;

    /* --------- Upsert línea del PL --------- */
    SELECT id INTO v_item_id
      FROM pl_packinglist_item
     WHERE packinglist_id = v_pl_id
       AND sku_cliente    = v_sku
       AND (producto_id   <=> v_prod_id)
       AND (lote_codigo   <=> v_lote)
     LIMIT 1;

    IF v_item_id IS NULL THEN
      INSERT INTO pl_packinglist_item (
        packinglist_id, sku_cliente, producto_id, descripcion,
        lote_codigo, fecha_produccion, fecha_vencimiento,
        expected_uv, expected_uc
      ) VALUES (
        v_pl_id, v_sku, v_prod_id, v_desc,
        v_lote, v_fprod, v_fvenc,
        COALESCE(v_exp_uv,0), COALESCE(v_exp_uc,0)
      );
      SET v_item_id = LAST_INSERT_ID();
    ELSE
      UPDATE pl_packinglist_item
         SET descripcion       = COALESCE(pl_packinglist_item.descripcion, v_desc),
             fecha_produccion  = COALESCE(pl_packinglist_item.fecha_produccion, v_fprod),
             fecha_vencimiento = COALESCE(pl_packinglist_item.fecha_vencimiento, v_fvenc),
             expected_uv       = COALESCE(expected_uv,0) + COALESCE(v_exp_uv,0),
             expected_uc       = COALESCE(expected_uc,0) + COALESCE(v_exp_uc,0)
       WHERE id = v_item_id;
    END IF;

    /* --------- Marcar fila del batch --------- */
    UPDATE pl_import_row
       SET packinglist_id = v_pl_id,
           item_id        = v_item_id,
           status         = 'OK',
           error_msg      = NULL
     WHERE id = v_row_id;

    -- limpiar contexto
    SET v_row_id = NULL;

  END LOOP;
  CLOSE cur;

  /* ---------- Ajustes finales ---------- */
  UPDATE pl_import_batch
     SET rows_ok    = (SELECT COUNT(*) FROM pl_import_row WHERE batch_id=p_batch_id AND status='OK'),
         rows_error = (SELECT COUNT(*) FROM pl_import_row WHERE batch_id=p_batch_id AND status='ERROR')
   WHERE id = p_batch_id;

  UPDATE pl_packinglist
     SET estado = 'IMPORTADO'
   WHERE id = v_pl_id;

  SET o_packinglist_id = v_pl_id;
END $$
