/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 7 (v4): Fix #1163 - TEMP tables con TEXT → ENGINE=InnoDB
 * Archivo: database/step7_pl_receive_sp_v4.sql
 * 
 * Cambios:
 * - Todas las tablas TEMPORARY del SP usan ENGINE=InnoDB
 *   (en especial _plan_pallets que tiene LONGTEXT).
 * ===================================================================== */

SET NAMES utf8mb4 $$

DROP PROCEDURE IF EXISTS sp_pl_recibir_desde_pl $$

CREATE PROCEDURE sp_pl_recibir_desde_pl(
  IN p_packinglist_codigo VARCHAR(64),
  IN p_deposito_code      VARCHAR(32),
  IN p_pos_code           VARCHAR(64),
  IN p_plan_json          JSON,
  IN p_check_items        TINYINT,
  IN p_check_all_if_full  TINYINT
)
BEGIN
  /* ----------------- DECLARACIONES ----------------- */
  -- Generales
  DECLARE v_pl_id        BIGINT UNSIGNED;
  DECLARE v_dep_id       INT UNSIGNED;
  DECLARE v_pl_code      VARCHAR(64);
  DECLARE v_seq          INT DEFAULT 0;

  DECLARE v_pal_code     VARCHAR(64);
  DECLARE v_pal_id       BIGINT UNSIGNED;
  DECLARE v_move_id      BIGINT UNSIGNED;
  DECLARE v_items_json   JSON;

  DECLARE v_done         INT DEFAULT 0;
  DECLARE v_idx          INT;
  DECLARE v_pos_each     VARCHAR(64);

  -- Modo AUTO (cursor)
  DECLARE v_pl_item_id   BIGINT UNSIGNED;
  DECLARE v_prod_id      BIGINT UNSIGNED;
  DECLARE v_lote_code    VARCHAR(64);
  DECLARE v_rem_uv       INT;
  DECLARE v_rem_uc       INT;
  DECLARE v_lote_id      BIGINT UNSIGNED;

  -- Cursor AUTO
  DECLARE cur_auto CURSOR FOR
    SELECT i.id AS pl_item_id, i.producto_id, i.lote_codigo,
           GREATEST(i.expected_uv - i.received_uv,0) AS rem_uv,
           GREATEST(i.expected_uc - i.received_uc,0) AS rem_uc
      FROM pl_packinglist_item i
     WHERE i.packinglist_id = v_pl_id
       AND (GREATEST(i.expected_uv - i.received_uv,0) > 0
            OR GREATEST(i.expected_uc - i.received_uc,0) > 0)
     ORDER BY i.id;

  -- Handlers
  DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done = 1;
  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  /* ----------------- Validaciones ----------------- */
  IF p_packinglist_codigo IS NULL OR p_packinglist_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_packinglist_codigo es requerido.';
  END IF;
  IF p_deposito_code IS NULL OR p_deposito_code = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_deposito_code es requerido.';
  END IF;

  /* Resolver PL y Depósito */
  SELECT id, codigo INTO v_pl_id, v_pl_code
    FROM pl_packinglist WHERE codigo = p_packinglist_codigo LIMIT 1;
  IF v_pl_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Packing List no encontrado.';
  END IF;

  SELECT id INTO v_dep_id FROM wh_deposito WHERE code = p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Depósito no encontrado.';
  END IF;

  /* Pre-estado: en descarga */
  UPDATE pl_packinglist
     SET estado = 'EN_DESCARGA'
   WHERE id = v_pl_id;

  /* Salida (resumen por pallet) */
  DROP TEMPORARY TABLE IF EXISTS _out_summary;
  CREATE TEMPORARY TABLE _out_summary (
    pallet_codigo VARCHAR(64),
    pallet_id     BIGINT UNSIGNED,
    move_id       BIGINT UNSIGNED,
    pos_code      VARCHAR(64),
    items_count   INT,
    uv_total      INT,
    uc_total      INT
  ) ENGINE=InnoDB;

  START TRANSACTION;

  /* ----------------- MODO A: AUTO ----------------- */
  IF p_plan_json IS NULL THEN
    SET v_done = 0;
    OPEN cur_auto;
    auto_loop: LOOP
      FETCH cur_auto INTO v_pl_item_id, v_prod_id, v_lote_code, v_rem_uv, v_rem_uc;
      IF v_done = 1 THEN LEAVE auto_loop; END IF;

      SET v_seq = v_seq + 1;
      SET v_pal_code = CONCAT('PAL-', v_pl_code, '-', LPAD(v_seq,3,'0'));

      SET v_items_json = JSON_ARRAY(
        JSON_OBJECT(
          'producto_id', v_prod_id,
          'lote_codigo', v_lote_code,
          'uv_cajas',    COALESCE(v_rem_uv,0),
          'uc_unidades', COALESCE(v_rem_uc,0)
        )
      );

      CALL sp_wh_ingreso_create(
        p_deposito_code,
        v_pal_code,
        p_pos_code,
        'CUARENTENA',
        v_items_json,
        0
      );

      SELECT id INTO v_pal_id FROM wh_pallet WHERE codigo = v_pal_code LIMIT 1;
      SELECT MAX(id) INTO v_move_id FROM wh_move WHERE pallet_id = v_pal_id AND tipo='IN';

      IF v_lote_code IS NOT NULL AND v_lote_code <> '' THEN
        SELECT id INTO v_lote_id FROM wh_lote WHERE producto_id=v_prod_id AND codigo=v_lote_code LIMIT 1;
      ELSE
        SET v_lote_id = NULL;
      END IF;

      INSERT INTO pl_rcv_link (packinglist_id, pl_item_id, producto_id, lote_id, pallet_id, move_id, uv_cajas, uc_unidades)
      VALUES (v_pl_id, v_pl_item_id, v_prod_id, v_lote_id, v_pal_id, v_move_id, COALESCE(v_rem_uv,0), COALESCE(v_rem_uc,0));

      INSERT INTO _out_summary (pallet_codigo, pallet_id, move_id, pos_code, items_count, uv_total, uc_total)
      VALUES (v_pal_code, v_pal_id, v_move_id, p_pos_code, 1, COALESCE(v_rem_uv,0), COALESCE(v_rem_uc,0));
    END LOOP;
    CLOSE cur_auto;

  /* ----------------- MODO B: PLAN JSON ----------------- */
  ELSE
    /* Pallets del plan → items_json como texto */
    DROP TEMPORARY TABLE IF EXISTS _plan_pallets;
    CREATE TEMPORARY TABLE _plan_pallets (
      idx           INT,
      pallet_codigo VARCHAR(64),
      pos_code      VARCHAR(64),
      items_json    LONGTEXT
    ) ENGINE=InnoDB;

    INSERT INTO _plan_pallets (idx, pallet_codigo, pos_code, items_json)
    SELECT
      jt.ord    AS idx,
      jt.pallet_codigo,
      jt.pos_code,
      jt.items
    FROM JSON_TABLE(
           CASE
             WHEN JSON_TYPE(p_plan_json) = 'OBJECT' THEN JSON_EXTRACT(p_plan_json, '$.pallets')
             ELSE p_plan_json
           END,
           '$[*]'
           COLUMNS(
             ord            FOR ORDINALITY,
             pallet_codigo  VARCHAR(64)   PATH '$.pallet_codigo',
             pos_code       VARCHAR(64)   PATH '$.pos_code' NULL ON EMPTY,
             items          VARCHAR(10000)PATH '$.items'    NULL ON EMPTY
           )
         ) AS jt;

    /* Expandir ítems planificados */
    DROP TEMPORARY TABLE IF EXISTS _plan_items;
    CREATE TEMPORARY TABLE _plan_items (
      idx           INT,
      pl_item_id    BIGINT UNSIGNED,
      sku_cliente   VARCHAR(64),
      producto_id   BIGINT UNSIGNED,
      lote_codigo   VARCHAR(64),
      uv            INT UNSIGNED,
      uc            INT UNSIGNED
    ) ENGINE=InnoDB;

    INSERT INTO _plan_items (idx, pl_item_id, sku_cliente, producto_id, lote_codigo, uv, uc)
    SELECT
      p.idx,
      it.pl_item_id,
      it.sku_cliente,
      it.producto_id,
      it.lote_codigo,
      COALESCE(CAST(it.uv AS UNSIGNED),0) AS uv,
      COALESCE(CAST(it.uc AS UNSIGNED),0) AS uc
    FROM _plan_pallets p,
         JSON_TABLE(
           p.items_json, '$[*]'
           COLUMNS(
             pl_item_id   BIGINT       PATH '$.pl_item_id'   NULL ON EMPTY,
             sku_cliente  VARCHAR(64)  PATH '$.sku_cliente'  NULL ON EMPTY,
             producto_id  BIGINT       PATH '$.producto_id'  NULL ON EMPTY,
             lote_codigo  VARCHAR(64)  PATH '$.lote_codigo'  NULL ON EMPTY,
             uv           VARCHAR(32)  PATH '$.uv'           NULL ON EMPTY,
             uc           VARCHAR(32)  PATH '$.uc'           NULL ON EMPTY
           )
         ) AS it;

    /* Completar refs contra el PL */
    UPDATE _plan_items pi
      JOIN pl_packinglist_item i
        ON i.packinglist_id = v_pl_id
       AND pi.pl_item_id IS NULL
       AND pi.sku_cliente = i.sku_cliente
     SET pi.pl_item_id = i.id,
         pi.producto_id = COALESCE(pi.producto_id, i.producto_id),
         pi.lote_codigo = COALESCE(pi.lote_codigo, i.lote_codigo);

    UPDATE _plan_items pi
      JOIN pl_packinglist_item i
        ON i.id = pi.pl_item_id
     SET pi.producto_id = COALESCE(pi.producto_id, i.producto_id),
         pi.lote_codigo = COALESCE(pi.lote_codigo, i.lote_codigo)
    WHERE i.packinglist_id = v_pl_id;

    IF EXISTS (SELECT 1 FROM _plan_items WHERE pl_item_id IS NULL) THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'Plan JSON: hay items sin pl_item_id resoluble (sku_cliente desconocido en el PL).';
    END IF;

    /* Pendientes actuales */
    DROP TEMPORARY TABLE IF EXISTS _remain;
    CREATE TEMPORARY TABLE _remain AS
    SELECT
      i.id AS pl_item_id,
      GREATEST(i.expected_uv - i.received_uv,0) AS rem_uv,
      GREATEST(i.expected_uc - i.received_uc,0) AS rem_uc,
      i.producto_id,
      i.lote_codigo
    FROM pl_packinglist_item i
    WHERE i.packinglist_id = v_pl_id;

    /* Validación: lo planificado no excede lo pendiente */
    DROP TEMPORARY TABLE IF EXISTS _planned_sum;
    CREATE TEMPORARY TABLE _planned_sum AS
    SELECT pl_item_id, SUM(uv) AS sum_uv, SUM(uc) AS sum_uc
      FROM _plan_items
     GROUP BY pl_item_id;

    IF EXISTS (
      SELECT 1
      FROM _planned_sum s
      JOIN _remain r ON r.pl_item_id = s.pl_item_id
      WHERE s.sum_uv > r.rem_uv OR s.sum_uc > r.rem_uc
    ) THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'Plan JSON excede cantidades pendientes para uno o más ítems.';
    END IF;

    /* Procesar cada pallet del plan */
    SET v_done = 0;
    BEGIN
      DECLARE cur_pal CURSOR FOR
        SELECT idx, pallet_codigo, COALESCE(pos_code, p_pos_code) AS pos_code_eff
          FROM _plan_pallets
         ORDER BY idx;
      DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done = 1;

      OPEN cur_pal;
      pal_loop: LOOP
        FETCH cur_pal INTO v_idx, v_pal_code, v_pos_each;
        IF v_done = 1 THEN LEAVE pal_loop; END IF;

        /* Construir JSON para sp_wh_ingreso_create */
        SELECT JSON_ARRAYAGG(
                 JSON_OBJECT(
                   'producto_id',  pi.producto_id,
                   'lote_codigo',  pi.lote_codigo,
                   'uv_cajas',     pi.uv,
                   'uc_unidades',  pi.uc
                 )
               )
          INTO v_items_json
        FROM _plan_items pi
        WHERE pi.idx = v_idx;

        IF v_items_json IS NULL THEN
          SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Plan JSON: pallet sin items.';
        END IF;

        CALL sp_wh_ingreso_create(
          p_deposito_code,
          v_pal_code,
          v_pos_each,
          'CUARENTENA',
          v_items_json,
          1
        );

        SELECT id INTO v_pal_id FROM wh_pallet WHERE codigo = v_pal_code LIMIT 1;
        SELECT MAX(id) INTO v_move_id FROM wh_move WHERE pallet_id = v_pal_id AND tipo='IN';

        INSERT INTO pl_rcv_link (packinglist_id, pl_item_id, producto_id, lote_id, pallet_id, move_id, uv_cajas, uc_unidades)
        SELECT
          v_pl_id,
          pi.pl_item_id,
          pi.producto_id,
          (SELECT id FROM wh_lote WHERE producto_id=pi.producto_id AND codigo=pi.lote_codigo LIMIT 1),
          v_pal_id,
          v_move_id,
          pi.uv,
          pi.uc
        FROM _plan_items pi
        WHERE pi.idx = v_idx;

        INSERT INTO _out_summary (pallet_codigo, pallet_id, move_id, pos_code, items_count, uv_total, uc_total)
        SELECT v_pal_code, v_pal_id, v_move_id, v_pos_each,
               COUNT(*), COALESCE(SUM(uv),0), COALESCE(SUM(uc),0)
          FROM _plan_items
         WHERE idx = v_idx;
      END LOOP;
      CLOSE cur_pal;
    END;
  END IF;

  /* ----------------- Post: checks y estado ----------------- */
  IF p_check_items = 1 THEN
    UPDATE pl_packinglist_item i
       SET i.checked_item = 1
     WHERE i.packinglist_id = v_pl_id
       AND COALESCE(i.expected_uv,0) = COALESCE(i.received_uv,0)
       AND COALESCE(i.expected_uc,0) = COALESCE(i.received_uc,0);
  END IF;

  IF NOT EXISTS (
       SELECT 1 FROM pl_packinglist_item
        WHERE packinglist_id = v_pl_id
          AND (GREATEST(expected_uv - received_uv,0) > 0 OR GREATEST(expected_uc - received_uc,0) > 0)
     ) THEN
    IF p_check_all_if_full = 1 THEN
      UPDATE pl_packinglist
         SET checked_all = 1,
             estado = 'VALIDADO'
       WHERE id = v_pl_id;
    ELSE
      UPDATE pl_packinglist
         SET estado = 'DESCARGADO'
       WHERE id = v_pl_id;
    END IF;
  ELSE
    UPDATE pl_packinglist
       SET estado = 'EN_DESCARGA'
     WHERE id = v_pl_id;
  END IF;

  COMMIT;

  /* Resultado */
  SELECT * FROM _out_summary ORDER BY pallet_codigo;
END $$
