/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 8: Cierre de Ingreso + KPIs de Descarga
 * Archivo: database/step8_pl_close_kpis.sql
 *
 * Incluye:
 * - SP sp_pl_cerrar_ingreso:
 *     Cierra la descarga del PL (fin de descarga, operarios) y actualiza
 *     el estado del PL: 'CERRADO' si no hay diferencias; si hay recibidos
 *     pero quedan diferencias → 'DESCARGADO' (no cierra).
 * - Vistas KPI:
 *     v_kpi_ingresos_descarga  → métricas por ingreso (tiempos, rendimientos)
 *     v_kpi_ingreso_detalle    → detalle por ítem del PL (esperado vs recibido)
 * ===================================================================== */

SET NAMES utf8mb4;

-- =====================================================================
-- 1) Stored Procedure: Cierre de Ingreso
-- =====================================================================
DROP PROCEDURE IF EXISTS sp_pl_cerrar_ingreso;
DELIMITER $$
CREATE PROCEDURE sp_pl_cerrar_ingreso(
  IN  p_packinglist_codigo   VARCHAR(64),
  IN  p_descarga_fin_at      DATETIME,      -- NULL = NOW()
  IN  p_operarios_cant       SMALLINT,      -- NULL = mantener
  IN  p_check_all_if_full    TINYINT,       -- 1 = marca checked_all si no hay difs
  OUT o_estado               VARCHAR(20)    -- estado final del PL
)
BEGIN
  /* --------- DECLARACIONES (deben ir primero) --------- */
  DECLARE v_pl_id         BIGINT UNSIGNED;
  DECLARE v_ingreso_id    BIGINT UNSIGNED;
  DECLARE v_fin           DATETIME;
  DECLARE v_has_diffs     INT DEFAULT 0;
  DECLARE v_any_received  BIGINT DEFAULT 0;
  DECLARE v_estado        VARCHAR(20);

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  /* --------- Validaciones --------- */
  IF p_packinglist_codigo IS NULL OR p_packinglist_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_packinglist_codigo es requerido.';
  END IF;

  SELECT id INTO v_pl_id FROM pl_packinglist WHERE codigo = p_packinglist_codigo LIMIT 1;
  IF v_pl_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Packing List no encontrado.';
  END IF;

  /* Tomar el último registro de ingreso asociado al PL */
  SELECT id INTO v_ingreso_id
    FROM pl_ingreso
   WHERE packinglist_id = v_pl_id
   ORDER BY id DESC
   LIMIT 1;

  IF v_ingreso_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'No existe pl_ingreso para este PL.';
  END IF;

  SET v_fin = COALESCE(p_descarga_fin_at, NOW());

  START TRANSACTION;

  /* Actualizar fin de descarga y operarios (si vino valor) */
  UPDATE pl_ingreso
     SET descarga_fin_at = v_fin,
         operarios_cant  = CASE
                             WHEN p_operarios_cant IS NULL OR p_operarios_cant <= 0
                               THEN operarios_cant
                             ELSE p_operarios_cant
                           END
   WHERE id = v_ingreso_id;

  /* ¿Quedan diferencias? ¿Se recibió algo? */
  SELECT
    SUM(CASE WHEN GREATEST(expected_uv - received_uv,0) > 0
              OR GREATEST(expected_uc - received_uc,0) > 0 THEN 1 ELSE 0 END),
    SUM(COALESCE(received_uv,0) + COALESCE(received_uc,0))
  INTO v_has_diffs, v_any_received
  FROM pl_packinglist_item
  WHERE packinglist_id = v_pl_id;

  IF COALESCE(v_has_diffs,0) = 0 THEN
    -- No hay diferencias → cerrar
    UPDATE pl_packinglist
       SET estado = 'CERRADO',
           checked_all = CASE WHEN p_check_all_if_full=1 THEN 1 ELSE checked_all END
     WHERE id = v_pl_id;
    SET v_estado = 'CERRADO';
  ELSE
    -- Hay diferencias → si hubo recepción, dejar en DESCARGADO
    UPDATE pl_packinglist
       SET estado = CASE WHEN COALESCE(v_any_received,0) > 0 THEN 'DESCARGADO' ELSE estado END
     WHERE id = v_pl_id;
    SELECT estado INTO v_estado FROM pl_packinglist WHERE id = v_pl_id;
  END IF;

  COMMIT;

  SET o_estado = v_estado;
  SELECT v_estado AS estado_final, v_ingreso_id AS ingreso_id, v_fin AS descarga_fin_at;
END $$
DELIMITER ;

-- =====================================================================
-- 2) Vistas KPI
-- =====================================================================

/*
 * v_kpi_ingresos_descarga:
 *   - 1 fila por PL (último ingreso asociado)
 *   - tiempos (cola e impacto), rendimientos por hora y por operario
 *   - totales esperados/recibidos/diferencias + pallets involucrados
 */
DROP VIEW IF EXISTS v_kpi_ingresos_descarga;
CREATE VIEW v_kpi_ingresos_descarga AS
WITH pl_last_ingreso AS (
  SELECT
    i.*,
    ROW_NUMBER() OVER (PARTITION BY i.packinglist_id ORDER BY i.id DESC) AS rn
  FROM pl_ingreso i
),
pl_totales AS (
  SELECT
    pli.packinglist_id,
    SUM(pli.expected_uv)                   AS exp_uv_total,
    SUM(pli.expected_uc)                   AS exp_uc_total,
    SUM(pli.received_uv)                   AS rcv_uv_total,
    SUM(pli.received_uc)                   AS rcv_uc_total,
    SUM(pli.expected_uv - pli.received_uv) AS diff_uv_total,
    SUM(pli.expected_uc - pli.received_uc) AS diff_uc_total
  FROM pl_packinglist_item pli
  GROUP BY pli.packinglist_id
),
pl_pallets AS (
  SELECT packinglist_id, COUNT(DISTINCT pallet_id) AS pallets_rcv
  FROM pl_rcv_link
  GROUP BY packinglist_id
)
SELECT
  pl.id                    AS packinglist_id,
  pl.codigo                AS packinglist_codigo,
  pl.cliente_ref,
  pl.fecha                 AS pl_fecha,
  pl.estado                AS pl_estado,
  ing.id                   AS ingreso_id,
  ing.fecha_ingreso,
  ing.llegada_at,
  ing.descarga_inicio_at,
  ing.descarga_fin_at,
  ing.operarios_cant,
  t.exp_uv_total,
  t.exp_uc_total,
  t.rcv_uv_total,
  t.rcv_uc_total,
  t.diff_uv_total,
  t.diff_uc_total,
  COALESCE(p.pallets_rcv,0) AS pallets_rcv,
  -- Tiempos
  TIMESTAMPDIFF(MINUTE, ing.llegada_at, ing.descarga_inicio_at) AS min_cola_llegada_a_inicio,
  TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) AS min_descarga_efectiva,
  TIMESTAMPDIFF(MINUTE, ing.llegada_at, ing.descarga_fin_at) AS min_total_llegada_a_fin,
  -- Rendimientos por hora
  CASE
    WHEN ing.descarga_inicio_at IS NOT NULL AND ing.descarga_fin_at IS NOT NULL
         AND TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) > 0
      THEN ROUND(t.rcv_uv_total / (TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) / 60), 2)
    ELSE NULL
  END AS uv_por_hora,
  CASE
    WHEN ing.descarga_inicio_at IS NOT NULL AND ing.descarga_fin_at IS NOT NULL
         AND TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) > 0
      THEN ROUND(t.rcv_uc_total / (TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) / 60), 2)
    ELSE NULL
  END AS uc_por_hora,
  CASE
    WHEN ing.operarios_cant IS NOT NULL AND ing.operarios_cant > 0
         AND ing.descarga_inicio_at IS NOT NULL AND ing.descarga_fin_at IS NOT NULL
         AND TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) > 0
      THEN ROUND(t.rcv_uv_total / (ing.operarios_cant * (TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) / 60)), 2)
    ELSE NULL
  END AS uv_por_operario_hora,
  CASE
    WHEN ing.operarios_cant IS NOT NULL AND ing.operarios_cant > 0
         AND ing.descarga_inicio_at IS NOT NULL AND ing.descarga_fin_at IS NOT NULL
         AND TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) > 0
      THEN ROUND(t.rcv_uc_total / (ing.operarios_cant * (TIMESTAMPDIFF(MINUTE, ing.descarga_inicio_at, ing.descarga_fin_at) / 60)), 2)
    ELSE NULL
  END AS uc_por_operario_hora
FROM pl_packinglist pl
LEFT JOIN pl_last_ingreso ing
       ON ing.packinglist_id = pl.id AND ing.rn = 1
LEFT JOIN pl_totales t
       ON t.packinglist_id = pl.id
LEFT JOIN pl_pallets p
       ON p.packinglist_id = pl.id;

-- Detalle por ítem (esperado vs recibido, difs, con descripción de producto)
DROP VIEW IF EXISTS v_kpi_ingreso_detalle;
CREATE VIEW v_kpi_ingreso_detalle AS
SELECT
  pl.id                     AS packinglist_id,
  pl.codigo                 AS packinglist_codigo,
  i.id                      AS pl_item_id,
  i.sku_cliente,
  i.producto_id,
  pr.denominacion           AS producto,
  i.descripcion             AS desc_pl,
  i.lote_codigo,
  i.fecha_produccion,
  i.fecha_vencimiento,
  i.expected_uv,
  i.expected_uc,
  i.received_uv,
  i.received_uc,
  (i.expected_uv - i.received_uv) AS diff_uv,
  (i.expected_uc - i.received_uc) AS diff_uc,
  i.checked_item
FROM pl_packinglist_item i
JOIN pl_packinglist pl ON pl.id = i.packinglist_id
LEFT JOIN para_productos pr ON pr.id = i.producto_id;
