/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 8 (patch v2): Cierre de Ingreso con autocreación de pl_ingreso
 * Archivo: database/step8_pl_close_kpis_patch_v2.sql
 *
 * Cambios:
 * - Si no existe pl_ingreso para el PL, el SP lo crea automáticamente.
 *   • deposito_id: se infiere del primer pallet recibido (pl_rcv_link→wh_pallet)
 *     o se usa DEP1 si no hay recepciones todavía.
 *   • tiempos: llegada/inicio se toman del primer movimiento asociado, si existe;
 *     si no, se usan NOW().
 * - Mantiene la lógica de cierre/estado del Paso 8.
 * ===================================================================== */

SET NAMES utf8mb4;

DROP PROCEDURE IF EXISTS sp_pl_cerrar_ingreso;
DELIMITER $$
CREATE PROCEDURE sp_pl_cerrar_ingreso(
  IN  p_packinglist_codigo   VARCHAR(64),
  IN  p_descarga_fin_at      DATETIME,      -- NULL = NOW()
  IN  p_operarios_cant       SMALLINT,      -- NULL = mantener (o setear si se autocrea)
  IN  p_check_all_if_full    TINYINT,       -- 1 = marca checked_all si no hay difs
  OUT o_estado               VARCHAR(20)    -- estado final del PL
)
BEGIN
  /* --------- DECLARACIONES --------- */
  DECLARE v_pl_id         BIGINT UNSIGNED;
  DECLARE v_ingreso_id    BIGINT UNSIGNED;
  DECLARE v_fin           DATETIME;
  DECLARE v_has_diffs     INT DEFAULT 0;
  DECLARE v_any_received  BIGINT DEFAULT 0;
  DECLARE v_estado        VARCHAR(20);

  DECLARE v_dep_guess     INT UNSIGNED;
  DECLARE v_llegada       DATETIME;
  DECLARE v_inicio        DATETIME;

  DECLARE EXIT HANDLER FOR SQLEXCEPTION
  BEGIN
    ROLLBACK;
    RESIGNAL;
  END;

  /* --------- Validaciones --------- */
  IF p_packinglist_codigo IS NULL OR p_packinglist_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'p_packinglist_codigo es requerido.';
  END IF;

  SELECT id INTO v_pl_id FROM pl_packinglist WHERE codigo = p_packinglist_codigo LIMIT 1;
  IF v_pl_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT = 'Packing List no encontrado.';
  END IF;

  /* Intentar obtener el ingreso existente */
  SELECT id INTO v_ingreso_id
    FROM pl_ingreso
   WHERE packinglist_id = v_pl_id
   ORDER BY id DESC
   LIMIT 1;

  /* --------- Autocreación de pl_ingreso si no existe --------- */
  IF v_ingreso_id IS NULL THEN
    /* 1) Inferir depósito desde lo recibido (pallets vinculados) */
    SELECT p.deposito_id INTO v_dep_guess
      FROM pl_rcv_link l
      JOIN wh_pallet p ON p.id = l.pallet_id
     WHERE l.packinglist_id = v_pl_id
     ORDER BY l.id
     LIMIT 1;

    /* 2) Si no hay recepción aún, usar DEP1 */
    IF v_dep_guess IS NULL THEN
      SELECT id INTO v_dep_guess FROM wh_deposito WHERE code = 'DEP1' LIMIT 1;
    END IF;

    IF v_dep_guess IS NULL THEN
      SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'No existe pl_ingreso y no se pudo inferir depósito. Cree DEP1 o cargue pl_ingreso manualmente.';
    END IF;

    /* 3) Inferir tiempos desde movimientos; si no hay, usar NOW() */
    SELECT MIN(m.created_at) INTO v_llegada
      FROM pl_rcv_link l
      JOIN wh_move m ON m.id = l.move_id
     WHERE l.packinglist_id = v_pl_id;

    SET v_inicio = v_llegada;
    IF v_llegada IS NULL THEN
      SET v_llegada = NOW();
      SET v_inicio  = v_llegada;
    END IF;

    INSERT INTO pl_ingreso (
      packinglist_id, deposito_id, fecha_ingreso,
      llegada_at, descarga_inicio_at, operarios_cant,
      doc_tipo, doc_numero, observacion
    )
    VALUES (
      v_pl_id, v_dep_guess, DATE(v_llegada),
      v_llegada, v_inicio, NULLIF(p_operarios_cant,0),
      NULL, NULL, 'Autocreado por sp_pl_cerrar_ingreso'
    );

    SET v_ingreso_id = LAST_INSERT_ID();
  END IF;

  /* --------- Cierre: actualizar fin y estado --------- */
  SET v_fin = COALESCE(p_descarga_fin_at, NOW());

  START TRANSACTION;

  UPDATE pl_ingreso
     SET descarga_fin_at = v_fin,
         operarios_cant  = CASE
                             WHEN p_operarios_cant IS NULL OR p_operarios_cant <= 0
                               THEN operarios_cant
                             ELSE p_operarios_cant
                           END
   WHERE id = v_ingreso_id;

  /* ¿Quedan diferencias? ¿Se recibió algo? */
  SELECT
    SUM(CASE WHEN GREATEST(expected_uv - received_uv,0) > 0
              OR GREATEST(expected_uc - received_uc,0) > 0 THEN 1 ELSE 0 END),
    SUM(COALESCE(received_uv,0) + COALESCE(received_uc,0))
  INTO v_has_diffs, v_any_received
  FROM pl_packinglist_item
  WHERE packinglist_id = v_pl_id;

  IF COALESCE(v_has_diffs,0) = 0 THEN
    -- No hay diferencias → cerrar
    UPDATE pl_packinglist
       SET estado = 'CERRADO',
           checked_all = CASE WHEN p_check_all_if_full=1 THEN 1 ELSE checked_all END
     WHERE id = v_pl_id;
    SET v_estado = 'CERRADO';
  ELSE
    -- Hay diferencias → si hubo recepción, dejar en DESCARGADO
    UPDATE pl_packinglist
       SET estado = CASE WHEN COALESCE(v_any_received,0) > 0 THEN 'DESCARGADO' ELSE estado END
     WHERE id = v_pl_id;
    SELECT estado INTO v_estado FROM pl_packinglist WHERE id = v_pl_id;
  END IF;

  COMMIT;

  SET o_estado = v_estado;
  SELECT v_estado AS estado_final, v_ingreso_id AS ingreso_id, v_fin AS descarga_fin_at;
END $$
DELIMITER ;
