/* =====================================================================
 * SOL - Sistema de Operaciones Logísticas
 * STEP 9: Esquema de Salidas / Embarques (prefijo so_*)
 * Archivo: database/step9_so_outbound.sql
 *
 * Alcance:
 *  - Catálogos de estado (pedido / pre-embarque / embarque) y motivos de devolución
 *  - Staging de Excel para pedidos de salida (so_import_*)
 *  - Pedido de salida por destinatario + ítems (so_pedido / so_pedido_dest / so_pedido_dest_item)
 *  - Pre-embarque / picking con tareas, tiempos y fotos (so_preembarque / so_pre_tarea / so_pre_pick)
 *  - Embarque (chofer/móvil, tiempos, tickets, temperatura) + ruteo por paradas (so_embarque / so_embarque_parada / so_parada_doc)
 *  - Vínculo de egreso físico a pedido (so_ship_link → actualiza shipped_* por triggers en paso posterior)
 *  - Retorno y devoluciones parametrizadas (so_retorno / so_devolucion)
 *  - Vistas de verificación y KPI básicas
 *
 * NOTA:
 *  - Este paso define estructura y vistas; los SPs/Triggers de egreso se agregarán en un paso siguiente.
 *  - Requiere tablas de parámetros ya existentes: para_productos, para_destinatarios, para_moviles, para_choferes, wh_*.
 * ===================================================================== */

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- =====================================================================
-- 0) Ajustes de Warehouse: agregar ambientes de preparación y despacho
-- =====================================================================
INSERT INTO wh_ambiente (code, nombre, descripcion, activo)
VALUES ('PREP','Preparación','Zona de preparación de pedidos',1)
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), descripcion=VALUES(descripcion), activo=1;

INSERT INTO wh_ambiente (code, nombre, descripcion, activo)
VALUES ('DESPACHO','Despacho','Andenes / docks de carga',1)
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), descripcion=VALUES(descripcion), activo=1;

-- =====================================================================
-- 1) Catálogos de estado y motivos
-- =====================================================================

DROP TABLE IF EXISTS so_pedido_estado;
CREATE TABLE so_pedido_estado (
  id      INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code    VARCHAR(32) NOT NULL,  -- RECIBIDO, EN_PREPARACION, PARCIAL, PREPARADO, EN_EMBARQUE, DESPACHADO, CERRADO, CANCELADO
  nombre  VARCHAR(100) NOT NULL,
  orden   TINYINT UNSIGNED NOT NULL DEFAULT 1,
  activo  TINYINT(1) NOT NULL DEFAULT 1,
  UNIQUE KEY uk_so_ped_estado (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

INSERT INTO so_pedido_estado (code,nombre,orden) VALUES
('RECIBIDO','Recibido',1),
('EN_PREPARACION','En preparación',2),
('PARCIAL','Parcialmente preparado',3),
('PREPARADO','Preparado',4),
('EN_EMBARQUE','En embarque',5),
('DESPACHADO','Despachado',6),
('CERRADO','Cerrado',7),
('CANCELADO','Cancelado',9)
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden);

DROP TABLE IF EXISTS so_preembarque_estado;
CREATE TABLE so_preembarque_estado (
  id      INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code    VARCHAR(32) NOT NULL, -- PENDIENTE, ASIGNADO, EN_PROCESO, COMPLETADO, CANCELADO
  nombre  VARCHAR(100) NOT NULL,
  orden   TINYINT UNSIGNED NOT NULL DEFAULT 1,
  activo  TINYINT(1) NOT NULL DEFAULT 1,
  UNIQUE KEY uk_so_pre_estado (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

INSERT INTO so_preembarque_estado (code,nombre,orden) VALUES
('PENDIENTE','Pendiente',1),
('ASIGNADO','Asignado',2),
('EN_PROCESO','En proceso',3),
('COMPLETADO','Completado',4),
('CANCELADO','Cancelado',9)
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden);

DROP TABLE IF EXISTS so_embarque_estado;
CREATE TABLE so_embarque_estado (
  id      INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code    VARCHAR(32) NOT NULL, -- EN_COLA, EN_CARGA, CARGADO, EN_RUTA, ENTREGADO, FINALIZADO, CANCELADO
  nombre  VARCHAR(100) NOT NULL,
  orden   TINYINT UNSIGNED NOT NULL DEFAULT 1,
  activo  TINYINT(1) NOT NULL DEFAULT 1,
  UNIQUE KEY uk_so_emb_estado (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

INSERT INTO so_embarque_estado (code,nombre,orden) VALUES
('EN_COLA','En cola',1),
('EN_CARGA','En carga',2),
('CARGADO','Cargado',3),
('EN_RUTA','En ruta',4),
('ENTREGADO','Entregado',5),
('FINALIZADO','Finalizado',6),
('CANCELADO','Cancelado',9)
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden);

DROP TABLE IF EXISTS para_devolucion_motivo;
CREATE TABLE para_devolucion_motivo (
  id      INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  code    VARCHAR(32) NOT NULL,  -- VENCIDO, DAÑADO, FALTANTE_DOC, CLIENTE_RECHAZA, OTRO
  nombre  VARCHAR(100) NOT NULL,
  activo  TINYINT(1) NOT NULL DEFAULT 1,
  UNIQUE KEY uk_dev_motivo (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

INSERT INTO para_devolucion_motivo (code,nombre) VALUES
('VENCIDO','Vencido'),
('DANADO','Dañado'),
('FALTANTE_DOC','Faltante de documentación'),
('CLIENTE_RECHAZA','Rechazo del cliente'),
('OTRO','Otro')
ON DUPLICATE KEY UPDATE nombre=VALUES(nombre);

-- =====================================================================
-- 2) Staging de Excel para pedidos de salida
-- =====================================================================

DROP TABLE IF EXISTS so_import_batch;
CREATE TABLE so_import_batch (
  id            BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  tipo          VARCHAR(32) NOT NULL DEFAULT 'PEDIDO_SALIDA',
  filename      VARCHAR(255) NULL,
  sheet_name    VARCHAR(128) NULL,
  rows_total    INT UNSIGNED NULL,
  rows_ok       INT UNSIGNED NULL,
  rows_error    INT UNSIGNED NULL,
  created_at    DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS so_import_row;
CREATE TABLE so_import_row (
  id              BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  batch_id        BIGINT UNSIGNED NOT NULL,
  rownum          INT UNSIGNED NOT NULL,
  raw             JSON NOT NULL,
  status          ENUM('NEW','OK','ERROR') NOT NULL DEFAULT 'NEW',
  error_msg       VARCHAR(255) NULL,
  pedido_id       BIGINT UNSIGNED NULL,
  pedido_dest_id  BIGINT UNSIGNED NULL,
  item_id         BIGINT UNSIGNED NULL,
  created_at      DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_soimp_batch FOREIGN KEY (batch_id) REFERENCES so_import_batch(id) ON DELETE CASCADE,
  INDEX idx_soimp_batch (batch_id, rownum)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 3) Pedido de salida (cabecera → destinatarios → ítems)
-- =====================================================================

DROP TABLE IF EXISTS so_pedido;
CREATE TABLE so_pedido (
  id               BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  codigo           VARCHAR(64) NOT NULL,              -- identificador del pedido (excel/ERP)
  cliente_ref      VARCHAR(64) NULL,                  -- tenant / ref externa
  cliente_id       BIGINT UNSIGNED NULL,              -- opcional (FK a para_clientes si existe)
  fecha_pedido     DATE NOT NULL,
  estado_id        INT UNSIGNED NOT NULL,             -- so_pedido_estado
  observacion      VARCHAR(255) NULL,
  import_batch_id  BIGINT UNSIGNED NULL,              -- so_import_batch.id
  created_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted_at       DATETIME NULL,
  UNIQUE KEY uk_soped_codigo (codigo),
  CONSTRAINT fk_soped_estado FOREIGN KEY (estado_id) REFERENCES so_pedido_estado(id),
  CONSTRAINT fk_soped_batch  FOREIGN KEY (import_batch_id) REFERENCES so_import_batch(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS so_pedido_dest;
CREATE TABLE so_pedido_dest (
  id               BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  pedido_id        BIGINT UNSIGNED NOT NULL,
  destinatario_id  BIGINT UNSIGNED NOT NULL,      -- para_destinatarios.id
  doc_tipo         VARCHAR(20) NULL,              -- FACTURA / REMITO / OTRO
  doc_numero       VARCHAR(64) NULL,
  nota             VARCHAR(255) NULL,
  created_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at       DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_sopedest_ped  FOREIGN KEY (pedido_id)       REFERENCES so_pedido(id) ON DELETE CASCADE,
  CONSTRAINT fk_sopedest_dest FOREIGN KEY (destinatario_id) REFERENCES para_destinatarios(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS so_pedido_dest_item;
CREATE TABLE so_pedido_dest_item (
  id                BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  pedido_dest_id    BIGINT UNSIGNED NOT NULL,
  producto_id       BIGINT UNSIGNED NOT NULL,     -- para_productos.id
  lote_codigo       VARCHAR(64) NULL,             -- opcional si el pedido exige lote
  expected_uv       INT NOT NULL DEFAULT 0,       -- cajas solicitadas
  expected_uc       INT NOT NULL DEFAULT 0,       -- unidades sueltas solicitadas
  prepared_uv       INT NOT NULL DEFAULT 0,       -- preparado hacia PREP
  prepared_uc       INT NOT NULL DEFAULT 0,
  shipped_uv        INT NOT NULL DEFAULT 0,       -- efectivamente despachado
  shipped_uc        INT NOT NULL DEFAULT 0,
  created_at        DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at        DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_sopdi_dest   FOREIGN KEY (pedido_dest_id) REFERENCES so_pedido_dest(id) ON DELETE CASCADE,
  CONSTRAINT fk_sopdi_prod   FOREIGN KEY (producto_id)    REFERENCES para_productos(id),
  INDEX idx_sopdi_key (pedido_dest_id, producto_id, lote_codigo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 4) Pre-embarque / Picking (tareas, tiempos, fotos)
-- =====================================================================

DROP TABLE IF EXISTS so_preembarque;
CREATE TABLE so_preembarque (
  id                     BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  codigo                 VARCHAR(64) NOT NULL,       -- ej.: PRE-0001
  pedido_id              BIGINT UNSIGNED NOT NULL,
  deposito_id            INT UNSIGNED NOT NULL,      -- wh_deposito.id
  estado_id              INT UNSIGNED NOT NULL,      -- so_preembarque_estado.id
  zona_posicion_id       BIGINT UNSIGNED NULL,       -- wh_posicion.id (área PREP asignada)
  montacarguista_user_id INT UNSIGNED NULL,          -- sys_users.id (futuro)
  asignado_at            DATETIME NULL,
  inicio_at              DATETIME NULL,              -- inicio global de pre-embarque
  fin_at                 DATETIME NULL,              -- fin global
  observacion            VARCHAR(255) NULL,
  created_at             DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at             DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uk_so_pre_codigo (codigo),
  CONSTRAINT fk_so_pre_ped   FOREIGN KEY (pedido_id)   REFERENCES so_pedido(id),
  CONSTRAINT fk_so_pre_dep   FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_so_pre_est   FOREIGN KEY (estado_id)   REFERENCES so_preembarque_estado(id),
  CONSTRAINT fk_so_pre_pos   FOREIGN KEY (zona_posicion_id) REFERENCES wh_posicion(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Tareas por rol (montacarga, preparación, control) con tiempos y foto
DROP TABLE IF EXISTS so_pre_tarea;
CREATE TABLE so_pre_tarea (
  id                BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  preembarque_id    BIGINT UNSIGNED NOT NULL,
  tipo              ENUM('MONTACARGA','PREPARACION','CONTROL') NOT NULL,
  responsable_user_id INT UNSIGNED NULL,   -- sys_users.id (futuro)
  inicio_at         DATETIME NULL,
  fin_at            DATETIME NULL,
  foto_url          VARCHAR(255) NULL,
  nota              VARCHAR(255) NULL,
  created_at        DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_so_pret_pre FOREIGN KEY (preembarque_id) REFERENCES so_preembarque(id) ON DELETE CASCADE,
  INDEX idx_so_pret_tipo (preembarque_id, tipo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Picks planificados/ejecutados: de posición → zona PREP (traza pallet/origen)
DROP TABLE IF EXISTS so_pre_pick;
CREATE TABLE so_pre_pick (
  id                  BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  preembarque_id      BIGINT UNSIGNED NOT NULL,
  pedido_dest_item_id BIGINT UNSIGNED NOT NULL,
  from_pos_id         BIGINT UNSIGNED NOT NULL,   -- wh_posicion (origen)
  to_pos_id           BIGINT UNSIGNED NULL,       -- wh_posicion (zona PREP)
  pallet_id           BIGINT UNSIGNED NULL,       -- wh_pallet (si aplica)
  lote_id             BIGINT UNSIGNED NULL,       -- wh_lote (si aplica)
  uv_cajas            INT NOT NULL DEFAULT 0,
  uc_unidades         INT NOT NULL DEFAULT 0,
  creado_por          INT UNSIGNED NULL,          -- sys_users.id (futuro)
  created_at          DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_spk_pre   FOREIGN KEY (preembarque_id)      REFERENCES so_preembarque(id) ON DELETE CASCADE,
  CONSTRAINT fk_spk_item  FOREIGN KEY (pedido_dest_item_id) REFERENCES so_pedido_dest_item(id),
  CONSTRAINT fk_spk_from  FOREIGN KEY (from_pos_id)         REFERENCES wh_posicion(id),
  CONSTRAINT fk_spk_to    FOREIGN KEY (to_pos_id)           REFERENCES wh_posicion(id),
  CONSTRAINT fk_spk_pal   FOREIGN KEY (pallet_id)           REFERENCES wh_pallet(id),
  CONSTRAINT fk_spk_lote  FOREIGN KEY (lote_id)             REFERENCES wh_lote(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 5) Embarque (asigna uno o varios pre-embarques a un móvil/chofer) + ruteo
-- =====================================================================

DROP TABLE IF EXISTS so_embarque;
CREATE TABLE so_embarque (
  id                BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  codigo            VARCHAR(64) NOT NULL,         -- EMB-0001
  deposito_id       INT UNSIGNED NOT NULL,        -- wh_deposito
  movil_id          BIGINT UNSIGNED NULL,         -- para_moviles
  chofer_id         BIGINT UNSIGNED NULL,         -- para_choferes
  estado_id         INT UNSIGNED NOT NULL,        -- so_embarque_estado
  -- Tiempos del ciclo
  creado_at         DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP, -- fecha/hora de armado del embarque
  llegada_at        DATETIME NULL,                 -- llegada del móvil al local
  carga_fecha       DATE NULL,                     -- fecha de carga (si se maneja separada)
  carga_inicio_at   DATETIME NULL,
  carga_fin_at      DATETIME NULL,
  ayudantes_cant    SMALLINT UNSIGNED NULL,
  salida_at         DATETIME NULL,                 -- salida del local
  -- Documentación / condiciones
  ticket_porteria   VARCHAR(64) NULL,
  ticket_bascula    VARCHAR(64) NULL,
  km_inicial        INT NULL,
  temp_salida_c     DECIMAL(5,2) NULL,            -- temperatura a la salida (refrigerados)
  observacion       VARCHAR(255) NULL,
  updated_at        DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uk_so_emb_codigo (codigo),
  CONSTRAINT fk_soemb_dep   FOREIGN KEY (deposito_id) REFERENCES wh_deposito(id),
  CONSTRAINT fk_soemb_movil FOREIGN KEY (movil_id)    REFERENCES para_moviles(id),
  CONSTRAINT fk_soemb_cho   FOREIGN KEY (chofer_id)   REFERENCES para_choferes(id),
  CONSTRAINT fk_soemb_est   FOREIGN KEY (estado_id)   REFERENCES so_embarque_estado(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Varios pre-embarques pueden pertenecer a un mismo embarque
DROP TABLE IF EXISTS so_embarque_pre;
CREATE TABLE so_embarque_pre (
  id             BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  embarque_id    BIGINT UNSIGNED NOT NULL,
  preembarque_id BIGINT UNSIGNED NOT NULL,
  UNIQUE KEY uk_soembpre (embarque_id, preembarque_id),
  CONSTRAINT fk_soembpre_emb FOREIGN KEY (embarque_id)    REFERENCES so_embarque(id)    ON DELETE CASCADE,
  CONSTRAINT fk_soembpre_pre FOREIGN KEY (preembarque_id) REFERENCES so_preembarque(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Ruteo por destinatario (paradas)
DROP TABLE IF EXISTS so_embarque_parada;
CREATE TABLE so_embarque_parada (
  id                   BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  embarque_id          BIGINT UNSIGNED NOT NULL,
  destinatario_id      BIGINT UNSIGNED NOT NULL,
  pedido_dest_id       BIGINT UNSIGNED NULL,     -- referencia opcional al pedido_dest atendido en esta parada
  orden                INT UNSIGNED NOT NULL DEFAULT 1,
  km_llegada           INT NULL,
  hora_llegada         DATETIME NULL,
  hora_inicio_descarga DATETIME NULL,
  hora_fin_descarga    DATETIME NULL,
  hora_fin_inspeccion  DATETIME NULL,            -- fin de inspección del móvil tras descarga
  hora_salida          DATETIME NULL,
  temp_descarga_c      DECIMAL(5,2) NULL,        -- temperatura de descarga (refrigerados)
  observacion          VARCHAR(255) NULL,
  created_at           DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at           DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_sopar_emb   FOREIGN KEY (embarque_id)     REFERENCES so_embarque(id) ON DELETE CASCADE,
  CONSTRAINT fk_sopar_dest  FOREIGN KEY (destinatario_id) REFERENCES para_destinatarios(id),
  CONSTRAINT fk_sopar_ped   FOREIGN KEY (pedido_dest_id)  REFERENCES so_pedido_dest(id),
  INDEX idx_sopar_route (embarque_id, orden)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Documentos atendidos en cada parada (facturas, remitos, etc.)
DROP TABLE IF EXISTS so_parada_doc;
CREATE TABLE so_parada_doc (
  id            BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  parada_id     BIGINT UNSIGNED NOT NULL,
  doc_tipo      VARCHAR(20) NOT NULL,
  doc_numero    VARCHAR(64) NOT NULL,
  created_at    DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_sopardoc_par FOREIGN KEY (parada_id) REFERENCES so_embarque_parada(id) ON DELETE CASCADE,
  UNIQUE KEY uk_sopardoc (parada_id, doc_tipo, doc_numero)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Vínculo de egreso físico con el pedido (para trazabilidad y update de shipped_*)
DROP TABLE IF EXISTS so_ship_link;
CREATE TABLE so_ship_link (
  id                  BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  pedido_dest_item_id BIGINT UNSIGNED NOT NULL,
  producto_id         BIGINT UNSIGNED NOT NULL,
  lote_id             BIGINT UNSIGNED NULL,
  pallet_id           BIGINT UNSIGNED NULL,
  move_id             BIGINT UNSIGNED NULL,   -- wh_move.id (OUT)
  embarque_id         BIGINT UNSIGNED NULL,
  parada_id           BIGINT UNSIGNED NULL,
  uv_cajas            INT NOT NULL DEFAULT 0,
  uc_unidades         INT NOT NULL DEFAULT 0,
  created_at          DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_sosl_item  FOREIGN KEY (pedido_dest_item_id) REFERENCES so_pedido_dest_item(id) ON DELETE CASCADE,
  CONSTRAINT fk_sosl_prod  FOREIGN KEY (producto_id)         REFERENCES para_productos(id),
  CONSTRAINT fk_sosl_lote  FOREIGN KEY (lote_id)             REFERENCES wh_lote(id),
  CONSTRAINT fk_sosl_pal   FOREIGN KEY (pallet_id)           REFERENCES wh_pallet(id),
  CONSTRAINT fk_sosl_move  FOREIGN KEY (move_id)             REFERENCES wh_move(id),
  CONSTRAINT fk_sosl_emb   FOREIGN KEY (embarque_id)         REFERENCES so_embarque(id),
  CONSTRAINT fk_sosl_par   FOREIGN KEY (parada_id)           REFERENCES so_embarque_parada(id),
  INDEX idx_sosl_lookup (pedido_dest_item_id, producto_id, lote_id, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 6) Retorno y devoluciones (vuelven a cuarentena)
-- =====================================================================

DROP TABLE IF EXISTS so_retorno;
CREATE TABLE so_retorno (
  id            BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  embarque_id   BIGINT UNSIGNED NOT NULL,
  llegada_at    DATETIME NOT NULL,
  observacion   VARCHAR(255) NULL,
  created_at    DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_soret_emb FOREIGN KEY (embarque_id) REFERENCES so_embarque(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

DROP TABLE IF EXISTS so_devolucion;
CREATE TABLE so_devolucion (
  id                BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  retorno_id        BIGINT UNSIGNED NOT NULL,
  destinatario_id   BIGINT UNSIGNED NULL,
  doc_tipo          VARCHAR(20) NULL,
  doc_numero        VARCHAR(64) NULL,
  motivo_id         INT UNSIGNED NOT NULL,       -- para_devolucion_motivo
  producto_id       BIGINT UNSIGNED NOT NULL,
  lote_id           BIGINT UNSIGNED NULL,
  qty_pallets       INT NOT NULL DEFAULT 0,
  qty_uv            INT NOT NULL DEFAULT 0,
  qty_uc            INT NOT NULL DEFAULT 0,
  responsable_user_id INT UNSIGNED NULL,         -- quien recibe la devolución
  created_at        DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_sodev_ret    FOREIGN KEY (retorno_id)      REFERENCES so_retorno(id) ON DELETE CASCADE,
  CONSTRAINT fk_sodev_dest   FOREIGN KEY (destinatario_id) REFERENCES para_destinatarios(id),
  CONSTRAINT fk_sodev_mot    FOREIGN KEY (motivo_id)       REFERENCES para_devolucion_motivo(id),
  CONSTRAINT fk_sodev_prod   FOREIGN KEY (producto_id)     REFERENCES para_productos(id),
  CONSTRAINT fk_sodev_lote   FOREIGN KEY (lote_id)         REFERENCES wh_lote(id),
  INDEX idx_sodev_lookup (retorno_id, producto_id, lote_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- =====================================================================
-- 7) Vistas de verificación y KPI básicas
-- =====================================================================

-- Resumen de pedido por destinatario (esperado / preparado / despachado / pendientes)
DROP VIEW IF EXISTS v_so_pedido_resumen;
CREATE VIEW v_so_pedido_resumen AS
SELECT
  p.id            AS pedido_id,
  p.codigo        AS pedido_codigo,
  p.fecha_pedido,
  pe.code         AS pedido_estado,
  d.id            AS pedido_dest_id,
  dest.razon_social AS destinatario,
  d.doc_tipo,
  d.doc_numero,
  SUM(i.expected_uv) AS expected_uv,
  SUM(i.expected_uc) AS expected_uc,
  SUM(i.prepared_uv) AS prepared_uv,
  SUM(i.prepared_uc) AS prepared_uc,
  SUM(i.shipped_uv)  AS shipped_uv,
  SUM(i.shipped_uc)  AS shipped_uc,
  SUM(i.expected_uv - i.shipped_uv) AS pending_uv,
  SUM(i.expected_uc - i.shipped_uc) AS pending_uc
FROM so_pedido p
JOIN so_pedido_estado pe ON pe.id = p.estado_id
JOIN so_pedido_dest d    ON d.pedido_id = p.id
JOIN para_destinatarios dest ON dest.id = d.destinatario_id
JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id
GROUP BY p.id, d.id;

-- KPI de embarque (tiempos de carga y totales, pallets/doc y ruteo básico)
DROP VIEW IF EXISTS v_so_embarque_kpi;
CREATE VIEW v_so_embarque_kpi AS
SELECT
  e.id            AS embarque_id,
  e.codigo        AS embarque_codigo,
  depp.code       AS deposito_code,
  e.creado_at,
  e.llegada_at,
  e.carga_inicio_at,
  e.carga_fin_at,
  e.salida_at,
  e.ayudantes_cant,
  ee.code         AS estado,
  e.km_inicial,
  e.temp_salida_c,
  -- tiempos (minutos)
  TIMESTAMPDIFF(MINUTE, e.llegada_at, e.carga_inicio_at) AS min_espera_carga,
  TIMESTAMPDIFF(MINUTE, e.carga_inicio_at, e.carga_fin_at) AS min_carga_efectiva,
  TIMESTAMPDIFF(MINUTE, e.llegada_at, e.salida_at) AS min_total_en_planta,
  -- conteos
  (SELECT COUNT(*) FROM so_embarque_pre ep WHERE ep.embarque_id = e.id) AS preembarques,
  (SELECT COUNT(*) FROM so_embarque_parada pa WHERE pa.embarque_id = e.id) AS paradas,
  (SELECT COUNT(*) FROM so_parada_doc pd
    JOIN so_embarque_parada pa ON pa.id=pd.parada_id
   WHERE pa.embarque_id = e.id) AS documentos
FROM so_embarque e
JOIN so_embarque_estado ee ON ee.id = e.estado_id
JOIN wh_deposito depp ON depp.id = e.deposito_id;

-- Ruteo por paradas (duraciones y tiempos derivados)
DROP VIEW IF EXISTS v_so_ruteo_paradas;
CREATE VIEW v_so_ruteo_paradas AS
SELECT
  e.codigo        AS embarque,
  p.orden,
  dest.razon_social AS destinatario,
  p.km_llegada,
  p.hora_llegada,
  p.hora_inicio_descarga,
  p.hora_fin_descarga,
  p.hora_fin_inspeccion,
  p.hora_salida,
  p.temp_descarga_c,
  TIMESTAMPDIFF(MINUTE, p.hora_llegada, p.hora_inicio_descarga) AS min_espera,
  TIMESTAMPDIFF(MINUTE, p.hora_inicio_descarga, p.hora_fin_descarga) AS min_descarga,
  TIMESTAMPDIFF(MINUTE, p.hora_llegada, p.hora_salida) AS min_entrega
FROM so_embarque_parada p
JOIN so_embarque e ON e.id = p.embarque_id
JOIN para_destinatarios dest ON dest.id = p.destinatario_id
ORDER BY e.id, p.orden;

-- Devoluciones por retorno (para control rápido)
DROP VIEW IF EXISTS v_so_devoluciones;
CREATE VIEW v_so_devoluciones AS
SELECT
  r.embarque_id,
  e.codigo         AS embarque,
  r.llegada_at     AS retorno_llegada,
  d.id             AS devolucion_id,
  dest.razon_social AS destinatario,
  d.doc_tipo, d.doc_numero,
  m.code           AS motivo,
  pr.denominacion  AS producto,
  l.codigo         AS lote,
  d.qty_pallets, d.qty_uv, d.qty_uc,
  d.created_at
FROM so_devolucion d
JOIN so_retorno r             ON r.id = d.retorno_id
JOIN so_embarque e            ON e.id = r.embarque_id
LEFT JOIN para_destinatarios dest ON dest.id = d.destinatario_id
JOIN para_devolucion_motivo m ON m.id = d.motivo_id
JOIN para_productos pr        ON pr.id = d.producto_id
LEFT JOIN wh_lote l           ON l.id = d.lote_id
ORDER BY d.created_at DESC;

SET FOREIGN_KEY_CHECKS = 1;
