/**
 * SOL - Build con Gulp
 * Tareas:
 *  - styles: compila Sass -> public/assets/css/app.css (con Autoprefixer)
 *  - serve: BrowserSync (proxy a sol.test), recarga en cambios
 *  - watch: observa SCSS/JS/PHP/Pug y recompila/recarga
 */

const gulp = require('gulp');
const sass = require('gulp-sass')(require('sass'));

// Compatibilidad ESM/CJS: gulp-autoprefixer@9 exporta como ESM (default)
const _gap = require('gulp-autoprefixer');
const autoprefixer = _gap && _gap.default ? _gap.default : _gap;

const livereload = require('gulp-livereload');
const browserSync = require('browser-sync').create();
const pug = require('gulp-pug');

const paths = {
  scssEntry: 'assets/scss/app.scss',
  scssWatch: 'assets/scss/**/*.scss',
  cssDest: 'public/assets/css',
  jsWatch: 'public/assets/js/**/*.js',
  phpWatch: ['views/**/*.php', 'public/**/*.php', 'app/**/*.php'],
  pugSrc: 'assets/pug/**/*.pug',
  pugDest: 'views'
};

function styles() {
  return gulp.src(paths.scssEntry, { allowEmpty: true, sourcemaps: true })
    .pipe(
      sass({ outputStyle: 'expanded' })
        .on('error', function (err) {
          console.error('[SASS ERROR]', err.messageFormatted || err.message);
          this.emit('end');
        })
    )
    .pipe(
      autoprefixer({
        cascade: false,
        grid: 'autoplace'
        // overrideBrowserslist: ['>0.5%', 'last 2 versions', 'Firefox ESR', 'not dead'] // opcional
      })
    )
    .pipe(gulp.dest(paths.cssDest, { sourcemaps: '.' }))
    .pipe(browserSync.stream())
    .pipe(livereload());
}

function viewsPug() {
  return gulp.src(paths.pugSrc, { base: 'assets/pug', allowEmpty: true })
    .pipe(
      pug({ pretty: true })
        .on('error', function (err) {
          console.error('[PUG ERROR]', err.message);
          this.emit('end');
        })
    )
    .pipe(gulp.dest(paths.pugDest))
    .pipe(browserSync.stream())
    .pipe(livereload());
}

function serve(done) {
  browserSync.init({
    proxy: 'http://sol.test',
    open: false,
    notify: false,
    snippetOptions: {
      rule: {
        match: /<\/body>/i,
        fn: function (snippet, match) { return snippet + match; }
      }
    }
  });
  livereload.listen();
  done();
}

function watch() {
  gulp.watch(paths.scssWatch, styles);
  gulp.watch(paths.pugSrc, viewsPug);
  gulp.watch(paths.jsWatch).on('change', browserSync.reload);
  gulp.watch(paths.phpWatch).on('change', browserSync.reload);
}

const dev = gulp.series(styles, gulp.parallel(serve, watch));
const build = gulp.series(styles);

exports.styles = styles;
exports.viewsPug = viewsPug;
exports.serve = serve;
exports.watch = watch;
exports.dev = dev;
exports.build = build;
exports.default = dev;
