<?php
/**
 * API para obtener posiciones de un rack específico
 * Devuelve todas las posiciones de wh_posicion filtradas por deposito_id y rack
 */

header('Content-Type: application/json');
require_once '../config/config.php';

// Verificar que el usuario esté autenticado
verificarLogin();

try {
    $db = getDBConnection();

    // Obtener parámetros
    $deposito_id = isset($_GET['deposito_id']) ? intval($_GET['deposito_id']) : 0;
    $rack = isset($_GET['rack']) ? intval($_GET['rack']) : 0;
    $ambiente_id = isset($_GET['ambiente_id']) ? intval($_GET['ambiente_id']) : 0;

    if ($deposito_id <= 0 || $rack <= 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Parámetros no válidos',
            'data' => []
        ]);
        exit;
    }

    // Consultar posiciones del rack
    $sql = "
        SELECT 
            p.*,
            d.nombre as deposito_nombre,
            a.nombre as ambiente_nombre
        FROM wh_posicion p
        LEFT JOIN wh_deposito d ON p.deposito_id = d.id
        LEFT JOIN wh_ambiente a ON p.ambiente_id = a.id
        WHERE p.deposito_id = :deposito_id
          AND p.rack = :rack
          AND p.activo = 1
    ";

    $params = [
        'deposito_id' => $deposito_id,
        'rack' => $rack
    ];

    if ($ambiente_id > 0) {
        $sql .= " AND p.ambiente_id = :ambiente_id";
        $params['ambiente_id'] = $ambiente_id;
    }

    $sql .= " ORDER BY p.columna, p.nivel, p.fondo";

    $stmt = $db->prepare($sql);
    $stmt->execute($params);

    $posiciones = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Limpiar espacios en nombres
    foreach ($posiciones as &$pos) {
        if (isset($pos['deposito_nombre'])) {
            $pos['deposito_nombre'] = trim($pos['deposito_nombre']);
        }
        if (isset($pos['ambiente_nombre'])) {
            $pos['ambiente_nombre'] = trim($pos['ambiente_nombre']);
        }
        if (isset($pos['title'])) {
            $pos['title'] = trim($pos['title']);
        }
    }

    $ambienteNombre = null;
    if ($ambiente_id > 0 && count($posiciones) > 0) {
        $ambienteNombre = $posiciones[0]['ambiente_nombre'] ?? null;
        if ($ambienteNombre) {
            $ambienteNombre = trim($ambienteNombre);
        }
    }

    // Log para debug
    $logAmbiente = $ambiente_id > 0 ? "ambiente $ambiente_id" : "sin filtro de ambiente";
    error_log("Posiciones obtenidas para depósito $deposito_id, rack $rack, $logAmbiente: " . count($posiciones));

    echo json_encode([
        'success' => true,
        'message' => 'Posiciones cargadas correctamente',
        'data' => $posiciones,
        'total' => count($posiciones),
        'deposito_id' => $deposito_id,
        'rack' => $rack,
        'ambiente_id' => $ambiente_id,
        'ambiente_nombre' => $ambienteNombre
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    error_log("Error obteniendo posiciones: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Error al cargar posiciones: ' . $e->getMessage(),
        'data' => []
    ]);
}
