/**
 * Renderizar layout visual del rack
 */

/**
 * Cargar y mostrar las posiciones de un rack
 */
function cargarPosicionesRack(depositoId, rack, ambienteId = "") {
  const layoutCard = document.getElementById("layout-card");
  const rackInfo = document.getElementById("rack-info");
  const posicionesContainer = document.getElementById("posiciones-container");

  // Mostrar el card
  layoutCard.style.display = "block";

  // Mostrar mensaje de carga
  rackInfo.className = "alert alert-info";
  rackInfo.innerHTML =
    '<i class="bi bi-hourglass-split"></i> <strong>Cargando posiciones...</strong>';
  posicionesContainer.innerHTML = "<p>Cargando datos...</p>";

  console.log(
    `📊 Cargando posiciones para Depósito ${depositoId}, Rack ${rack}, Ambiente ${
      ambienteId || "(todos)"
    }`
  );

  // Hacer petición AJAX
  const ambienteQuery = ambienteId ? `&ambiente_id=${ambienteId}` : "";

  fetch(
    `api/get_posiciones.php?deposito_id=${depositoId}&rack=${rack}${ambienteQuery}`
  )
    .then((response) => response.json())
    .then((data) => {
      console.log("📦 Posiciones recibidas:", data);

      if (data.success && data.data.length > 0) {
        // Mostrar información del rack
        rackInfo.className = "alert alert-success";
        rackInfo.innerHTML = `
          <i class="bi bi-check-circle"></i> 
          <strong>Rack ${rack}</strong> - 
          ${data.total} posiciones encontradas
          ${
            data.ambiente_nombre
              ? `<br><small>Ambiente: ${data.ambiente_nombre}</small>`
              : ""
          }
        `;

        // Renderizar el layout visual
        renderizarLayoutRack(data.data);

        console.log(`✅ ${data.total} posiciones cargadas correctamente`);
      } else {
        rackInfo.className = "alert alert-warning";
        rackInfo.innerHTML =
          '<i class="bi bi-exclamation-triangle"></i> <strong>No se encontraron posiciones para este rack</strong>';
        posicionesContainer.innerHTML = "<p>No hay datos disponibles</p>";
        console.warn("⚠️ No se encontraron posiciones");
      }
    })
    .catch((error) => {
      console.error("❌ Error cargando posiciones:", error);
      rackInfo.className = "alert alert-danger";
      rackInfo.innerHTML =
        '<i class="bi bi-x-circle"></i> <strong>Error al cargar posiciones</strong>';
      posicionesContainer.innerHTML = "<p>Error: " + error.message + "</p>";
    });
}

/**
 * Renderizar el layout visual del rack
 */
function renderizarLayoutRack(posiciones) {
  const container = document.getElementById("posiciones-container");

  // Renderizar por columnas, una debajo de otra
  let html = renderizarRackPorColumnas(posiciones);

  container.innerHTML = html;

  // Inicializar eventos de interacción para las posiciones
  inicializarEventosPosiciones();
}

/**
 * Renderizar rack por columnas (cada columna es un bloque)
 */
function renderizarRackPorColumnas(posiciones) {
  // Obtener columnas únicas ordenadas
  const columnas = [
    ...new Set(posiciones.map((p) => parseInt(p.columna))),
  ].sort((a, b) => a - b);

  let html = '<div class="rack-por-columnas">';

  // Renderizar cada columna como un bloque separado
  columnas.forEach((columna) => {
    const posicionesColumna = posiciones.filter(
      (p) => parseInt(p.columna) === columna
    );

    html += renderizarColumna(columna, posicionesColumna);
  });

  html += "</div>";

  return html;
}

/**
 * Renderizar una columna completa
 */
function renderizarColumna(columna, posiciones) {
  const letra = String.fromCharCode(64 + parseInt(columna)); // 1=A, 2=B, etc.

  // Separar por lado
  const lado1 = posiciones.filter((p) => p.lado === "A");
  const lado2 = posiciones.filter((p) => p.lado === "B");

  // Obtener niveles únicos (ordenados descendente)
  const niveles1 = [...new Set(lado1.map((p) => parseInt(p.nivel)))].sort(
    (a, b) => b - a
  );
  const niveles2 = [...new Set(lado2.map((p) => parseInt(p.nivel)))].sort(
    (a, b) => b - a
  );

  let html = `
    <div class="columna-block">
      <h4 class="columna-title">LADO 1 - Columna ${letra}</h4>
      <div class="rack-grid">
        <table class="rack-table rack-table-compact">
          <thead>
            <tr>
              <th>Ambiente</th>
              <th>Nivel</th>
              <th colspan="3">A</th>
            </tr>
            <tr>
              <th></th>
              <th></th>
              <th class="fondo-header">A</th>
              <th class="fondo-header">B</th>
              <th class="fondo-header">C</th>
            </tr>
          </thead>
          <tbody>
  `;

  // Renderizar niveles para LADO 1
  niveles1.forEach((nivel) => {
  const ambienteFila = obtenerAmbienteFila(lado1, nivel);
  html += `<tr><td class="ambiente-cell">${ambienteFila}</td><td class="nivel-cell">${nivel}</td>`;

    [1, 2, 3].forEach((fondo) => {
      const pos = lado1.find(
        (p) => parseInt(p.nivel) === nivel && parseInt(p.fondo) === fondo
      );
      html += renderizarPosicion(pos, nivel, fondo);
    });

    html += "</tr>";
  });

  html += `
          </tbody>
        </table>
      </div>
  `;

  // Si hay LADO 2, renderizarlo
  if (lado2.length > 0) {
    html += `
      <h4 class="columna-title">LADO 2 - Columna ${letra}</h4>
      <div class="rack-grid">
        <table class="rack-table rack-table-compact">
          <thead>
            <tr>
              <th>Ambiente</th>
              <th>Nivel</th>
              <th colspan="3">B</th>
            </tr>
            <tr>
              <th></th>
              <th></th>
              <th class="fondo-header">A</th>
              <th class="fondo-header">B</th>
              <th class="fondo-header">C</th>
            </tr>
          </thead>
          <tbody>
    `;

    // Renderizar niveles para LADO 2
    niveles2.forEach((nivel) => {
  const ambienteFila = obtenerAmbienteFila(lado2, nivel);
  html += `<tr><td class="ambiente-cell">${ambienteFila}</td><td class="nivel-cell">${nivel}</td>`;

      [1, 2, 3].forEach((fondo) => {
        const pos = lado2.find(
          (p) => parseInt(p.nivel) === nivel && parseInt(p.fondo) === fondo
        );
        html += renderizarPosicion(pos, nivel, fondo);
      });

      html += "</tr>";
    });

    html += `
          </tbody>
        </table>
      </div>
    `;
  }

  html += `</div>`;

  return html;
}

function obtenerAmbienteFila(posiciones, nivel) {
  const posicion = posiciones.find(
    (p) => parseInt(p.nivel) === parseInt(nivel, 10)
  );

  if (posicion && posicion.ambiente_nombre) {
    return escapeHTML(posicion.ambiente_nombre.trim());
  }

  return "—";
}

function escapeHTML(str) {
  return str
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#39;");
}

/**
 * Renderizar una posición individual
 */
function renderizarPosicion(pos, nivel, fondo) {
  if (pos) {
    const claseOcupado = parseInt(pos.ocupado) ? "ocupado" : "libre";
    const clasePicked = parseInt(pos.picked) ? "picked" : "";
    const tooltip = `${pos.code_full}
Ambiente: ${pos.ambiente_nombre}
Capacidad: ${pos.capacidad_pallets} pallets
Ocupado: ${pos.ocupado ? "Sí" : "No"}`;

    return `
      <td class="posicion-cell ${claseOcupado} ${clasePicked}" 
          title="${tooltip}"
          data-id="${pos.id}"
          data-code="${pos.code ?? ""}">
        
      </td>
    `;
  } else {
    return '<td class="posicion-cell vacio"></td>';
  }
}

/**
 * Ocultar el card de layout
 */
function ocultarLayout() {
  const layoutCard = document.getElementById("layout-card");
  if (layoutCard) {
    layoutCard.style.display = "none";
  }
  console.log("🚫 Layout ocultado");
}

/**
 * Inicializa los eventos de clic sobre cada posición renderizada
 */
function inicializarEventosPosiciones() {
  const celdas = document.querySelectorAll(
    ".posicion-cell[data-id]:not(.vacio)"
  );

  celdas.forEach((celda) => {
    celda.addEventListener("click", manejarClickPosicion);
  });
}

/**
 * Maneja el clic sobre una posición específica
 */
function manejarClickPosicion(event) {
  const celda = event.currentTarget;
  const posicionId = celda.dataset.id;
  const posicionCode = celda.dataset.code || "Sin código";

  if (!posicionId) {
    return;
  }

  if (navigator.vibrate) {
    navigator.vibrate(15);
  }

  mostrarModalPosicion(posicionId, posicionCode);
}

/**
 * Muestra el modal con la información básica de la posición seleccionada
 */
function mostrarModalPosicion(posicionId, posicionCode) {
  const modalElement = document.getElementById("posicionModal");
  if (!modalElement) {
    console.warn("⚠️ No se encontró el modal de posición en el DOM");
    return;
  }

  const idField = modalElement.querySelector('[data-posicion-field="id"]');
  const codeField = modalElement.querySelector('[data-posicion-field="code"]');
  const codeHiddenField = modalElement.querySelector(
    '[data-posicion-field="code-hidden"]'
  );
  const form = modalElement.querySelector("#posicion-form");

  if (form) {
    form.reset();
  }

  if (idField) {
    if ("value" in idField) {
      idField.value = posicionId;
    } else {
      idField.textContent = posicionId;
    }
  }

  if (codeField) {
    if ("value" in codeField) {
      codeField.value = posicionCode;
    } else {
      codeField.textContent = posicionCode;
    }
  }

  if (codeHiddenField) {
    codeHiddenField.value = posicionCode;
  }

  const modalInstance = bootstrap.Modal.getOrCreateInstance(modalElement);
  modalInstance.show();
}
