<?php
/**
 * SOL - Sistema de Operaciones Logísticas
 * Front Controller con rutas derivadas del menú (Single Source of Truth)
 *
 * - Las rutas se infieren desde config/menu.php vía app/Support/MenuRouter.php
 * - / y /login (GET)  → views/auth/login.php
 * - /login (POST)     → backend/auth_login.php
 * - /logout           → destruye sesión y vuelve a /login
 *
 * Cualquier URL presente en config/menu.php (con 'url' no-nula) se resuelve a:
 *   views/<url>/index.php   (p.ej. '/parametros/productos' → 'views/parametros/productos/index.php')
 */

declare(strict_types=1);

try {
    // -------------------------------------------------------
    // 1) Cargar configuración base del proyecto
    // -------------------------------------------------------
    $configPath = __DIR__ . '/../config/config.php';
    if (!is_file($configPath)) {
        throw new RuntimeException("No se encontró config.php en: {$configPath}");
    }
    require_once $configPath;

    // -------------------------------------------------------
    // 2) Cargar helpers de ruteo basados en el menú
    // -------------------------------------------------------
    $menuRouter = BASE_PATH . '/app/Support/MenuRouter.php';
    if (!is_file($menuRouter)) {
        throw new RuntimeException("Falta app/Support/MenuRouter.php en: {$menuRouter}");
    }
    require_once $menuRouter;

    // -------------------------------------------------------
    // 3) Normalizar request
    // -------------------------------------------------------
    $rawPath = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?? '/';
    $rawPath = rtrim($rawPath, "/");
    // Siempre trabajar con path que comienza con '/'
    $path = $rawPath === '' ? '/' : $rawPath;

    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

    // Helper para redirección segura
    $go = function (string $to) {
        if (!headers_sent()) {
            header('Location: ' . $to);
        } else {
            echo '<script>location.href=' . json_encode($to) . ';</script>';
        }
        exit;
    };

    // -------------------------------------------------------
    // 4) Casos especiales: raíz, login, logout
    // -------------------------------------------------------
    if ($path === '/' || $path === '/login') {
        if ($method === 'POST') {
            // Procesa login internamente (sin exponer /backend)
            $loginPath = BASE_PATH . '/backend/auth_login.php';
            if (!is_file($loginPath)) {
                throw new RuntimeException("Backend de login no encontrado: {$loginPath}");
            }
            require $loginPath;
            exit;
        }
        // GET: mostrar login
        $viewLogin = BASE_PATH . '/views/auth/login.php';
        if (!is_file($viewLogin)) {
            throw new RuntimeException("Vista de login no encontrada: {$viewLogin}");
        }
        require $viewLogin;
        exit;
    }

    if ($path === '/logout') {
        // Solo por GET/POST destruimos sesión y regresamos al login
        $_SESSION = [];
        if (session_status() === PHP_SESSION_ACTIVE) {
            if (ini_get('session.use_cookies')) {
                $params = session_get_cookie_params();
                setcookie(session_name(), '', time() - 42000, $params['path'], $params['domain'], $params['secure'], $params['httponly']);
            }
            session_destroy();
        }
        $go(url('/login') . '?e=' . urlencode('Sesión cerrada.'));
    }

    // -------------------------------------------------------
    // 4.5) Compatibilidad: antigua ruta /home
    // -------------------------------------------------------
    if ($path === '/home') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        route_require_view('views/home/index.php');
        exit;
    }

    // -------------------------------------------------------
    // 4.6) Rutas especiales para PDFs de preparación
    // -------------------------------------------------------
    if ($path === '/salidas/preparacion/doc' && $method === 'GET') {
        // Gate de sesión para esta ruta
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        
        // Determinar tipo de PDF
        if (isset($_GET['movimientos']) && $_GET['movimientos'] === '1') {
            // PDF de movimientos de preparación
            $movimientosPath = BASE_PATH . '/views/salidas/preparacion/doc_movimientos.php';
            if (!is_file($movimientosPath)) {
                throw new RuntimeException("Vista de movimientos PDF no encontrada: {$movimientosPath}");
            }
            require $movimientosPath;
            exit;
        } elseif (isset($_GET['reposicion']) && $_GET['reposicion'] === '1') {
            // PDF de reposición para montacarguista
            $reposicionPath = BASE_PATH . '/views/salidas/preparacion/doc_reposicion.php';
            if (!is_file($reposicionPath)) {
                throw new RuntimeException("Vista de reposición PDF no encontrada: {$reposicionPath}");
            }
            require $reposicionPath;
            exit;
        } else {
            // PDF documento normal
            $docPath = BASE_PATH . '/views/salidas/preparacion/doc.php';
            if (!is_file($docPath)) {
                throw new RuntimeException("Vista de documento no encontrada: {$docPath}");
            }
            require $docPath;
            exit;
        }
    }

    // -------------------------------------------------------
    // 4.7) Reportes -> Quiebre de stock (PDF)
    // -------------------------------------------------------
    if ($path === '/reportes/stock_quiebre/pdf' && $method === 'GET') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        $pdfPath = BASE_PATH . '/views/reportes/stock_quiebre/pdf.php';
        if (!is_file($pdfPath)) {
            throw new RuntimeException("Vista de PDF no encontrada: {$pdfPath}");
        }
        require $pdfPath;
        exit;
    }

    if ($path === '/reportes/pedidos/pdf' && $method === 'GET') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        $pdfPath = BASE_PATH . '/views/reportes/pedidos/pdf.php';
        if (!is_file($pdfPath)) {
            throw new RuntimeException("Vista de PDF no encontrada: {$pdfPath}");
        }
        require $pdfPath;
        exit;
    }

    if ($path === '/reportes/ingresos/pdf' && $method === 'GET') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        $pdfPath = BASE_PATH . '/views/reportes/ingresos/pdf.php';
        if (!is_file($pdfPath)) {
            throw new RuntimeException("Vista de PDF no encontrada: {$pdfPath}");
        }
        require $pdfPath;
        exit;
    }

    if ($path === '/reportes/salidas/pdf' && $method === 'GET') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        $pdfPath = BASE_PATH . '/views/reportes/salidas/pdf.php';
        if (!is_file($pdfPath)) {
            throw new RuntimeException("Vista de PDF no encontrada: {$pdfPath}");
        }
        require $pdfPath;
        exit;
    }

    if ($path === '/reportes/pedidos_hora/pdf' && $method === 'GET') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        $pdfPath = BASE_PATH . '/views/reportes/pedidos_hora/pdf.php';
        if (!is_file($pdfPath)) {
            throw new RuntimeException("Vista de PDF no encontrada: {$pdfPath}");
        }
        require $pdfPath;
        exit;
    }

    if ($path === '/reportes/moviles_disponibles/pdf' && $method === 'GET') {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
        $pdfPath = BASE_PATH . '/views/reportes/moviles_disponibles/pdf.php';
        if (!is_file($pdfPath)) {
            throw new RuntimeException("Vista de PDF no encontrada: {$pdfPath}");
        }
        require $pdfPath;
        exit;
    }

    // -------------------------------------------------------
    // 5) Rutas derivadas del menú (Single Source of Truth)
    // -------------------------------------------------------
    // Construye mapa: '/ruta' => ['view' => 'views/.../index.php', 'gate' => bool, 'exists' => bool, 'id'=>..., 'label'=>...]
    $routes = route_map_from_menu(null, true);

    $entry = $routes[$path] ?? null;
    if ($entry === null) {
        // 404 si no existe en el menú (o no visible)
        http_response_code(404);
        echo "<h1>404 - Página no encontrada</h1>";
        echo "<p>La ruta <code>" . htmlspecialchars($path) . "</code> no está definida en el menú (config/menu.php).</p>";
        exit;
    }

    // 5.1) Gate de sesión (por defecto TRUE, salvo override en el menú)
    $gate = (bool)($entry['gate'] ?? true);
    if ($gate) {
        if (empty($_SESSION['usuario_id'])) {
            $go(url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
        }
    }

    // 5.2) Validar existencia de la vista y renderizar
    if (isset($entry['exists']) && $entry['exists'] === false) {
        http_response_code(404);
        echo "<h1>404 - Vista no encontrada</h1>";
        echo "<p>Se esperaba el archivo <code>" . htmlspecialchars($entry['view']) . "</code> para la ruta <code>" . htmlspecialchars($path) . "</code>.</p>";
        exit;
    }

    // Carga de la vista (con verificación)
    route_require_view($entry['view']);
    exit;

} catch (Throwable $e) {
    // -------------------------------------------------------
    // 6) Manejo de errores (debug friendly)
    // -------------------------------------------------------
    $debug = function_exists('env') ? (bool)env('APP_DEBUG', false) : true;
    if ($debug) {
        http_response_code(500);
        echo "<h1>500 - Error</h1>";
        echo "<pre style='white-space:pre-wrap'>";
        echo "Mensaje: " . htmlspecialchars($e->getMessage()) . "\n";
        echo "Archivo: " . htmlspecialchars($e->getFile()) . ":" . (int)$e->getLine() . "\n\n";
        echo htmlspecialchars($e->getTraceAsString());
        echo "</pre>";
    } else {
        http_response_code(500);
        echo "<h1>500 - Error interno</h1>";
    }
    if (function_exists('error_log')) {
        error_log("[SOL/index] " . $e->getMessage() . " @ " . $e->getFile() . ":" . $e->getLine());
    }
}

exit;
