<?php
/**
 * SOL - Sistema de Operaciones Logísticas
 * Front Controller con manejo robusto de errores
 * - /           y /login  → views/auth/login.php  (GET), procesa login en POST
 * - /home                   → views/home/index.php
 * - /logout                 → destruye sesión y vuelve a /login
 * - /ingresos               → (stub) cargará views/ingresos/index.php en el próximo paso
 * - /parametros/productos   → views/parametros/productos/index.php (P1)
 */

declare(strict_types=1);

try {
    // Carga config (autoload + dotenv + helpers + sesión + constants)
    $configPath = __DIR__ . '/../config/config.php';
    if (!is_file($configPath)) {
        throw new RuntimeException("No se encontró config.php en: {$configPath}");
    }
    require_once $configPath;

    $uri    = trim(parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?? '/', '/');
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

    // Helper interno para cargar vistas de forma segura
    $loadView = function (string $relPath): void {
        $full = BASE_PATH . '/' . ltrim($relPath, '/');
        if (!is_file($full)) {
            throw new RuntimeException("Vista no encontrada: {$full}");
        }
        require $full;
    };

    switch ($uri) {
        case '':
            if ($method === 'POST') {
                // Procesa login internamente (sin exponer /backend)
                $loginPath = BASE_PATH . '/backend/auth_login.php';
                if (!is_file($loginPath)) {
                    throw new RuntimeException("Backend de login no encontrado: {$loginPath}");
                }
                require $loginPath;
                exit;
            }
            $loadView('views/auth/login.php');
            break;

        case 'login':
            if ($method === 'POST') {
                $loginPath = BASE_PATH . '/backend/auth_login.php';
                if (!is_file($loginPath)) {
                    throw new RuntimeException("Backend de login no encontrado: {$loginPath}");
                }
                require $loginPath;
                exit;
            }
            $loadView('views/auth/login.php');
            break;

        case 'home':
            // Gate opcional: exige sesión para home
            if (empty($_SESSION['usuario_id'])) {
                if (!headers_sent()) {
                    header('Location: ' . url('/login') . '?e=' . urlencode('Debe autenticarse para ingresar.'));
                } else {
                    echo '<script>location.href=' . json_encode(url('/login?e=Debe%20autenticarse%20para%20ingresar.')) . ';</script>';
                }
                exit;
            }
            $loadView('views/home/index.php');
            break;

        case 'parametros/clientes':
            // La vista realiza su propio gate de sesión (P1)
            $loadView('views/parametros/clientes/index.php');
            break;
        case 'parametros/clases':
            // La vista realiza su propio gate de sesión (P1)
            $loadView('views/parametros/clases/index.php');
            break;
        case 'parametros/grupos':
            // La vista realiza su propio gate de sesión (P1)
            $loadView('views/parametros/grupos/index.php');
            break;
        case 'parametros/productos':
            // La vista realiza su propio gate de sesión (P1)
            $loadView('views/parametros/productos/index.php');
            break;

        case 'logout':
            // Solo por GET/POST destruimos sesión y regresamos al login
            $_SESSION = [];
            if (session_status() === PHP_SESSION_ACTIVE) {
                if (ini_get('session.use_cookies')) {
                    $params = session_get_cookie_params();
                    setcookie(session_name(), '', time() - 42000, $params['path'], $params['domain'], $params['secure'], $params['httponly']);
                }
                session_destroy();
            }
            if (!headers_sent()) {
                header('Location: ' . url('/login') . '?e=' . urlencode('Sesión cerrada.'));
            } else {
                echo '<script>location.href=' . json_encode(url('/login?e=Sesión%20cerrada.')) . ';</script>';
            }
            exit;

        case 'ingresos':
            // En el próximo paso crearemos: views/ingresos/index.php
            $loadView('views/ingresos/index.php');
            break;

        default:
            http_response_code(404);
            echo "<h1>404 - Página no encontrada</h1>";
            break;
    }

} catch (Throwable $e) {
    // Si estás en debug (APP_DEBUG=true), muestra el error exacto
    $debug = function_exists('env') ? (bool)env('APP_DEBUG', false) : true;
    if ($debug) {
        http_response_code(500);
        echo "<h1>500 - Error</h1>";
        echo "<pre style='white-space:pre-wrap'>";
        echo "Mensaje: " . htmlspecialchars($e->getMessage()) . "\n";
        echo "Archivo: " . htmlspecialchars($e->getFile()) . ":" . (int)$e->getLine() . "\n\n";
        echo htmlspecialchars($e->getTraceAsString());
        echo "</pre>";
    } else {
        http_response_code(500);
        echo "<h1>500 - Error interno</h1>";
    }
    if (function_exists('error_log')) {
        error_log("[SOL/index] " . $e->getMessage() . " @ " . $e->getFile() . ":" . $e->getLine());
    }
}

exit;
