function initializeBaseDataTable(config) {
  const {
    tableId,
    ajaxUrl,
    columns,
    editorFields = false, // Si es falso, desactiva el Editor
    order = [[0, 'desc']], // Orden predeterminado
    ordering = true,
    search = '',
    tableButtons = [], // Botones personalizados adicionales
    editorButtons = ["create", "edit", "remove"], // Botones del editor
    serverSide = false, // Configuración de serverSide
  } = config;

  let editor = null; // Inicializamos editor como null por defecto

  // Solo inicializar el editor si editorFields está definido
  if (editorFields) {
    editor = new DataTable.Editor({
      ajax: ajaxUrl,
      fields: editorFields,
      table: `#${tableId}`,
      i18n: {
        create: {
          button: "Nuevo",
          title: "Crear un nuevo registro",
          submit: "Crear",
        },
        edit: {
          button: "Modificar",
          title: "Modificar registro",
          submit: "Actualizar",
        },
        remove: {
          button: "Eliminar",
          title: "Eliminar registro",
          submit: "Eliminar",
          confirm: {
            _: "¿Estás seguro de que quieres eliminar %d registros?",
            1: "¿Estás seguro de que quieres eliminar este registro?",
          },
        },
        error: {
          system: "Ocurrió un error, por favor contacta al administrador del sistema.",
        },
        multi: {
          title: "Valores múltiples",
          info: "Los elementos seleccionados contienen valores diferentes para este campo. Para editar y establecer todos los elementos en el mismo valor, haz clic aquí; de lo contrario, conservarán sus valores individuales.",
          restore: "Cancelar cambios",
          noMulti: "Este campo se puede editar individualmente, pero no como parte de un grupo.",
        },
        datetime: {
          previous: "Anterior",
          next: "Siguiente",
          months: [
            "Enero",
            "Febrero",
            "Marzo",
            "Abril",
            "Mayo",
            "Junio",
            "Julio",
            "Agosto",
            "Septiembre",
            "Octubre",
            "Noviembre",
            "Diciembre",
          ],
          weekdays: ["Dom", "Lun", "Mar", "Mié", "Jue", "Vie", "Sáb"],
        },
      },
    });
  }

  const buttons = [];
  if (editor) {
    if (editorButtons.includes("create"))
      buttons.push({ extend: "create", editor });
    if (editorButtons.includes("edit"))
      buttons.push({ extend: "edit", editor });
    if (editorButtons.includes("remove"))
      buttons.push({ extend: "remove", editor });
  }

  buttons.push({
    extend: "collection",
    text: "Exportar",
    buttons: [
      { extend: "copyHtml5", text: "Copiar", footer: true },
      // { extend: "excelHtml5", footer: true },
      {
        extend: "csvHtml5",
        text: "Excel",
        footer: true,
        fieldSeparator: ";"
      },
      { extend: "csvHtml5", text: "CSV", footer: true },
      { extend: "pdfHtml5", footer: true },
    ],
  });

  buttons.push(...tableButtons);

  // Actualizar el `dom` para incluir los botones de exportar
  const table = $(`#${tableId}`).DataTable({
    ajax: ajaxUrl,
    serverSide: serverSide,
    columns: columns,
    dom: "<'row'<'col-sm-12 col-md-12 col-lg-1'l><'col-sm-1 col-md-12 col-lg-8'B>" +
    "<'col-sm-12 col-md-12 col-lg-3'>>" +
    "<'row'<'col-sm-12'tr>>" +
    "<'row'<'col-sm-12 col-md-12 col-lg-5'><'col-sm-12 col-md-12 col-lg-7'p>>",
    language: {
      sProcessing: "Procesando...",
      sLengthMenu: "_MENU_",
      sZeroRecords: "No se encontraron resultados",
      sEmptyTable: "Ningún dato disponible en esta tabla",
      sInfo:
        "Mostrando registros del _START_ al _END_ de un total de _TOTAL_ registros",
      sInfoEmpty: "Mostrando registros del 0 al 0 de un total de 0 registros",
      sInfoFiltered: "(filtrado de un total de _MAX_ registros)",
      search: "",
      searchPlaceholder: "Buscar",
      processing: "Procesando, por favor espere..."
    },
    buttons: buttons,
    ordering: ordering,
    order: order,
    search: search,
    select: true,
    // Forzar siempre el orden por la primera columna en descendente
    orderFixed: [[0, 'desc']],

    // Desactivar el orden en las cabeceras de las columnas
    columnDefs: [
      { targets: '_all', orderable: false }
    ],

    initComplete: function () {
      this.api()
        .columns()
        .every(function () {
          let column = this;
          let title = column.header().textContent;

          // Create input element
          let input = document.createElement("input");
          input.placeholder = title;
          column.header().replaceChildren(input);

          // Event listener for user input
          input.addEventListener("keyup", () => {
            if (column.search() !== this.value) {
              column.search(input.value).draw();
            }
          });
        });
    },
  });


  if (editor) {
    $(`#${tableId}`).on("click", "tbody td:not(:first-child)", function () {
      editor.inline(this, { onBlur: "submit" });
    });
  }

  return { editor, table };
}
