/**
 * SOL · DataTables + Editor (Base)
 * Exponen: window.initializeBaseDataTable(config)
 *
 * Requisitos de carga (ya configurados en getDataTablesConfig()):
 *  - DataTables core + Bootstrap 5
 *  - Buttons (+ HTML5)
 *  - JSZip + pdfmake (+ vfs_fonts)
 *  - Select + DateTime (si se usan)
 *  - Editor (local; comercial)
 *
 * Nota:
 *  - Esta función NO llama a ninguna API por sí sola.
 *  - Se usará desde scripts específicos por página (p. ej. productosEditor.js).
 */

(function (global) {
  'use strict';

  // Pequeñas utilidades
  function warnMissing(what) {
    // eslint-disable-next-line no-console
    console.warn('[SOL:DT] Falta dependencia:', what);
  }

  // Detectar jQuery + DataTables + Editor
  var $ = global.jQuery || global.$;
  var DataTableNS = global.DataTable || (global.$ && global.$.fn && global.$.fn.dataTable);
  var EditorCtor = (global.DataTable && global.DataTable.Editor)
                || (global.$ && global.$.fn && global.$.fn.dataTable && global.$.fn.dataTable.Editor)
                || null;

  if (!$) warnMissing('jQuery');
  if (!DataTableNS) warnMissing('DataTables');
  if (!EditorCtor)  warnMissing('Editor (DataTables Editor)');

  /**
   * Inicializa DataTable (+ Editor opcional) en una tabla.
   * @param {Object} config
   *   - tableId: string (ID del <table> sin '#') [requerido]
   *   - ajaxUrl: string (endpoint para DataTables/Editor) [recomendado]
   *   - columns: Array<DataTables.ColumnDefs> [requerido]
   *   - editorFields: Array<Editor.Field> | false (si false, no se crea Editor)
   *   - order: [[idx, 'asc'|'desc']] (por defecto [[0,'desc']])
   *   - ordering: boolean (true)
   *   - search: string ('')
   *   - tableButtons: Array (botones extra)
   *   - editorButtons: Array<string> (["create","edit","remove"])
   *   - serverSide: boolean (false)
   *   - dataTableOptions: Object (opciones extra para $.DataTable)
   *   - editorOptions: Object (opciones extra para new Editor)
   *   - language: Object (override de textos)
   * @returns {{ editor: any|null, table: any }}
   */
  function initializeBaseDataTable(config) {
    if (!config || !config.tableId) {
      throw new Error('[SOL:DT] tableId es requerido.');
    }
    if (!$ || !$.fn || !$.fn.DataTable) {
      throw new Error('[SOL:DT] jQuery DataTables no está disponible.');
    }

    var tableId       = config.tableId;
    var ajaxUrl       = config.ajaxUrl || null;
    var columns       = config.columns || [];
    var editorFields  = (config.editorFields === false) ? false : (config.editorFields || false);
    var order         = config.order || [[0, 'desc']];
    var ordering      = (typeof config.ordering === 'boolean') ? config.ordering : true;
    var search        = (typeof config.search === 'string') ? config.search : '';
    var tableButtons  = Array.isArray(config.tableButtons) ? config.tableButtons.slice() : [];
    var editorButtons = Array.isArray(config.editorButtons) ? config.editorButtons : ['create','edit','remove'];
    var serverSide    = !!config.serverSide;
    var dtExtra       = config.dataTableOptions || {};
    var edExtra       = config.editorOptions || {};
    var langOverride  = config.language || {};

    var editor = null;

    // -------------------------------------------------------
    // Editor (opcional)
    // -------------------------------------------------------
    if (editorFields && EditorCtor) {
      var baseEditorOpts = {
        ajax: ajaxUrl,
        fields: editorFields,
        table: '#' + tableId,
        i18n: {
          create: { button: 'Nuevo',     title: 'Crear un nuevo registro', submit: 'Crear' },
          edit:   { button: 'Modificar', title: 'Modificar registro',      submit: 'Actualizar' },
          remove: {
            button: 'Eliminar', title: 'Eliminar registro', submit: 'Eliminar',
            confirm: { _: '¿Seguro que deseas eliminar %d registros?', 1: '¿Seguro que deseas eliminar este registro?' }
          },
          error: { system: 'Ocurrió un error, por favor contacta al administrador.' },
          multi: {
            title: 'Valores múltiples',
            info: 'Los elementos seleccionados contienen valores diferentes para este campo.',
            restore: 'Cancelar cambios',
            noMulti: 'Este campo se edita individualmente, no en grupo.'
          },
          datetime: {
            previous: 'Anterior', next: 'Siguiente',
            months: ['Enero','Febrero','Marzo','Abril','Mayo','Junio','Julio','Agosto','Septiembre','Octubre','Noviembre','Diciembre'],
            weekdays: ['Dom','Lun','Mar','Mié','Jue','Vie','Sáb']
          }
        }
      };
      // Merge superficial de opciones extra del editor
      var editorOpts = Object.assign({}, baseEditorOpts, edExtra || {});
      editor = new EditorCtor(editorOpts);
    }

    // -------------------------------------------------------
    // Botonera
    // -------------------------------------------------------
    var buttons = [];
    if (editor) {
      if (editorButtons.includes('create')) buttons.push({ extend: 'create', editor: editor });
      if (editorButtons.includes('edit'))   buttons.push({ extend: 'edit',   editor: editor });
      if (editorButtons.includes('remove')) buttons.push({ extend: 'remove', editor: editor });
    }

    // Exportación
    buttons.push({
      extend: 'collection',
      text: 'Exportar',
      buttons: [
        { extend: 'copyHtml5', text: 'Copiar', footer: true },
        // { extend: 'excelHtml5', footer: true }, // si prefieres Excel nativo
        { extend: 'csvHtml5',   text: 'Excel', footer: true, fieldSeparator: ';' }, // "Excel" vía CSV con ';'
        { extend: 'csvHtml5',   text: 'CSV',   footer: true },
        { extend: 'pdfHtml5',                   footer: true }
      ]
    });

    // Agregar extras de la página
    buttons.push.apply(buttons, tableButtons);

    // -------------------------------------------------------
    // Idioma base (ES) + override
    // -------------------------------------------------------
    var language = Object.assign({
      sProcessing:   'Procesando...',
      sLengthMenu:   '_MENU_',
      sZeroRecords:  'No se encontraron resultados',
      sEmptyTable:   'Ningún dato disponible en esta tabla',
      sInfo:         'Mostrando _START_ a _END_ de _TOTAL_',
      sInfoEmpty:    'Mostrando 0 a 0 de 0',
      sInfoFiltered: '(filtrado de _MAX_)',
      search:        '',
      searchPlaceholder: 'Buscar',
      processing:    'Procesando, por favor espere...'
    }, langOverride || {});

    // -------------------------------------------------------
    // DOM layout con botones + tabla responsive
    // -------------------------------------------------------
    var dom = "" +
      "<'row'<'col-12 col-lg-1'l><'col-12 col-lg-8'B><'col-12 col-lg-3'f>>" +
      "<'row'<'col-12'tr>>" +
      "<'row'<'col-12 col-lg-5'i><'col-12 col-lg-7'p>>";

    // -------------------------------------------------------
    // Inicializar DataTable
    // -------------------------------------------------------
    var dtOpts = Object.assign({
      ajax: ajaxUrl,
      serverSide: serverSide,
      columns: columns,
      dom: dom,
      language: language,
      buttons: buttons,
      ordering: ordering,
      order: order,
      search: search,
      orderFixed: [[0, 'desc']],
      columnDefs: [
        { targets: '_all', orderable: false }
      ],
      // Si quieres filtros en los headers, activa este bloque en tu script por página
      // y asegúrate de que el thead exista o se construya con "columns.title"
      // initComplete: function () {
      //   var api = this.api();
      //   api.columns().every(function () {
      //     var column = this;
      //     var title = column.header().textContent;
      //     var input = document.createElement('input');
      //     input.placeholder = title;
      //     column.header().replaceChildren(input);
      //     input.addEventListener('keyup', function () {
      //       if (column.search() !== input.value) column.search(input.value).draw();
      //     });
      //   });
      // }
    }, dtExtra || {});

    var table = $('#' + tableId).DataTable(dtOpts);

    // -------------------------------------------------------
    // Edición inline (si hay Editor)
    // -------------------------------------------------------
    if (editor) {
      $('#' + tableId).on('click', 'tbody td:not(:first-child)', function () {
        editor.inline(this, { onBlur: 'submit' });
      });
    }

    return { editor: editor, table: table };
  }

  // Exponer globalmente
  global.initializeBaseDataTable = initializeBaseDataTable;

})(window);
