// public/page-scripts/pages/control/conteo.js
// -----------------------------------------------------------
// Control · Conteo por ubicación (layout externo + modal de conteo)
// Requiere:
//   - public/page-scripts/warehouse/layout-core.js
//   - public/page-scripts/warehouse/render-microtiles.js
//   - public/page-scripts/warehouse/render-shelf-strips.js
//   - public/page-scripts/warehouse/render-column-cards.js
// -----------------------------------------------------------
(function () {
  "use strict";
  $("#flt-fecha").attr("name", "fecha");
  function qs(name) {
    const m = new RegExp("[?&]" + name + "=([^&#]*)").exec(
      window.location.search
    );
    return m ? decodeURIComponent(m[1].replace(/\+/g, " ")) : "";
  }

  function toggleExpectedColumns(show) {
    const $table = $("#tblConteoItems");
    // thead
    $table.find("thead th:nth-child(5), thead th:nth-child(6)").toggle(!!show);
    // body
    $table.find("tbody tr").each(function () {
      $(this).find("td:nth-child(5), td:nth-child(6)").toggle(!!show);
    });
    // tfoot totales esperadas (ajusta el selector a tu markup real)
    $("#t_uc_exp, #t_uv_exp").closest("th,td").toggle(!!show);
  }

  // Notificador por defecto (sobrescribible si hay uno global)
  const notify =
    window.notify ||
    function (type, title, message) {
      const msg = `[${type?.toUpperCase?.() || "INFO"}] ${title || ""} ${
        message || ""
      }`.trim();
      console[type === "error" ? "error" : "log"](msg);
    };

  $(function () {
    if (!window.WarehouseLayout) {
      console.error(
        "[conteo] WarehouseLayout no está disponible. ¿Incluiste layout-core.js?"
      );
      return;
    }

    // Usar ruta ABSOLUTA (vhost apunta a public/)
    const API = "/api/control/conteo_layout.php";

    // === UI refs =============================================================
    const $modal = $("#mdlConteoPos");
    const $modalSubtitle = $("#mdl-conteo-pos-subtitle");
    const $modalHeader = $("#mdl-conteo-pos-header");
    const $tblBody = $("#tblConteoItems tbody");
    const $btnGuardar = $("#btnGuardarConteo");
    let conteoPayload = null;

    // helpers numeritos
    function num0(v) {
      const n = parseInt(v, 10);
      return Number.isFinite(n) ? n : 0;
    }
    function fmt(n) {
      return Number(n || 0).toLocaleString("es-PY");
    }

    // ---------- Mostrar modal seguro ----------
    function showModalSafe() {
      try {
        if (window.bootstrap && bootstrap.Modal) {
          new bootstrap.Modal($modal[0]).show();
          return true;
        }
        if (typeof $modal.modal === "function") {
          // fallback (BS4)
          $modal.modal("show");
          return true;
        }
      } catch (e) {
        console.error("[conteo] No se pudo abrir el modal:", e);
      }
      console.error("[conteo] Bootstrap Modal no disponible.");
      return false;
    }

    // ---------- Interceptor CAPTURA ----------
    document.addEventListener(
      "click",
      function (e) {
        const el =
          e.target &&
          e.target.closest(
            ".warehouse-layout .pos-tile, .warehouse-layout [data-fondo-id]"
          );
        if (!el) return;
        e.preventDefault();
        e.stopImmediatePropagation();
        e.stopPropagation();
        openConteoFromEl(el);
      },
      true // CAPTURE
    );

    // Delegado backup (por si alguna vez se saca el interceptor)
    $(document).on(
      "click",
      ".warehouse-layout .pos-tile, .warehouse-layout [data-fondo-id]",
      function (e) {
        e.preventDefault();
        e.stopImmediatePropagation();
        e.stopPropagation();
        openConteoFromEl(this);
      }
    );

    // ---------- Lógica de modal ----------
    function openConteoFromEl(el) {
      const $el = $(el);

      // 1) intentar por data-fondo-id
      let fondoId = $el.data("fondo-id");

      // 2) fallback: parsear del title "ID fondo: N"
      if (!fondoId) {
        const t = ($el.attr("title") || "").toString();
        const m = t.match(/ID\s*fondo:\s*(\d+)/i);
        if (m) fondoId = parseInt(m[1], 10);
      }

      // 3) si sigue sin ID, no podemos abrir modal
      if (!fondoId) return;

      // codeFull opcional: intenta atributo o compón desde title (Ubicación: ...)
      const codeFull =
        $el.data("code-full") ||
        $el.attr("data-code-full") ||
        (($el.attr("title") || "").match(/Ubicación:\s*([^\n]+)/) || [])[1] ||
        "";

      const fechaSel = $("#flt-fecha").val() || "";

      // Estado inicial + spinner
      $modalSubtitle.text(codeFull || "Posición");
      $modalHeader.html(
        '<div class="text-muted small">Cargando detalle…</div>'
      );
      $tblBody.empty();
      $("#t_uc_exp, #t_uv_exp, #t_uc_cnt, #t_uv_cnt").text("0");
      $btnGuardar.prop("disabled", true);
      conteoPayload = null;
      showModalSafe();

      // Pedimos detalle
      $.getJSON(API, { detail: 1, fondo_id: fondoId, fecha: fechaSel })
        .done(function (res) {
          if (!res || res.ok !== true) {
            const msg =
              (res && (res.error || res.message)) ||
              "No se pudo cargar el detalle";
            notify("error", "Detalle", msg);
            $modalHeader.html(
              '<div class="text-danger small">No se pudo cargar el detalle.</div>'
            );
            return;
          }

          const hdr = res.header || {};
          $modalHeader.html(
            [
              `<div class="row g-2">`,
              ` <div class="col-12 col-md-4"><strong>Depósito:</strong> ${
                hdr.deposito_code || "-"
              }</div>`,
              ` <div class="col-12 col-md-4"><strong>Rack:</strong> ${
                hdr.rack_code || "-"
              }</div>`,
              ` <div class="col-12 col-md-4"><strong>Posición:</strong> ${
                hdr.pos_code || "-"
              }</div>`,
              ` <div class="col-12 col-md-4"><strong>Pallet:</strong> ${
                hdr.pallet_code || "-"
              }</div>`,
              ` <div class="col-12 col-md-4"><strong>Estado:</strong> ${
                hdr.estado || "-"
              }</div>`,
              // FIX: no referenciar variable fuera de scope → usar res.date o input
              ` <div class="col-12 col-md-4"><strong>Fecha:</strong> ${
                res.date || $("#flt-fecha").val() || "-"
              }</div>`,
              `</div>`,
            ].join("")
          );

          const items = Array.isArray(res.items) ? res.items : [];
          let tUCexp = 0,
            tUVexp = 0,
            tUCcnt = 0,
            tUVcnt = 0;

          items.forEach((it, idx) => {
            const tr = $(`
              <tr data-item-idx="${idx}">
                <td>${it.sku || ""}</td>
                <td>${it.denominacion || ""}</td>
                <td>${it.lote || ""}</td>
                <td>${it.vencimiento || ""}</td>
                <td class="text-end">${fmt(it.uc_esperadas)}</td>
                <td class="text-end">${fmt(it.uv_esperadas)}</td>
                <td class="text-end">
                  <input type="number" min="0" step="1" class="form-control form-control-sm text-end inp-uc-cnt" value="${num0(
                    it.uc_contadas
                  )}">
                </td>
                <td class="text-end">
                  <input type="number" min="0" step="1" class="form-control form-control-sm text-end inp-uv-cnt" value="${num0(
                    it.uv_contadas
                  )}">
                </td>
                <td>
                  <input type="text" class="form-control form-control-sm inp-obs" value="${
                    it.obs ? String(it.obs) : ""
                  }">
                </td>
              </tr>
            `);
            $tblBody.append(tr);

            tUCexp += +num0(it.uc_esperadas);
            tUVexp += +num0(it.uv_esperadas);
            tUCcnt += +num0(it.uc_contadas);
            tUVcnt += +num0(it.uv_contadas);
          });

          $("#t_uc_exp").text(fmt(tUCexp));
          $("#t_uv_exp").text(fmt(tUVexp));
          $("#t_uc_cnt").text(fmt(tUCcnt));
          $("#t_uv_cnt").text(fmt(tUVcnt));

          conteoPayload = {
            fondo_id: fondoId,
            fecha: res.date || $("#flt-fecha").val() || null,
            header: hdr,
            items: items.map((it) => ({
              producto_id: it.producto_id || null,
              lote: it.lote || null,
              vencimiento: it.vencimiento || null,
              uc_esperadas: num0(it.uc_esperadas),
              uv_esperadas: num0(it.uv_esperadas),
              uc_contadas: num0(it.uc_contadas),
              uv_contadas: num0(it.uv_contadas),
              obs: it.obs || null,
            })),
          };

          $btnGuardar.prop("disabled", false);

          toggleExpectedColumns(!!res.show_expected);

        })
        .fail(function (jqXHR) {
          const j = jqXHR.responseJSON || {};
          notify("error", "Detalle", j.error || "Error de comunicación");
          $modalHeader.html(
            '<div class="text-danger small">Error de comunicación.</div>'
          );
        });
    }

    // Recalcular totales en vivo
    $(document).on(
      "input",
      "#tblConteoItems .inp-uc-cnt, #tblConteoItems .inp-uv-cnt",
      function () {
        let tUC = 0,
          tUV = 0;
        $("#tblConteoItems tbody tr").each(function () {
          const uc = +($(this).find(".inp-uc-cnt").val() || 0);
          const uv = +($(this).find(".inp-uv-cnt").val() || 0);
          tUC += uc;
          tUV += uv;
        });
        $("#t_uc_cnt").text(fmt(tUC));
        $("#t_uv_cnt").text(fmt(tUV));
      }
    );

    // Guardar conteo (endpoint lo agregamos luego)
    // Guardar conteo (POST real a ?save=1)
    $btnGuardar.on("click", function () {
      if (!conteoPayload) return;

      // refrescar valores desde inputs antes de enviar
      $("#tblConteoItems tbody tr").each(function (i) {
        if (!conteoPayload.items[i]) return;
        conteoPayload.items[i].uc_contadas = num0(
          $(this).find(".inp-uc-cnt").val()
        );
        conteoPayload.items[i].uv_contadas = num0(
          $(this).find(".inp-uv-cnt").val()
        );
        conteoPayload.items[i].obs = String(
          $(this).find(".inp-obs").val() || ""
        );
      });

      // opcional: nota general de la posición (si agregás un textarea en el modal)
      // conteoPayload.nota = $("#inpNotaGeneral").val() || null;

      $.ajax({
        url: "/api/control/conteo_layout.php?save=1",
        method: "POST",
        contentType: "application/json; charset=utf-8",
        dataType: "json",
        data: JSON.stringify(conteoPayload),
      })
        .done(function (res) {
          if (!res || res.ok !== true) {
            notify(
              "error",
              "Conteo",
              (res && (res.error || res.message)) || "No se pudo guardar"
            );
            return;
          }
          notify("success", "Conteo", "Conteo registrado.");
          try {
            // cerrar modal
            if (window.bootstrap && bootstrap.Modal.getInstance($modal[0])) {
              bootstrap.Modal.getInstance($modal[0]).hide();
            } else if (typeof $modal.modal === "function") {
              $modal.modal("hide");
            }
          } catch (e) {}

          // Re-render del layout para que se marque como contado
          // (usa la fecha seleccionada en el filtro)
          $("#frmConteo").trigger("submit");
        })
        .fail(function (jqXHR) {
          const j = jqXHR.responseJSON || {};
          notify("error", "Conteo", j.error || "Error de comunicación");
        });

      toggleExpectedColumns(!!res.show_expected);
    });

    // ----- Filtros / render del layout --------------------------------------
    const $form = $("#frmConteo");
    const $dep = $("#flt-deposito");
    const $rack = $("#flt-rack");
    const $view = $("#flt-view");
    const $legend = $("#chk-legend");
    const $canvas = $(".warehouse-layout");

    function loadDepositos(selectedId) {
      return $.getJSON(API + "?meta=depositos")
        .then(function (data) {
          const list = data && data.depositos ? data.depositos : [];
          $dep
            .empty()
            .append($("<option/>", { value: "", text: "Seleccione..." }));
          list.forEach(function (d) {
            $dep.append(
              $("<option/>", { value: d.id, text: d.code || "DEP" + d.id })
            );
          });
          if (selectedId) $dep.val(String(selectedId));
        })
        .fail(function (jqXHR) {
          const j = jqXHR.responseJSON || {};
          notify(
            "error",
            "Depósitos",
            j.error || "No se pudieron cargar los depósitos"
          );
        });
    }

    function loadRacks(depositoId, selectedRackCode) {
      $rack
        .empty()
        .append($("<option/>", { value: "", text: "Seleccione..." }));
      if (!depositoId) return $.Deferred().resolve().promise();
      return $.getJSON(
        API + "?meta=racks&deposito_id=" + encodeURIComponent(depositoId)
      )
        .then(function (data) {
          const list = data && data.racks ? data.racks : [];
          list.forEach(function (r) {
            $rack.append($("<option/>", { value: r.code, text: r.code }));
          });
          if (selectedRackCode) $rack.val(String(selectedRackCode));
        })
        .fail(function (jqXHR) {
          const j = jqXHR.responseJSON || {};
          notify(
            "error",
            "Racks",
            j.error || "No se pudieron cargar los racks"
          );
        });
    }

    const init = {
      fecha: qs("fecha") || new Date().toISOString().slice(0, 10),
      deposito_id: qs("deposito_id"),
      rack: qs("rack"),
      view: (qs("view") || "microtiles").toLowerCase(),
      legend: true,
    };
    if (init.fecha) $form.find("#flt-fecha").val(init.fecha);
    if (init.view) $view.val(init.view);

    loadDepositos(init.deposito_id).then(function () {
      return loadRacks($dep.val(), init.rack);
    });

    $dep.on("change", function () {
      loadRacks($(this).val(), null);
    });

    function applyLayout() {
      const depId = $dep.val();
      const rackCode = $rack.val();
      const modeRaw = ($view.val() || "microtiles").toLowerCase();
      const mode = modeRaw.replace(/_/g, "-");
      const showLegend = !!$legend.is(":checked");

      if (!depId || !rackCode) {
        notify("error", "Validación", "Seleccione Depósito y Rack.");
        return;
      }

      $canvas.attr("data-dep", "DEP" + String(depId));
      $canvas.attr("data-rack", String(rackCode));
      $canvas.attr("data-mode", mode);
      $canvas.find(".warehouse-layout__canvas").empty();

      try {
        window.WarehouseLayout.init($(".warehouse-layout"), {
          apiUrl: API,
          defaultMode: mode,
          enableLegend: showLegend,
          injectControls: false,
          modeSelector: "#flt-view",
          depositoSelector: "#flt-deposito",
          rackSelector: "#flt-rack",
          formSelector: "#frmConteo",
          notify: function (type, title, text) {
            if (String(type).toLowerCase() === "info") return; // silenciar info
            (
              window.notify ||
              function (t, ti, te) {
                console[t === "error" ? "error" : "log"](`${ti}: ${te}`);
              }
            )(type, title, text);
          },
        });
      } catch (e) {
        console.error("[conteo] Error inicializando WarehouseLayout:", e);
        notify("error", "Render", "No se pudo inicializar el layout.");
      }
    }

    $form.on("submit", function (e) {
      e.preventDefault();
      applyLayout();
    });
    $view.on("change", function () {
      if ($dep.val() && $rack.val()) applyLayout();
    });
    $legend.on("change", function () {
      if ($dep.val() && $rack.val()) applyLayout();
    });

    if (init.deposito_id && init.rack) {
      $.when(loadDepositos(init.deposito_id)).then(function () {
        $.when(loadRacks(init.deposito_id, init.rack)).then(function () {
          applyLayout();
        });
      });
    }
  });
})();
