// Control · Inventario / Conteos
(function(window, $) {
  'use strict';

  // Base URL
  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') || 
                 (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + path.replace(/^\//, '');
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3500, showConfirmButton: false });
      return;
    }
    console.log(`[${type}] ${title}: ${text}`);
  }

  function formatNumber(num) {
    return new Intl.NumberFormat('es-PY').format(num || 0);
  }

  function formatDate(dateStr) {
    if (!dateStr) return '-';
    return new Date(dateStr).toLocaleDateString('es-PY');
  }

  $(function() {
    const dt = $('#conteosTable').DataTable({
      ajax: {
        url: joinUrl('api/control/inventario_conteos.php'),
        data: function(d) {
          d.estado = $('#filtroEstado').val();
          d.tipo = $('#filtroTipo').val();
          return d;
        },
        dataSrc: 'data'
      },
      order: [[2, 'desc']], // Ordenar por fecha_conteo DESC
      columns: [
        { 
          data: 'codigo',
          title: 'Código',
          render: function(data, type, row) {
            return `<a href="${joinUrl('control/inventario/detalle?id=' + row.id)}" class="text-primary">${data}</a>`;
          }
        },
        { data: 'fecha_conteo', title: 'Fecha', render: formatDate },
        { 
          data: 'estado', 
          title: 'Estado',
          render: function(data) {
            const badges = {
              'BORRADOR': 'secondary',
              'EN_PROCESO': 'info',
              'FINALIZADO': 'warning',
              'APROBADO': 'success',
              'RECHAZADO': 'danger'
            };
            return `<span class="badge bg-${badges[data] || 'secondary'}">${data}</span>`;
          }
        },
        { 
          data: 'tipo',
          title: 'Tipo',
          render: function(data) {
            const tipos = {
              'GENERAL': 'primary',
              'CICLICO': 'info',
              'PARCIAL': 'warning'
            };
            return `<span class="badge bg-${tipos[data] || 'secondary'}">${data}</span>`;
          }
        },
        { data: 'deposito', title: 'Depósito' },
        { data: 'cliente', title: 'Cliente', defaultContent: 'Todos' },
        { 
          data: 'items_count', 
          title: 'Items',
          className: 'text-end',
          render: formatNumber
        },
        { 
          data: 'total_unidades', 
          title: 'Total Unidades',
          className: 'text-end',
          render: formatNumber
        },
        { data: 'creado_por', title: 'Usuario' },
        {
          data: null,
          title: 'Acciones',
          orderable: false,
          render: function(data, type, row) {
            let html = '';
            if (row.estado === 'BORRADOR' || row.estado === 'EN_PROCESO') {
              html += `<button class="btn btn-sm btn-primary btn-edit" data-id="${row.id}" title="Editar"><i class="fa fa-edit"></i></button> `;
              html += `<button class="btn btn-sm btn-danger btn-delete" data-id="${row.id}" title="Eliminar"><i class="fa fa-trash"></i></button>`;
            } else {
              html += `<button class="btn btn-sm btn-info btn-view" data-id="${row.id}" title="Ver"><i class="fa fa-eye"></i></button>`;
            }
            return html;
          }
        }
      ],
      language: { url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json' }
    });

    // Filtros
    $('#btnAplicarFiltro').on('click', function() {
      dt.ajax.reload();
    });

    // Nuevo conteo
    $('#btnNuevoConteo').on('click', function() {
      window.location.href = joinUrl('control/inventario/nuevo');
    });

    // Editar
    $('#conteosTable').on('click', '.btn-edit', function() {
      const id = $(this).data('id');
      window.location.href = joinUrl('control/inventario/editar?id=' + id);
    });

    // Ver
    $('#conteosTable').on('click', '.btn-view', function() {
      const id = $(this).data('id');
      window.location.href = joinUrl('control/inventario/detalle?id=' + id);
    });

    // Eliminar
    $('#conteosTable').on('click', '.btn-delete', function() {
      const id = $(this).data('id');
      const codigo = $(this).closest('tr').find('td:first a').text();
      
      Swal.fire({
        title: '¿Eliminar conteo?',
        text: `Se eliminará el conteo ${codigo} y todos sus items`,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, eliminar',
        cancelButtonText: 'Cancelar'
      }).then((result) => {
        if (result.isConfirmed) {
          $.ajax({
            url: joinUrl('api/control/inventario_conteo_delete.php'),
            method: 'POST',
            data: { id: id },
            dataType: 'json',
            success: function(response) {
              if (response.ok) {
                notify('success', 'Eliminado', 'Conteo eliminado correctamente');
                dt.ajax.reload();
              } else {
                notify('error', 'Error', response.error || 'No se pudo eliminar');
              }
            },
            error: function() {
              notify('error', 'Error', 'Error de conexión');
            }
          });
        }
      });
    });
  });

})(window, jQuery);
