// Control · Inventario / Nuevo Conteo
(function(window, $) {
  'use strict';

  // Base URL
  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') || 
                 (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + path.replace(/^\//, '');
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3500, showConfirmButton: false });
      return;
    }
    console.log(`[${type}] ${title}: ${text}`);
  }

  function formatNumber(num) {
    return new Intl.NumberFormat('es-PY').format(num || 0);
  }

  let conteoId = 0;
  let items = [];
  let productoInfo = {}; // Cache de info de productos

  $(function() {
    conteoId = parseInt($('#conteo_id').val()) || 0;
    const isEdit = conteoId > 0;

    // Cargar combos
    cargarDepositos();
    cargarClientes();
    cargarProductos();

    if (isEdit) {
      cargarConteo();
      cargarItems();
    } else {
      // Generar código automático
      generarCodigo();
    }

    // ========================================================================
    // EVENTOS - CABECERA
    // ========================================================================
    
    $('#btnCancelar').on('click', function() {
      window.location.href = joinUrl('control/inventario');
    });

    $('#frmConteo').on('submit', function(e) {
      e.preventDefault();
      guardarCabecera();
    });

    // ========================================================================
    // EVENTOS - ITEMS
    // ========================================================================

    $('#btnAgregarItem').on('click', function() {
      if (!conteoId) {
        notify('warning', 'Atención', 'Debe guardar la cabecera primero');
        return;
      }
      abrirModalItem();
    });

    $('#btnGuardarItem').on('click', function() {
      guardarItem();
    });

    $('#tblItems').on('click', '.btn-edit-item', function() {
      const index = $(this).data('index');
      editarItem(index);
    });

    $('#tblItems').on('click', '.btn-delete-item', function() {
      const index = $(this).data('index');
      eliminarItem(index);
    });

    // Auto-calcular total unidades cuando cambian las cantidades (usando delegación)
    $(document).on('input change keyup', '#item_cantidad_pallets, #item_cantidad_cajas, #item_cantidad_unidades', function() {
      calcularTotalUnidades();
    });

    // Cuando selecciona un producto, cargar su info
    $('#item_producto_id').on('change', function() {
      const productoId = $(this).val();
      if (productoId) {
        cargarInfoProducto(productoId);
      } else {
        $('#infoProducto').hide();
      }
    });

    // ========================================================================
    // FUNCIONES - CABECERA
    // ========================================================================

    function generarCodigo() {
      const hoy = new Date();
      const año = hoy.getFullYear();
      const mes = String(hoy.getMonth() + 1).padStart(2, '0');
      const dia = String(hoy.getDate()).padStart(2, '0');
      const codigo = `CNT-${año}${mes}${dia}-001`;
      $('#codigo').val(codigo);
    }

    function cargarDepositos() {
      $.getJSON(joinUrl('api/parametros/depositos_combo.php'), function(response) {
        if (response.ok && response.data) {
          response.data.forEach(function(item) {
            $('#deposito_id').append($('<option>', { value: item.id, text: item.text }));
          });
        }
      });
    }

    function cargarClientes() {
      $.getJSON(joinUrl('api/parametros/clientes_combo.php'), function(response) {
        if (response.ok && response.data) {
          response.data.forEach(function(item) {
            $('#cliente_id').append($('<option>', { value: item.id, text: item.text }));
          });
        }
      });
    }

    function cargarProductos() {
      $.getJSON(joinUrl('api/parametros/productos_combo.php'), function(response) {
        if (response.ok && response.data) {
          response.data.forEach(function(item) {
            $('#item_producto_id').append($('<option>', { 
              value: item.id, 
              text: item.text 
            }));
          });
        }
      });
    }

    function guardarCabecera() {
      const data = {
        id: conteoId,
        codigo: $('#codigo').val(),
        fecha_conteo: $('#fecha_conteo').val(),
        deposito_id: $('#deposito_id').val(),
        cliente_id: $('#cliente_id').val() || null,
        tipo: $('#tipo').val(),
        estado: $('#estado').val(),
        observaciones: $('#observaciones').val()
      };

      $.ajax({
        url: joinUrl('api/control/inventario_conteo_save.php'),
        method: 'POST',
        data: JSON.stringify(data),
        contentType: 'application/json',
        dataType: 'json',
        success: function(response) {
          if (response.ok) {
            notify('success', 'Guardado', 'Conteo guardado correctamente');
            if (!conteoId) {
              conteoId = response.id;
              $('#conteo_id').val(conteoId);
              $('#seccionItems').show();
            }
          } else {
            notify('error', 'Error', response.error || 'No se pudo guardar');
          }
        },
        error: function() {
          notify('error', 'Error', 'Error de conexión');
        }
      });
    }

    function cargarConteo() {
      $.getJSON(joinUrl(`api/control/inventario_conteo_get.php?id=${conteoId}`), function(response) {
        if (response.ok && response.data) {
          const c = response.data;
          $('#codigo').val(c.codigo);
          $('#fecha_conteo').val(c.fecha_conteo);
          $('#deposito_id').val(c.deposito_id);
          $('#cliente_id').val(c.cliente_id || '');
          $('#tipo').val(c.tipo);
          $('#estado').val(c.estado);
          $('#observaciones').val(c.observaciones || '');
        }
      });
    }

    // ========================================================================
    // FUNCIONES - ITEMS
    // ========================================================================

    function cargarItems() {
      console.log('cargarItems() llamado, conteo_id:', conteoId);
      $.getJSON(joinUrl(`api/control/inventario_items.php?conteo_id=${conteoId}`), function(response) {
        console.log('Items recibidos:', response);
        if (response.ok && response.data) {
          items = response.data;
          console.log('Array items:', items.length, items);
          renderizarItems();
        }
      });
    }

    function renderizarItems() {
      console.log('renderizarItems() llamado, items.length:', items.length);
      const $tbody = $('#tblItems tbody');
      $tbody.empty();
      console.log('tbody limpiado');

      if (items.length === 0) {
        $tbody.append(`
          <tr>
            <td colspan="10" class="text-center text-muted">
              No hay items. Haga clic en "Agregar SKU" para comenzar.
            </td>
          </tr>
        `);
        actualizarTotales();
        return;
      }

      items.forEach(function(item, index) {
        $tbody.append(`
          <tr>
            <td>${item.sku}</td>
            <td>${item.denominacion}</td>
            <td>${item.lote || '-'}</td>
            <td>${item.posicion || '-'}</td>
            <td class="text-end">${formatNumber(item.cantidad_pallets)}</td>
            <td class="text-end">${formatNumber(item.cantidad_cajas)}</td>
            <td class="text-end">${formatNumber(item.cantidad_unidades)}</td>
            <td class="text-end fw-bold">${formatNumber(item.total_unidades)}</td>
            <td class="small">${item.observaciones || '-'}</td>
            <td>
              <button class="btn btn-sm btn-primary btn-edit-item" data-index="${index}" title="Editar">
                <i class="fa fa-edit"></i>
              </button>
              <button class="btn btn-sm btn-danger btn-delete-item" data-index="${index}" title="Eliminar">
                <i class="fa fa-trash"></i>
              </button>
            </td>
          </tr>
        `);
      });

      actualizarTotales();
    }

    function actualizarTotales() {
      const totales = items.reduce((acc, item) => {
        acc.pallets += parseInt(item.cantidad_pallets) || 0;
        acc.cajas += parseInt(item.cantidad_cajas) || 0;
        acc.unidades += parseInt(item.cantidad_unidades) || 0;
        acc.total_unidades += parseInt(item.total_unidades) || 0;
        return acc;
      }, { pallets: 0, cajas: 0, unidades: 0, total_unidades: 0 });

      $('#total_pallets').text(formatNumber(totales.pallets));
      $('#total_cajas').text(formatNumber(totales.cajas));
      $('#total_unidades').text(formatNumber(totales.unidades));
      $('#total_total_unidades').text(formatNumber(totales.total_unidades));
    }

    function abrirModalItem(item = null) {
      $('#mdlItemTitle').text(item ? 'Editar Item' : 'Agregar Item');
      $('#frmItem')[0].reset();
      $('#item_conteo_id').val(conteoId);
      $('#infoProducto').hide();

      if (item) {
        $('#item_id').val(item.id);
        $('#item_producto_id').val(item.producto_id);
        $('#item_lote').val(item.lote || '');
        $('#item_posicion').val(item.posicion || '');
        $('#item_cantidad_pallets').val(item.cantidad_pallets);
        $('#item_cantidad_cajas').val(item.cantidad_cajas);
        $('#item_cantidad_unidades').val(item.cantidad_unidades);
        $('#item_total_unidades').val(item.total_unidades);
        $('#item_total_unidades_display').val(formatNumber(item.total_unidades));
        $('#item_observaciones').val(item.observaciones || '');
        
        // Cargar info de producto y guardar datos para cálculo
        if (item.uc_por_caja !== undefined) {
          $('#item_producto_id').data('uc-por-caja', item.uc_por_caja || 0);
        }
        if (item.cajas_por_pallet !== undefined) {
          $('#item_producto_id').data('cajas-por-pallet', item.cajas_por_pallet || 0);
        }
        
        cargarInfoProducto(item.producto_id);
      }

      $('#mdlItem').modal('show');
    }

    function cargarInfoProducto(productoId) {
      // Si ya está en cache, usarlo
      if (productoInfo[productoId]) {
        mostrarInfoProducto(productoInfo[productoId]);
        return;
      }

      $.getJSON(joinUrl(`api/parametros/producto_get.php?id=${productoId}`), function(response) {
        if (response.ok && response.data) {
          productoInfo[productoId] = response.data;
          mostrarInfoProducto(response.data);
        }
      });
    }

    function mostrarInfoProducto(producto) {
      $('#info_producto_nombre').text(producto.denominacion);
      $('#info_uc_por_caja').text(producto.unidades_por_caja || '-');
      $('#info_cajas_por_pallet').text(producto.cajas_por_pallet || '-');
      $('#infoProducto').show();
      
      // Guardar para cálculo
      $('#item_producto_id').data('uc-por-caja', producto.unidades_por_caja || 0);
      $('#item_producto_id').data('cajas-por-pallet', producto.cajas_por_pallet || 0);
      
      calcularTotalUnidades();
    }

    function calcularTotalUnidades() {
      const pallets = parseInt($('#item_cantidad_pallets').val()) || 0;
      const cajas = parseInt($('#item_cantidad_cajas').val()) || 0;
      const unidades = parseInt($('#item_cantidad_unidades').val()) || 0;
      
      const ucPorCaja = parseInt($('#item_producto_id').data('uc-por-caja')) || 0;
      const cajasPorPallet = parseInt($('#item_producto_id').data('cajas-por-pallet')) || 0;

      // Total = (pallets × cajas/pallet × uc/caja) + (cajas × uc/caja) + unidades
      const totalPallets = pallets * cajasPorPallet * ucPorCaja;
      const totalCajas = cajas * ucPorCaja;
      const total = totalPallets + totalCajas + unidades;

      console.log('Cálculo:', {pallets, cajas, unidades, ucPorCaja, cajasPorPallet, totalPallets, totalCajas, total});
      
      // Actualizar campo oculto con valor real y campo visible con formato
      const $hidden = $('#item_total_unidades');
      const $display = $('#item_total_unidades_display');
      
      console.log('Campos encontrados:', {hidden: $hidden.length, display: $display.length});
      
      $hidden.val(total);
      $display.val(formatNumber(total));
      
      console.log('Valores asignados:', {hidden: $hidden.val(), display: $display.val()});
    }

    function guardarItem() {
      const data = {
        id: $('#item_id').val() || null,
        conteo_id: conteoId,
        producto_id: $('#item_producto_id').val(),
        lote: $('#item_lote').val() || '',
        posicion: $('#item_posicion').val() || '',
        cantidad_pallets: $('#item_cantidad_pallets').val(),
        cantidad_cajas: $('#item_cantidad_cajas').val(),
        cantidad_unidades: $('#item_cantidad_unidades').val(),
        total_unidades: $('#item_total_unidades').val(),
        observaciones: $('#item_observaciones').val()
      };

      $.ajax({
        url: joinUrl('api/control/inventario_item_save.php'),
        method: 'POST',
        data: JSON.stringify(data),
        contentType: 'application/json',
        dataType: 'json',
        success: function(response) {
          if (response.ok) {
            notify('success', 'Guardado', 'Item guardado correctamente');
            $('#frmItem')[0].reset(); // Limpiar formulario
            $('#item_producto_id').val('').trigger('change'); // Limpiar select
            $('#infoProducto').hide(); // Ocultar info de producto
            $('#mdlItem').modal('hide');
            cargarItems();
          } else {
            notify('error', 'Error', response.error || 'No se pudo guardar');
          }
        },
        error: function() {
          notify('error', 'Error', 'Error de conexión');
        }
      });
    }

    function editarItem(index) {
      abrirModalItem(items[index]);
    }

    function eliminarItem(index) {
      const item = items[index];
      
      Swal.fire({
        title: '¿Eliminar item?',
        text: `Se eliminará ${item.sku} del conteo`,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, eliminar',
        cancelButtonText: 'Cancelar'
      }).then((result) => {
        if (result.isConfirmed) {
          $.ajax({
            url: joinUrl('api/control/inventario_item_delete.php'),
            method: 'POST',
            data: JSON.stringify({ id: item.id }),
            contentType: 'application/json',
            dataType: 'json',
            success: function(response) {
              if (response.ok) {
                notify('success', 'Eliminado', 'Item eliminado correctamente');
                cargarItems();
              } else {
                notify('error', 'Error', response.error || 'No se pudo eliminar');
              }
            },
            error: function() {
              notify('error', 'Error', 'Error de conexión');
            }
          });
        }
      });
    }

  });

})(window, jQuery);
