// public/page-scripts/pages/control/mapeo.js
// -----------------------------------------------------------
// Control · Mapeo por ubicación (layout externo sin modal)
// Requiere:
//   - public/page-scripts/warehouse/layout-core.js
//   - public/page-scripts/warehouse/render-microtiles.js
//   - public/page-scripts/warehouse/render-shelf-strips.js
//   - public/page-scripts/warehouse/render-column-cards.js
// -----------------------------------------------------------
(function () {
  "use strict";

  // Notificador por defecto (si no hay uno global)
  const notify =
    window.notify ||
    function (type, title, message) {
      const msg = `[${type?.toUpperCase?.() || "INFO"}] ${title || ""} ${message || ""}`.trim();
      console[type === "error" ? "error" : "log"](msg);
    };

  function qs(name) {
    const m = new RegExp("[?&]" + name + "=([^&#]*)").exec(window.location.search);
    return m ? decodeURIComponent(m[1].replace(/\+/g, " ")) : "";
  }

  $(function () {
    if (!window.WarehouseLayout) {
      console.error("[mapeo] WarehouseLayout no está disponible. ¿Incluiste layout-core.js?");
      return;
    }

    // API propia de MAPEO (meta + layout)
    const API = "/api/control/mapeo_layout.php";

    // === UI refs =============================================================
    const $form   = $("#frmMapeo");
    const $dep    = $("#flt-deposito");
    const $rack   = $("#flt-rack");
    const $view   = $("#flt-view");
    const $legend = $("#chk-legend");
    const $canvas = $(".warehouse-layout");

    // ---------- Carga de filtros (meta) ----------
    function loadDepositos(selectedId) {
      return $.getJSON(API + "?meta=depositos")
        .then(function (data) {
          const list = data && data.depositos ? data.depositos : [];
          $dep.empty().append($("<option/>", { value: "", text: "Seleccione..." }));
          list.forEach(function (d) {
            $dep.append($("<option/>", { value: d.id, text: d.code || "DEP" + d.id }));
          });
          if (selectedId) $dep.val(String(selectedId));
        })
        .fail(function (jqXHR) {
          const j = jqXHR.responseJSON || {};
          notify("error", "Depósitos", j.error || "No se pudieron cargar los depósitos");
        });
    }

    function loadRacks(depositoId, selectedRackCode) {
      $rack.empty().append($("<option/>", { value: "", text: "Seleccione..." }));
      if (!depositoId) return $.Deferred().resolve().promise();
      return $.getJSON(API + "?meta=racks&deposito_id=" + encodeURIComponent(depositoId))
        .then(function (data) {
          const list = data && data.racks ? data.racks : [];
          list.forEach(function (r) {
            $rack.append($("<option/>", { value: r.code, text: r.code }));
          });
          if (selectedRackCode) $rack.val(String(selectedRackCode));
        })
        .fail(function (jqXHR) {
          const j = jqXHR.responseJSON || {};
          notify("error", "Racks", j.error || "No se pudieron cargar los racks");
        });
    }

    // ---------- Inicial ----------
    const init = {
      deposito_id: qs("deposito_id"),
      rack:        qs("rack"),
      view:       (qs("view") || "microtiles").toLowerCase(),
      legend:      true
    };
    if (init.view) $view.val(init.view);

    // Popular filtros y (si vienen por query) renderizar
    loadDepositos(init.deposito_id).then(function () {
      return loadRacks($dep.val(), init.rack);
    });

    $dep.on("change", function () {
      loadRacks($(this).val(), null);
    });

    // ---------- Render del layout ----------
    function applyLayout() {
      const depId      = $dep.val();
      const rackCode   = $rack.val();
      const modeRaw    = ($view.val() || "microtiles").toLowerCase();
      const mode       = modeRaw.replace(/_/g, "-");
      const showLegend = !!$legend.is(":checked");

      if (!depId || !rackCode) {
        notify("error", "Validación", "Seleccione Depósito y Rack.");
        return;
      }

      // Atributos informativos en el canvas
      $canvas.attr("data-dep", "DEP" + String(depId));
      $canvas.attr("data-rack", String(rackCode));
      $canvas.attr("data-mode", mode);
      $canvas.find(".warehouse-layout__canvas").empty();

      try {
        window.WarehouseLayout.init($(".warehouse-layout"), {
          apiUrl:           API,
          defaultMode:      mode,
          enableLegend:     showLegend,
          injectControls:   false,
          modeSelector:     "#flt-view",
          depositoSelector: "#flt-deposito",
          rackSelector:     "#flt-rack",
          formSelector:     "#frmMapeo",
          // En MAPEO no abrimos modal ni capturamos click: solo visualizar ocupación
          notify: function (type, title, text) {
            if (String(type).toLowerCase() === "info") return; // minimizar ruido
            (window.notify || function (t, ti, te) {
              console[t === "error" ? "error" : "log"](`${ti}: ${te}`);
            })(type, title, text);
          },
        });
      } catch (e) {
        console.error("[mapeo] Error inicializando WarehouseLayout:", e);
        notify("error", "Render", "No se pudo inicializar el layout.");
      }
    }

    // Submit + cambios de vista/leyenda re-renderizan
    $form.on("submit", function (e) {
      e.preventDefault();
      applyLayout();
    });
    $view.on("change", function () {
      if ($dep.val() && $rack.val()) applyLayout();
    });
    $legend.on("change", function () {
      if ($dep.val() && $rack.val()) applyLayout();
    });

    // Si ya viene todo en la URL, render inicial
    if (init.deposito_id && init.rack) {
      $.when(loadDepositos(init.deposito_id)).then(function () {
        $.when(loadRacks(init.deposito_id, init.rack)).then(function () {
          applyLayout();
        });
      });
    }
  });
})();
