/* public/page-scripts/pages/deposito/layout.js */
(function () {
  "use strict";

  const $ = window.jQuery;
  const BASE = (document.querySelector('meta[name="base-url"]')?.content || '').replace(/\/+$/, '');

  // ========= helpers =========
  function escapeH(s) {
    try {
      return String(s ?? '')
        .replaceAll('&', '&amp;')
        .replaceAll('<', '&lt;')
        .replaceAll('>', '&gt;')
        .replaceAll('"', '&quot;')
        .replaceAll("'", '&#039;');
    } catch {
      return String(s ?? '');
    }
  }

  // color de texto legible sobre un fondo hex (#RRGGBB)
  function contrastFor(hex) {
    if (!hex) return '#212529';
    hex = String(hex).trim();
    if (hex[0] !== '#') hex = '#' + hex;
    if (!/^#[0-9a-fA-F]{6}$/.test(hex)) return '#212529';
    const r = parseInt(hex.slice(1, 3), 16);
    const g = parseInt(hex.slice(3, 5), 16);
    const b = parseInt(hex.slice(5, 7), 16);
    const yiq = (r * 299 + g * 587 + b * 114) / 1000;
    return yiq >= 150 ? '#212529' : '#ffffff';
  }

  function indexToLetters(n) {
    let s = '';
    while (n > 0) {
      n--;
      s = String.fromCharCode(65 + (n % 26)) + s;
      n = Math.floor(n / 26);
    }
    return s || '0';
  }

  // ========= estilos embebidos (una vez) =========
  (function ensureStyles() {
    if (document.getElementById('deposito-layout-styles')) return;
    const css = `
.layout-toolbar{display:flex;gap:.75rem;flex-wrap:wrap;margin-bottom:.75rem}
.layout-legend{margin:.5rem 0 1rem 0}
.deposito-layout-rack{margin-bottom:24px}
.rack-header{font-weight:700;margin-bottom:8px}
.grid-row{display:grid;grid-template-columns:repeat(24,1fr);gap:8px}
.level-label{font-weight:600;margin:8px 0 4px}
.entry-cell{background:#e2e8f0;border:1px solid #cbd5e1;border-radius:6px;color:#334155;display:flex;align-items:center;justify-content:center;height:46px;font-weight:700}
.layout-col{display:flex;flex-direction:column;gap:6px;align-items:stretch}
.col-header{font-weight:600;text-align:center;margin-bottom:2px}
.pos-tile{position:relative;width:100%;height:60px;border-radius:6px;background:#fff;border:1px solid #e6e6e6;display:flex;flex-direction:column;align-items:center;justify-content:center;box-sizing:border-box;cursor:pointer;user-select:none}
.pos-tile .big{font-size:1rem;line-height:1}
.pos-tile .muted{font-size:.72rem;line-height:1;opacity:.9}
.pos-tile.empty{background:#ffffff;border-style:dashed;color:transparent}
.pos-tile.pos-free{background:#f8fafc;border-color:#e2e8f0;color:#475569}
.pos-tile.pos-multi{outline:2px dashed rgba(0,0,0,.18)}
.pos-tile:hover{transform:translateY(-1px);box-shadow:0 3px 10px rgba(0,0,0,.06)}
.aisle-row{position:relative}
.aisle-label{position:absolute;left:50%;transform:translateX(-50%);top:6px;color:#64748b;font-weight:600}
`;
    const st = document.createElement('style');
    st.id = 'deposito-layout-styles';
    st.appendChild(document.createTextNode(css));
    document.head.appendChild(st);
  })();

  // ========= carga de combos de filtros (si existen en la página) =========
  async function fetchJSON(url) {
    const res = await fetch(url, { credentials: 'same-origin' });
    if (!res.ok) throw new Error(`HTTP ${res.status}`);
    return res.json();
  }

  async function loadDepositos() {
    const sel = document.getElementById('fDeposito');
    if (!sel) return;
    const data = await fetchJSON(`${BASE}/api/parametros/depositos.php?meta=select`);
    const rows = Array.isArray(data) ? data : (data?.data || []);
    sel.innerHTML = '<option value="">(Seleccionar)</option>';
    rows.forEach(d => {
      const opt = document.createElement('option');
      opt.value = d.id;
      opt.textContent = `${d.code ?? ''} ${d.nombre ?? ''}`.trim();
      sel.appendChild(opt);
    });
  }

  async function loadAmbientes(depId) {
    const sel = document.getElementById('fAmbiente');
    if (!sel) return;
    const url = depId
      ? `${BASE}/api/parametros/ambientes.php?deposito_id=${encodeURIComponent(depId)}`
      : `${BASE}/api/parametros/ambientes.php`;
    const data = await fetchJSON(url);
    const rows = Array.isArray(data) ? data : (data?.data || []);
    sel.innerHTML = '<option value="">(Todos)</option>';
    rows.forEach(a => {
      const opt = document.createElement('option');
      opt.value = a.id;
      opt.textContent = `${a.code ?? ''} ${a.nombre ?? ''}`.trim();
      sel.appendChild(opt);
    });
  }

  // ========= render del layout (usa colores de producto) =========
  function renderLayout(container, data) {
    if (!Array.isArray(data) || !data.length) {
      container.innerHTML = '<div class="alert alert-info mb-0">Sin datos para mostrar.</div>';
      return;
    }

    // group por rack -> columna
    const racks = {}; // { rack: { col: [posiciones...] } }
    data.forEach(p => {
      const r = Number(p.rack ?? p.rack_num ?? 0) || 0;
      const c = Number(p.columna ?? p.col ?? 0) || 0;
      (racks[r] ??= {})[c] ??= [];
      racks[r][c].push(p);
    });

    const rackKeys = Object.keys(racks).map(Number).sort((a,b)=>a-b);

    // leyenda simple
    const legend = document.createElement('div');
    legend.className = 'layout-legend';
    legend.innerHTML = `
      <span style="display:inline-flex;align-items:center;gap:6px;margin-right:12px"><span style="display:inline-block;width:14px;height:14px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:3px"></span> Libre</span>
      <span style="display:inline-flex;align-items:center;gap:6px;margin-right:12px"><span style="display:inline-block;width:14px;height:14px;background:#e2e8f0;border:1px solid #cbd5e1;border-radius:3px"></span> Sin color</span>
      <span style="display:inline-flex;align-items:center;gap:6px;margin-right:12px"><span style="display:inline-block;width:14px;height:14px;background:#000;border:1px solid #000;border-radius:3px"></span> Con color (por producto)</span>
    `;
    container.appendChild(legend);

    // orientación global (si la API lo trae)
    let globalOrient = 'N';
    const o0 = String(data[0]?.deposito_orientacion || data[0]?.orientacion || 'N').toUpperCase();
    if ('NESW'.includes(o0)) globalOrient = o0;

    // filas de entrada arriba si N, abajo si S
    if (globalOrient === 'N') appendEntryRow(container);
    if (globalOrient === 'N') appendEntryRow(container);

    rackKeys.forEach((rk, ridx) => {
      const rackBox = document.createElement('div');
      rackBox.className = 'deposito-layout-rack';

      const header = document.createElement('div');
      header.className = 'rack-header';
      header.textContent = 'RACK ' + indexToLetters(rk);
      rackBox.appendChild(header);

      const colsObj = racks[rk];
      const colKeys = Object.keys(colsObj).map(Number).sort((a,b)=>a-b);

      // grid de 24 columnas, reservar 2 a la izquierda si E, 2 a la derecha si W
      const TOTAL = 24;
      const entryLeft  = (globalOrient === 'E') ? 2 : 0;
      const entryRight = (globalOrient === 'W') ? 2 : 0;
      const usable = Math.max(1, TOTAL - (entryLeft + entryRight));

      const nCols = Math.max(1, colKeys.length);
      const base = Math.floor(usable / nCols);
      const extra = usable % nCols;
      const spans = new Array(nCols).fill(0).map((_,i)=> base + (i < extra ? 1 : 0));

      let cursor = 1 + entryLeft;
      const starts = spans.map(s => {
        const from = cursor;
        cursor += s;
        return from;
      });

      // niveles del rack
      const nivSet = {};
      colKeys.forEach(ck => colsObj[ck].forEach(p => nivSet[Number(p.nivel)||0] = true));
      const niveles = Object.keys(nivSet).map(Number).sort((a,b)=>a-b);
      if (!niveles.length) niveles.push(0);

      niveles.forEach(nv => {
        const lbl = document.createElement('div');
        lbl.className = 'level-label';
        lbl.textContent = 'NIVEL ' + nv;
        rackBox.appendChild(lbl);

        const row = document.createElement('div');
        row.className = 'grid-row mb-2';

        for (let i=0;i<entryLeft;i++) {
          const ent = document.createElement('div');
          ent.className = 'entry-cell';
          ent.textContent = 'ENT';
          row.appendChild(ent);
        }

        colKeys.forEach((ck, i) => {
          const colWrap = document.createElement('div');
          colWrap.className = 'layout-col';
          colWrap.style.gridColumn = `${starts[i]} / span ${spans[i]}`;

          const colHdr = document.createElement('div');
          colHdr.className = 'col-header';
          colHdr.textContent = indexToLetters(ck);
          colWrap.appendChild(colHdr);

          const poss = colsObj[ck].filter(p => Number(p.nivel) === Number(nv));
          if (!poss.length) {
            const empty = document.createElement('div');
            empty.className = 'pos-tile empty';
            colWrap.appendChild(empty);
          } else {
            // agrupar por fondo
            const byF = {};
            poss.forEach(p => {
              const f = Number(p.fondo) || 1;
              (byF[f] ??= []).push(p);
            });
            const fKeys = Object.keys(byF).map(Number).sort((a,b)=>a-b);
            fKeys.forEach(fk => {
              const items = byF[fk];

              // elegimos el color (si hay pallets, tomamos el color del producto; si hay más de uno, el del 1º)
              const first = items[0] || {};
              // distintas posibilidades que puede traer tu API:
              const bg =
                first.color_hex ||
                first.producto_color ||
                first.producto_color_web ||
                first.color_web ||
                null;

              const tile = document.createElement('div');
              tile.className = 'pos-tile';
              // estado visual básico
              if (items.length === 0) tile.classList.add('pos-free');
              if (items.length > 1)  tile.classList.add('pos-multi');

              if (bg) {
                const hex = String(bg).startsWith('#') ? String(bg) : `#${bg}`;
                tile.style.backgroundColor = hex;
                tile.style.color = contrastFor(hex);
                tile.style.borderColor = hex;
              }

              // número de fondo (grande) y “etiqueta” abajo
              const big = document.createElement('div');
              big.className = 'big';
              big.textContent = String(fk);

              const muted = document.createElement('div');
              muted.className = 'muted';
              const prodCode = first.producto_code || first.sku || '';
              const prodName = first.producto_nombre || first.producto_denominacion || '';
              const lote     = first.lote_uc || first.lote || '';
              muted.textContent = (prodCode || prodName || lote)
                ? `${prodCode || prodName}${lote ? ' ' + lote : ''}`
                : '';

              tile.appendChild(big);
              tile.appendChild(muted);

              // id de posición para el modal
              const posId = first.posicion_id || first.id || first.pos_id || 0;
              tile.dataset.posId = posId;

              tile.addEventListener('click', () => {
                if (typeof window.openPosicionModal === 'function') {
                  window.openPosicionModal({ id: Number(posId) });
                } else {
                  // fallback local
                  openPosModalFallback(Number(posId));
                }
              });

              colWrap.appendChild(tile);
            });
          }

          row.appendChild(colWrap);
        });

        for (let i=0;i<entryRight;i++) {
          const ent = document.createElement('div');
          ent.className = 'entry-cell';
          ent.textContent = 'ENT';
          row.appendChild(ent);
        }

        rackBox.appendChild(row);
      });

      container.appendChild(rackBox);

      // separador pasillo entre racks
      if (ridx < rackKeys.length - 1) {
        appendAisle(container);
      }
    });

    if (globalOrient === 'S') appendEntryRow(container), appendEntryRow(container);
  }

  function appendEntryRow(container) {
    const row = document.createElement('div');
    row.className = 'grid-row mb-2';
    const ent = document.createElement('div');
    ent.className = 'entry-cell';
    ent.textContent = 'ENT';
    ent.style.gridColumn = '1 / -1';
    row.appendChild(ent);
    container.appendChild(row);
  }
  function appendAisle(container) {
    const row = document.createElement('div');
    row.className = 'grid-row mb-2 aisle-row';
    const filler = document.createElement('div');
    filler.style.gridColumn = '1 / -1';
    filler.style.minHeight = '42px';
    row.appendChild(filler);
    const label = document.createElement('div');
    label.className = 'aisle-label';
    label.style.gridColumn = '1 / -1';
    label.textContent = 'PASILLO';
    row.appendChild(label);
    container.appendChild(row);
  }

  // ========= modal “Detalle de posición” =========
  // API esperada: GET /api/deposito/posicion_pallets.php?posicion_id=123
  async function openPosModalFallback(posId) {
    const modalEl = document.getElementById('mdlPosicion');
    const bodyEl  = document.getElementById('mdlPosicionBody');
    if (!modalEl || !bodyEl) return;
    bodyEl.innerHTML = '<div class="text-muted">Cargando…</div>';

    try {
      const res = await fetch(`${BASE}/api/deposito/posicion_pallets.php?posicion_id=${encodeURIComponent(posId)}`, { credentials: 'same-origin' });
      if (!res.ok) throw new Error(`HTTP ${res.status}`);
      const json = await res.json();
      bodyEl.innerHTML = renderPosDetails(json);
    } catch (e) {
      console.error('[layout] detalle posición:', e);
      bodyEl.innerHTML = '<div class="alert alert-danger mb-0">No se pudo cargar el detalle.</div>';
    }

    try {
      (bootstrap.Modal.getInstance(modalEl) || new bootstrap.Modal(modalEl)).show();
    } catch {
      modalEl.style.display = 'block';
    }
  }

  function renderPosDetails(resp) {
    const pallets = Array.isArray(resp?.pallets) ? resp.pallets : (resp?.data || []);
    if (!pallets.length) return '<div class="alert alert-info mb-0">Sin pallets en esta posición.</div>';

    let html = '';
    pallets.forEach(pal => {
      const pallet = pal.pallet || pal;
      const items  = pal.items  || pal.detalles || [];
      const title  = pallet.codigo || pallet.code || `Pallet #${pallet.id ?? ''}`;
      const totUv  = Number(pal?.totals?.qty_uv ?? pallet?.uv ?? 0);
      const totUc  = Number(pal?.totals?.qty_uc ?? pallet?.uc ?? 0);

      html += `
      <div class="card mb-3">
        <div class="card-header d-flex justify-content-between align-items-center">
          <span class="pal-title">${escapeH(title)}</span>
          <div class="d-flex align-items-center flex-wrap gap-2">
            <span class="badge text-bg-light">UV: ${escapeH(String(totUv))}</span>
            <span class="badge text-bg-light">UC: ${escapeH(String(totUc))}</span>
            <span class="badge text-bg-secondary">Ítems: ${escapeH(String(items.length))}</span>
          </div>
        </div>
        <div class="card-body p-2">
          ${items.length ? renderItemsTable(items) : '<div class="text-muted">Sin ítems</div>'}
        </div>
      </div>`;
    });
    return html;
  }

  function renderItemsTable(items) {
    const rows = items.map(it => {
      const prod = it.producto || {};
      const name = [prod.sku || prod.code || '', prod.denominacion || prod.nombre || '']
        .filter(Boolean).join(' · ');
      const lote = it.lote_info?.code || it.lote || it.lote_uc || '';
      const vto  = it.lote_info?.fecha_venc || it.fecha_venc || it.fecha_vto || '';
      const uv   = Number(it.qty_uv || 0), uc = Number(it.qty_uc || 0);
      return `
        <tr>
          <td>${escapeH(name)}</td>
          <td>${escapeH(lote)}</td>
          <td>${escapeH(String(vto || ''))}</td>
          <td class="text-end">${escapeH(String(uv))}</td>
          <td class="text-end">${escapeH(String(uc))}</td>
        </tr>`;
    }).join('');
    return `
      <div class="table-responsive">
        <table class="table table-sm table-striped align-middle mb-0">
          <thead><tr><th>Producto</th><th>Lote</th><th>Venc</th><th class="text-end">UV</th><th class="text-end">UC</th></tr></thead>
          <tbody>${rows}</tbody>
        </table>
      </div>`;
  }

  // ========= wiring de eventos (si hay formulario de filtros) =========
  document.addEventListener('DOMContentLoaded', async function () {
    const frm = document.getElementById('frmLayoutFilters');
    const grid = document.getElementById('layoutGrid');
    const msg = document.getElementById('layoutMsg');
    const btnRefresh = document.getElementById('btnRefreshLayout');
    const btnExport = document.getElementById('btnExportLayoutJson');
    const selDep = document.getElementById('fDeposito');

    async function refresh(e) {
      if (e) e.preventDefault();
      if (!grid) return;

      const dep = document.getElementById('fDeposito')?.value || '';
      const amb = document.getElementById('fAmbiente')?.value || '';
      let url = `${BASE}/api/deposito/layout.php`;
      const q = [];
      if (dep) q.push(`deposito_id=${encodeURIComponent(dep)}`);
      if (amb) q.push(`ambiente_id=${encodeURIComponent(amb)}`);
      if (q.length) url += `?${q.join('&')}`;

      msg && (msg.textContent = 'Cargando…');
      grid.innerHTML = '';

      try {
        const data = await fetchJSON(url);
        msg && (msg.textContent = '');
        renderLayout(grid, data);
      } catch (err) {
        console.error('[layout] load error:', err);
        msg && (msg.textContent = `Error: ${err.message}`);
      }
    }

    // combos
    try { await loadDepositos(); } catch {}
    try { await loadAmbientes(); } catch {}
    if (selDep) selDep.addEventListener('change', () => loadAmbientes(selDep.value));

    // botones
    frm && frm.addEventListener('submit', refresh);
    btnRefresh && btnRefresh.addEventListener('click', refresh);
    btnExport && btnExport.addEventListener('click', function () {
      const json = grid?.querySelector('pre')?.textContent || '[]';
      const blob = new Blob([json], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url; a.download = 'layout.json';
      document.body.appendChild(a); a.click(); a.remove();
      URL.revokeObjectURL(url);
    });

    // carga inicial
    refresh();
  });

  // ========= API pública opcional: openPosicionModal =========
  // Si esta función ya existe globalmente en otra vista, usará esa.
  window.openPosicionModal = window.openPosicionModal || function ({ id }) {
    return openPosModalFallback(Number(id));
  };

})();
