/* eslint-disable no-console */
(function () {
  "use strict";

  // =======================
  // BASE
  // =======================
  const $ = window.jQuery;
  if (!$) { console.error("[rotacion] jQuery no está disponible."); return; }

  const BASE = (document.querySelector('meta[name="base-url"]')?.content || "").replace(/\/+$/, "");

  const API = {
    // Catálogos
    depositos: `${BASE}/api/parametros/depositos.php?meta=select`,
    ambientesByDeposito: (depId) => `${BASE}/api/parametros/ambientes.php?deposito_id=${encodeURIComponent(depId)}`,
    racks: (depId, ambId) => `${BASE}/api/deposito/racks.php?deposito_id=${encodeURIComponent(depId)}${ambId ? `&ambiente_id=${encodeURIComponent(ambId)}` : ''}`,
    // Layout HTML
    rackLayoutHTML: (depId, rackId, mode = "microtiles") =>
      `${BASE}/api/deposito/rack_layout.php?deposito_id=${encodeURIComponent(depId)}&rack_id=${encodeURIComponent(rackId)}&mode=${encodeURIComponent(mode)}`,
    // Pallets en posición (NUEVO endpoint)
    posicionPallets: (posId) => `${BASE}/api/deposito/posicion_pallets_movimiento.php?pos_id=${encodeURIComponent(posId)}`,
    // Buffer (bulk): guarda items en wh_move_buffer
    moveBuffer: `${BASE}/api/deposito/move_buffer.php`,
    // Commit real (inserta en wh_move y actualiza posicion de pallets)
    moveCommit: `${BASE}/api/deposito/move_commit.php`
  };

  // =======================
  // NODOS
  // =======================
  const $depOrigen   = $("#deposito_id_origen");
  const $rackOrigen  = $("#rack_id_origen");
  const $ambOrigen   = $("#ambiente_id_origen");
  const $depDestino  = $("#deposito_id_destino");
  const $rackDestino = $("#rack_id_destino");
  const $ambDestino  = $("#ambiente_id_destino");

  const $formOrigen  = $("#formOrigen");
  const $formDestino = $("#formDestino");

  const $layoutDesde = $("#layoutDesde");
  const $layoutHasta = $("#layoutHasta");

  const $btnConfirm  = $("#btnConfirmarMovimiento");
  const $btnCancel   = $("#btnCancelarMovimiento");

  // =======================
  // ESTADO
  // =======================
  const cache = { depositos: null, racks: Object.create(null), ambientes: Object.create(null) };
  let lastRackFetchControllerOrigen  = null;
  let lastRackFetchControllerDestino = null;
  let lastAmbFetchControllerOrigen   = null;
  let lastAmbFetchControllerDestino  = null;

  // Selección y plan
  let selectedFrom = null; // {pos_id, el}
  let selectedTo   = null; // {pos_id, el}
  const plan = []; // items: { deposito_from, deposito_to, from_pos_id, to_pos_id, pallets:[{id,code,producto_id,lote_id,uv,uc}], ts }

  // =======================
  // HABILITAR/INHABILITAR CONFIRMAR (scope global del módulo)
  // =======================
  function evaluarHabilitarConfirmar() {
    const baseOk = Boolean($depOrigen.val() && $rackOrigen.val()); // sólo requerimos origen para planear
    const allHaveDest = plan.length > 0 && plan.every(p => !!p.to_pos_id);
    $btnConfirm.prop("disabled", !(baseOk && allHaveDest));
  }

  // =======================
  // SELECT2 (opcional y seguro)
  // =======================
  const SELECT2_JS = "https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js";
  let _scriptLoading = {};
  function loadScriptOnce(src) {
    if (_scriptLoading[src]) return _scriptLoading[src];
    _scriptLoading[src] = new Promise((resolve, reject) => {
      const s = document.createElement("script");
      s.src = src; s.async = true;
      s.onload = () => resolve(true);
      s.onerror = () => reject(new Error("No se pudo cargar: " + src));
      document.head.appendChild(s);
    });
    return _scriptLoading[src];
  }
  async function ensureSelect2Available() {
    if ($.fn && $.fn.select2) return true;
    try { await loadScriptOnce(SELECT2_JS); return !!($.fn && $.fn.select2); }
    catch { return false; }
  }
  function initSelect2IfAvailable($el, placeholder) {
    try {
      if ($.fn && $.fn.select2 && !$el.data("select2")) {
        $el.select2({ width: "100%", placeholder: placeholder || "Seleccione…", allowClear: true });
      }
    } catch (e) { console.warn("[rotacion] initSelect2IfAvailable warning:", e); }
  }

  // =======================
  // UI helpers para selects
  // =======================
  function setLoading($select, msg) {
    $select.prop("disabled", true).empty().append(new Option(msg || "Cargando…", "", true, true));
    if ($select.data("select2")) $select.trigger("change.select2");
  }
  function setError($select, msg) {
    $select.prop("disabled", true).empty().append(new Option(msg || "Error", "", true, true));
    if ($select.data("select2")) $select.trigger("change.select2");
  }
  function clearAndEnable($select) {
    $select.prop("disabled", false).empty().append(new Option("", "", false, false));
    if ($select.data("select2")) $select.trigger("change.select2");
  }
  function fillOptions($select, rows, { value = "id", text = "nombre", extra = "code" } = {}) {
    clearAndEnable($select);
    (rows || []).forEach(r => {
      const v = r?.[value];
      const label = (r?.[text] ?? r?.[extra] ?? String(v || ""));
      const code = r?.[extra] ? ` [${r[extra]}]` : "";
      $select.append(new Option(`${label}${code}`, v, false, false));
    });
    if ($select.data("select2")) $select.trigger("change.select2");
  }
  function normalizeList(json) {
    if (Array.isArray(json)) return json;
    if (json && Array.isArray(json.data)) return json.data;
    return [];
  }
  async function fetchJSON(url, controller) {
    const res = await fetch(url, { credentials: "same-origin", signal: controller?.signal });
    if (!res.ok) throw new Error(`HTTP ${res.status}`);
    return res.json();
  }

  // =======================
  // DATOS
  // =======================
  async function cargarDepositos() {
    if (Array.isArray(cache.depositos)) {
      fillOptions($depOrigen, cache.depositos);
      fillOptions($depDestino, cache.depositos);
      return;
    }
    setLoading($depOrigen, "Cargando depósitos…");
    setLoading($depDestino, "Cargando depósitos…");
    try {
      const json = await fetchJSON(API.depositos);
      const rows = normalizeList(json);
      cache.depositos = rows;
      fillOptions($depOrigen, rows);
      fillOptions($depDestino, rows);
    } catch (err) {
      console.error("Error cargando depósitos:", err);
      setError($depOrigen, "Error al cargar");
      setError($depDestino, "Error al cargar");
      window.Swal?.fire?.("Error", "No se pudieron cargar los depósitos.", "error");
    }
  }

  async function cargarAmbientes($selectAmb, depositoId, side /* 'origen'|'destino' */) {
    if (!depositoId) {
      $selectAmb.prop("disabled", true);
      if ($selectAmb.data("select2")) $selectAmb.val(null).trigger("change");
      return;
    }
    if (Array.isArray(cache.ambientes[depositoId])) {
      fillOptions($selectAmb, cache.ambientes[depositoId], { value: 'id', text: 'nombre', extra: 'code' });
      return;
    }
    setLoading($selectAmb, "Cargando ambientes…");
    try {
      if (side === 'origen') {
        if (lastAmbFetchControllerOrigen) lastAmbFetchControllerOrigen.abort();
        lastAmbFetchControllerOrigen = new AbortController();
      } else {
        if (lastAmbFetchControllerDestino) lastAmbFetchControllerDestino.abort();
        lastAmbFetchControllerDestino = new AbortController();
      }
      const controller = (side === 'origen') ? lastAmbFetchControllerOrigen : lastAmbFetchControllerDestino;
      const json = await fetchJSON(API.ambientesByDeposito(depositoId), controller);
      const rows = normalizeList(json);
      cache.ambientes[depositoId] = rows;
      fillOptions($selectAmb, rows, { value: 'id', text: 'nombre', extra: 'code' });
    } catch (err) {
      if (err?.name === 'AbortError') return;
      console.error('Error cargando ambientes:', err);
      setError($selectAmb, 'Error al cargar');
      window.Swal?.fire?.('Error', 'No se pudieron cargar los ambientes.', 'error');
    }
  }

  async function cargarRacks($selectRack, depositoId, ambienteId, side /* 'origen'|'destino' */) {
    if (!depositoId) {
      $selectRack.prop("disabled", true);
      if ($selectRack.data("select2")) $selectRack.val(null).trigger("change");
      return;
    }
    const cacheKey = ambienteId ? `${depositoId}::${ambienteId}` : `${depositoId}::all`;
    if (Array.isArray(cache.racks[cacheKey])) {
      fillOptions($selectRack, cache.racks[cacheKey], { value: "id", text: "nombre", extra: "code" });
      return;
    }
    setLoading($selectRack, "Cargando racks…");

    try {
      if (side === "origen") {
        if (lastRackFetchControllerOrigen) lastRackFetchControllerOrigen.abort();
        lastRackFetchControllerOrigen = new AbortController();
      } else {
        if (lastRackFetchControllerDestino) lastRackFetchControllerDestino.abort();
        lastRackFetchControllerDestino = new AbortController();
      }
      const controller = (side === "origen") ? lastRackFetchControllerOrigen : lastRackFetchControllerDestino;
      const json = await fetchJSON(API.racks(depositoId, ambienteId), controller);
      const rows = normalizeList(json);
      cache.racks[cacheKey] = rows;
      fillOptions($selectRack, rows, { value: "id", text: "nombre", extra: "code" });
    } catch (err) {
      if (err?.name === "AbortError") return;
      console.error("Error cargando racks:", err);
      setError($selectRack, "Error al cargar");
      window.Swal?.fire?.("Error", "No se pudieron cargar los racks.", "error");
    }
  }

  // =======================
  // LAYOUT HTML
  // =======================
  function layoutLoaderHTML(depId, rackId) {
    return `
      <div class="card">
        <div class="card-body d-flex align-items-center gap-2">
          <div class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></div>
          <div>Cargando layout (depósito ${depId}, rack ${rackId})…</div>
        </div>
      </div>`;
  }

  async function renderRackLayout(target$div, depositoId, rackId, mode = "microtiles") {
    target$div.html(layoutLoaderHTML(depositoId, rackId));
    try {
      const res = await fetch(API.rackLayoutHTML(depositoId, rackId, mode), { credentials: "same-origin" });
      if (!res.ok) throw new Error(`HTTP ${res.status}`);
      const html = await res.text();
      target$div.html(html);
      bindLayoutClicks(); // re-bind después de re-render
    } catch (err) {
      console.error("[rotacion] Error renderizando layout:", err);
      target$div.html(`<div class="alert alert-danger">No se pudo cargar el layout del depósito/rack seleccionado.</div>`);
      window.Swal?.fire?.("Error", "No se pudo cargar el layout.", "error");
    }
  }

  // =======================
  // LIMPIEZA LADO DESTINO
  // =======================
  function clearDestinoSide(showPlaceholder = true) {
    // quitar selección visual destino
    if (selectedTo?.el) $(selectedTo.el).removeClass('is-selected-to');
    selectedTo = null;

    // limpiar selects destino y deshabilitar rack hasta que se elija depósito
    if ($depDestino.data('select2')) $depDestino.val(null).trigger('change');
    else $depDestino.val('');

    if ($rackDestino.data('select2')) $rackDestino.val(null).trigger('change');
    else $rackDestino.val('');
    $rackDestino.prop('disabled', true);

    // limpiar layout derecho
    if (showPlaceholder) {
      $layoutHasta.html('<div class="text-muted">Elegí un depósito y rack de destino para ver el layout…</div>');
    } else {
      $layoutHasta.empty();
    }

    // deshabilitar confirmar si aplica
    evaluarHabilitarConfirmar();
  }

  // =======================
  // SELECCIÓN EN LAYOUTS
  // =======================
  function bindLayoutClicks() {
    // ORIGEN: al hacer click → abrir modal de pallets de esa posición
    $layoutDesde.off('click', '.microtile-cell.clickable')
      .on('click', '.microtile-cell.clickable', function () {
        if (selectedFrom?.el) $(selectedFrom.el).removeClass('is-selected-from');
        selectedFrom = { pos_id: $(this).data('pos-id'), el: this };
        $(this).addClass('is-selected-from');
        openPalletsModalForFrom(); // abre modal y permite elegir 1..n pallets
      });

    // DESTINO: al hacer click → sólo marcar destino
    $layoutHasta.off('click', '.microtile-cell.clickable')
      .on('click', '.microtile-cell.clickable', function () {
        if (selectedTo?.el) $(selectedTo.el).removeClass('is-selected-to');
        selectedTo = { pos_id: $(this).data('pos-id'), el: this };
        $(this).addClass('is-selected-to');

        // Si hay items en plan sin destino asignado, setear este destino al último pendiente
        const pendingIndex = (() => {
          for (let i = plan.length - 1; i >= 0; i--) {
            if (!plan[i].to_pos_id) return i;
          }
          return -1;
        })();
        if (pendingIndex >= 0) {
          plan[pendingIndex].to_pos_id = +selectedTo.pos_id;
          try { console.log('[rotacion] Asignado destino a item del plan', plan[pendingIndex]); } catch {}
          evaluarHabilitarConfirmar();
        }
      });
  }

  // =======================
  // MODAL DE PALLETS (ORIGEN)
  // =======================
  function ensureModal() {
    // Si la vista ya define el modal (#modalPallets), no lo recreamos.
    if ($('#modalPallets').length) return;
    const html = `
<div class="modal fade" id="modalPallets" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Seleccionar pallets (posición origen)</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <div id="palletsContainer"><div class="text-muted">Cargando pallets…</div></div>
      </div>
      <div class="modal-footer">
        <!-- Compatibilidad con markup existente: usar btnPalletSelect si está presente en la vista -->
        <button id="btnPalletSelect" class="btn btn-primary" disabled>Seleccionar</button>
      </div>
    </div>
  </div>
</div>`;
    $('body').append(html);
  }

  function openModalPallets() {
    ensureModal();
    const modal = new bootstrap.Modal(document.getElementById('modalPallets'));
    modal.show();
    return modal;
  }

  async function openPalletsModalForFrom() {
    if (!selectedFrom?.pos_id) return;
    openModalPallets();
    await loadPalletsForPosition(selectedFrom.pos_id);
  }

  async function loadPalletsForPosition(posId) {
    const $ct = $('#palletsContainer').empty().append('<div class="text-muted">Cargando pallets…</div>');
    try {
      const res = await fetch(API.posicionPallets(posId), { credentials: 'same-origin' });
      if (!res.ok) throw new Error(`HTTP ${res.status}`);
      const json = await res.json();

      const rows = Array.isArray(json) ? json : (json?.data || []);
      if (!rows.length) {
        $ct.html('<div class="alert alert-info mb-0">Esta posición no tiene pallets.</div>');
        $('#btnAgregarAlPlan').prop('disabled', true);
        return;
      }

      // checkboxes para 1..n pallets
      const list = rows.map(p => `
        <label class="list-group-item d-flex align-items-center gap-2">
          <input class="form-check-input pallet-check" type="checkbox"
                 data-pallet-id="${p.id}" data-code="${p.code || ''}"
                 data-producto-id="${p.producto_id || ''}" data-lote-id="${p.lote_id || ''}"
                 data-uv="${p.uv || 0}" data-uc="${p.uc || 0}">
          <span class="fw-semibold">${p.code || ('PAL-' + p.id)}</span>
          <span class="badge text-bg-light">UV: ${p.uv ?? 0}</span>
          <span class="badge text-bg-light">UC: ${p.uc ?? 0}</span>
          ${p.producto_nombre ? `<span class="text-muted ms-auto">${p.producto_nombre}</span>` : ''}
        </label>
      `).join('');

  $ct.html(`<div class="list-group">${list}</div>`);
  const $selectBtn = $('#btnPalletSelect').length ? $('#btnPalletSelect') : $('#btnAgregarAlPlan');
  $selectBtn.prop('disabled', true);

      $ct.off('change', '.pallet-check').on('change', '.pallet-check', () => {
        const any = $ct.find('.pallet-check:checked').length > 0;
        $selectBtn.prop('disabled', !any);
      });

      $selectBtn.off('click').on('click', () => {

        const checked = $ct.find('.pallet-check:checked');
        const pallets = [];
        checked.each(function () {
          const $c = $(this);
          pallets.push({
            id: +$c.data('pallet-id'),
            code: String($c.data('code') || ''),
            producto_id: $c.data('producto-id') ? +$c.data('producto-id') : null,
            lote_id: $c.data('lote-id') ? +$c.data('lote-id') : null,
            uv: +($c.data('uv') || 0),
            uc: +($c.data('uc') || 0)
          });
        });

        // Log visible en consola de la selección realizada
        try {
          console.log('[rotacion] Selección origen confirmada:', {
            from_pos_id: selectedFrom?.pos_id,
            pallets,
            nota: 'Asigná el destino clickeando una posición a la derecha'
          });
        } catch (e) { /* noop */ }

        plan.push({
          deposito_from: +($depOrigen.val() || 0),
          deposito_to:   ($depDestino.val() ? +$depDestino.val() : null),
          from_pos_id:   +selectedFrom.pos_id,
          to_pos_id:     selectedTo?.pos_id ? +selectedTo.pos_id : null,
          pallets,
          ts: new Date().toISOString()
        });

  // limpiar selección visual (con guardas)
  if (selectedFrom?.el) $(selectedFrom.el).removeClass('is-selected-from');
  if (selectedTo?.el) $(selectedTo.el).removeClass('is-selected-to');
  selectedFrom = null; selectedTo = null;

        // cerrar modal y habilitar confirmar
        const modalEl = document.getElementById('modalPallets');
        if (modalEl) {
          const inst = bootstrap.Modal.getInstance(modalEl) || new bootstrap.Modal(modalEl);
          inst.hide();
        }
        evaluarHabilitarConfirmar();

        window.Swal?.fire?.('Listo', 'Se agregó el movimiento al plan.', 'success');
      });

    } catch (err) {
      console.error('[rotacion] Error cargando pallets:', err);
      $('#palletsContainer').html('<div class="alert alert-danger mb-0">No se pudo cargar la lista de pallets.</div>');
      const $selectBtn = $('#btnPalletSelect').length ? $('#btnPalletSelect') : $('#btnAgregarAlPlan');
      $selectBtn.prop('disabled', true);
    }
  }

  // =======================
  // CONFIRMAR (guardar en buffer)
  // =======================
  async function commitPlanToBuffer() {
    if (!plan.length) {
      return window.Swal?.fire?.('Atención', 'No hay movimientos en el plan.', 'warning');
    }
    // Validar que todos tengan destino
    const missing = plan.filter(p => !p.to_pos_id).length;
    if (missing > 0) {
      return window.Swal?.fire?.('Atención', `Falta asignar DESTINO a ${missing} movimiento(s).`, 'warning');
    }
    try {
      const res = await fetch(API.moveBuffer, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'same-origin',
        body: JSON.stringify({ items: plan }) // el PHP debe aceptar items[]
      });
      const json = await res.json();
      if (!res.ok || json?.ok === false) throw new Error(json?.error || 'Error al guardar buffer');

      // Luego del buffer, ejecutar commit real (wh_move + actualizar posiciones)
      const res2 = await fetch(API.moveCommit, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'same-origin',
        body: JSON.stringify({ items: plan })
      });
      const json2 = await res2.json();
      if (!res2.ok || json2?.ok === false) throw new Error(json2?.error || 'Error al confirmar movimientos');

      // limpiar plan luego de commit exitoso
      plan.length = 0;
      $btnConfirm.prop('disabled', true);
      window.Swal?.fire?.('Éxito', 'Movimientos confirmados.', 'success');

      // recargar layouts para reflejar cambios
  const depO = $depOrigen.val(), rackO = $rackOrigen.val();
  if (depO && rackO) await renderRackLayout($layoutDesde, depO, rackO, "microtiles");
  // limpiar lado destino para permitir un nuevo movimiento
  clearDestinoSide(true);

    } catch (err) {
      console.error('[rotacion] commit buffer error:', err);
      window.Swal?.fire?.('Error', String(err?.message || 'No se pudieron confirmar los movimientos.'), 'error');
    }
  }

  // =======================
  // INIT
  // =======================
  $(document).ready(async function () {
    // 1) Select2 opcional
    const hasSelect2 = await ensureSelect2Available();
    if (hasSelect2) {
      initSelect2IfAvailable($depOrigen,  "Depósito (origen)…");
      initSelect2IfAvailable($ambOrigen,  "Ambiente (origen)…");
      initSelect2IfAvailable($rackOrigen, "Rack (origen)…");
      initSelect2IfAvailable($depDestino, "Depósito (destino)…");
      initSelect2IfAvailable($ambDestino, "Ambiente (destino)…");
      initSelect2IfAvailable($rackDestino,"Rack (destino)…");
    } else {
      console.warn("[rotacion] Continuando sin Select2 (fallback nativo).");
    }

    // 2) Estados iniciales
  $ambOrigen.prop("disabled", true);
  $ambDestino.prop("disabled", true);
  $rackOrigen.prop("disabled", true);
  $rackDestino.prop("disabled", true);
    $btnConfirm.prop("disabled", true);

    // 3) Cargar catálogos
    await cargarDepositos();

    // 4) Dependencias selects

    $depOrigen.on("change", function () {
      const depId = $(this).val();
      if ($ambOrigen.data("select2")) $ambOrigen.val(null).trigger("change");
      if ($rackOrigen.data("select2")) $rackOrigen.val(null).trigger("change");
      cargarAmbientes($ambOrigen, depId, "origen");
      $ambOrigen.prop('disabled', !depId);
      cargarRacks($rackOrigen, depId, $ambOrigen.val(), "origen");
      evaluarHabilitarConfirmar();
    });

    $depDestino.on("change", function () {
      const depId = $(this).val();
      if ($ambDestino.data("select2")) $ambDestino.val(null).trigger("change");
      if ($rackDestino.data("select2")) $rackDestino.val(null).trigger("change");
      cargarAmbientes($ambDestino, depId, "destino");
      $ambDestino.prop('disabled', !depId);
      cargarRacks($rackDestino, depId, $ambDestino.val(), "destino");
      evaluarHabilitarConfirmar();
    });

    $ambOrigen.on('change', function () {
      const depId = $depOrigen.val();
      if ($rackOrigen.data("select2")) $rackOrigen.val(null).trigger("change");
      cargarRacks($rackOrigen, depId, $(this).val(), 'origen');
      evaluarHabilitarConfirmar();
    });

    $ambDestino.on('change', function () {
      const depId = $depDestino.val();
      if ($rackDestino.data("select2")) $rackDestino.val(null).trigger("change");
      cargarRacks($rackDestino, depId, $(this).val(), 'destino');
      evaluarHabilitarConfirmar();
    });

  $depOrigen.add($rackOrigen).add($depDestino).add($rackDestino).on("change", evaluarHabilitarConfirmar);

    // 5) Submits → render layouts
    $formOrigen.on("submit", function (ev) {
      ev.preventDefault();
      const depId = $depOrigen.val();
      const rackId = $rackOrigen.val();
      if (!depId || !rackId) { return window.Swal?.fire?.("Atención", "Seleccioná Depósito y Rack (origen).", "warning"); }
      renderRackLayout($layoutDesde, depId, rackId, "microtiles");
    });

    $formDestino.on("submit", function (ev) {
      ev.preventDefault();
      const depId = $depDestino.val();
      const rackId = $rackDestino.val();
      if (!depId || !rackId) { return window.Swal?.fire?.("Atención", "Seleccioná Depósito y Rack (destino).", "warning"); }
      renderRackLayout($layoutHasta, depId, rackId, "microtiles");
    });

    // 6) Confirmar → guardar en buffer
    $btnConfirm.off('click').on('click', commitPlanToBuffer);

    // 7) Cancelar → limpiar plan y destino
    $btnCancel.off('click').on('click', function () {
      // quitar selecciones visuales
      if (selectedFrom?.el) $(selectedFrom.el).removeClass('is-selected-from');
      if (selectedTo?.el) $(selectedTo.el).removeClass('is-selected-to');
      selectedFrom = null; selectedTo = null;

      // limpiar plan
      if (plan.length) plan.length = 0;
      clearDestinoSide(true);
      $btnConfirm.prop('disabled', true);
      try { console.log('[rotacion] Plan cancelado y destino limpio'); } catch {}
      window.Swal?.fire?.('Cancelado', 'Se limpió el plan y el formulario de destino.', 'info');
    });
  });
})();
