// public/page-scripts/pages/ingresos/packing_scripts.js
(function (window, $) {
  "use strict";

  // -------------------------
  // Utils
  // -------------------------
  const baseUrl =
    document.querySelector('meta[name="base-url"]')?.getAttribute("content") ||
    (typeof window.BASE_URL !== "undefined" ? window.BASE_URL : "/");

  function joinUrl(p) {
    const b = baseUrl.endsWith("/") ? baseUrl : baseUrl + "/";
    return b + String(p || "").replace(/^\/+/, "");
  }

  function notify(type, title, text) {
    if (window.Swal) {
      const icon = type === "error" ? "error" : type === "success" ? "success" : "info";
      window.Swal.fire({ icon, title, text, timer: 2600, showConfirmButton: false });
    } else {
      const prefix = type === "error" ? "ERROR: " : type === "success" ? "OK: " : "";
      alert(prefix + (title ? title + " - " : "") + (text || ""));
    }
  }

  function isNumeric(val) {
    return typeof val === "number" || (!!val && !isNaN(val));
  }

  // Detecta si el UI está en modo oscuro (varias heurísticas)
  function isDarkMode() {
    try {
      const doc = document.documentElement;
      // Bootstrap 5 puede usar data-bs-theme
      if (doc && doc.getAttribute('data-bs-theme') === 'dark') return true;
      // clases comunes
      if (document.body && (document.body.classList.contains('dark') || document.body.classList.contains('dark-mode'))) return true;
      // prefer-color-scheme
      if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) return true;
    } catch (e) {
      // ignore
    }
    return false;
  }

  function formatNumber(n) {
    if (!isNumeric(n)) return n;
    const parts = String(Math.trunc(Number(n))).split("");
    let out = "", cnt = 0;
    while (parts.length) {
      out = parts.pop() + out;
      cnt++;
      if (parts.length && cnt % 3 === 0) out = "." + out;
    }
    return out;
  }

  function sumColumn(api, colIdx) {
    let total = 0;
    api
      .column(colIdx, { page: "current" })
      .data()
      .each(function (v) {
        const num = typeof v === "string" ? Number(v.replace(/\./g, "").replace(",", ".")) : Number(v);
        if (!isNaN(num)) total += num;
      });
    return total;
  }

  // -------------------------
  // Estado local
  // -------------------------
  let currentBatch = null;
  let dtPreview = null;
  let dtMoves = null;
  let dtItems = null;
  let lastPLId = null; // <- PL confirmado (para "Ver detalle")

  // -------------------------
  // DOM
  // -------------------------
  const $form = $("#frmUploadPL");
  const $btnConfirm = $("#btnConfirmPL");
  const $tblPreview = $("#tblPL");
  const $tblMoves = $("#tblPLMoves");
  const $cardsHost = $("#plSummaryCardsHost");

  function getPreviewCard() {
    return $tblPreview.length ? $tblPreview.closest(".card") : $();
  }

  // -------------------------
  // Preview DataTable (reconstruye thead/tbody)
  // -------------------------
  function buildPreviewTable(columns, data) {
    if (!$tblPreview.length) return;

    const $card = getPreviewCard();
    if ($card.length) $card.show();

    if (dtPreview) {
      dtPreview.clear();
      dtPreview.destroy(true);
      dtPreview = null;
    }

    const cols = Array.isArray(columns) ? columns : [];
    const theadHtml =
      "<tr>" + cols.map(c => `<th>${(c.title || c.data || "").toString()}</th>`).join("") + "</tr>";

    if ($tblPreview.find("thead").length === 0) $tblPreview.prepend("<thead></thead>");
    $tblPreview.find("thead").html(theadHtml);

    if ($tblPreview.find("tbody").length === 0) $tblPreview.append("<tbody></tbody>");
    else $tblPreview.find("tbody").empty();

    const dtCols = cols.map((c) => ({
      data: c.data,
      title: c.title || c.data,
      render: function (val, type) {
        if (c.type === "number") {
          if (type === "display" || type === "filter") return formatNumber(val);
        }
        return val;
      },
    }));

    dtPreview = $tblPreview.DataTable({
      data: Array.isArray(data) ? data : [],
      columns: dtCols,
      deferRender: true,
      paging: true,
      pageLength: 25,
      order: [],
      language: { url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json" },
      footerCallback: function () {
        if ($tblPreview.find("tfoot").length === 0) {
          const tfoot = document.createElement("tfoot");
          const tr = document.createElement("tr");
          for (let i = 0; i < dtCols.length; i++) {
            const th = document.createElement("th");
            tr.appendChild(th);
          }
          tfoot.appendChild(tr);
          $tblPreview.append(tfoot);
        }
        const api = this.api();
        dtCols.forEach((c, idx) => {
          let text = "";
          if (c.type === "number") {
            const total = sumColumn(api, idx);
            text = formatNumber(total);
          }
          $(api.column(idx).footer()).html(text);
        });
      },
    });
  }

  // -------------------------
  // Moves DataTable (placeholder futuro)
  // -------------------------
  function buildMovesTable(columns, data) {
    if (!$tblMoves.length) return;

    if (dtMoves) {
      dtMoves.clear();
      dtMoves.destroy(true);
      dtMoves = null;
    }

    let dtCols = Array.isArray(columns) && columns.length
      ? columns.map((c) => ({
          data: c.data,
          title: c.title || c.data,
          render: function (val, type) {
            if (c.type === "number") {
              if (type === "display" || type === "filter") return formatNumber(val);
            }
            return val;
          },
        }))
      : [
          { data: "id",          title: "ID",         type: "number" },
          { data: "move_code",   title: "Move code" },
          { data: "tipo",        title: "Tipo" },
          { data: "status",      title: "Status" },
          { data: "pallet",      title: "Pallet" },
          { data: "position",    title: "Posición" },
          { data: "started_at",  title: "Iniciado" },
          { data: "finished_at", title: "Finalizado" },
        ];

    if ($tblMoves.find("thead").length === 0) $tblMoves.prepend("<thead></thead>");
    $tblMoves.find("thead").html(
      "<tr>" + dtCols.map(c => `<th>${(c.title || c.data || "").toString()}</th>`).join("") + "</tr>"
    );
    if ($tblMoves.find("tbody").length === 0) $tblMoves.append("<tbody></tbody>");
    else $tblMoves.find("tbody").empty();

    dtMoves = $tblMoves.DataTable({
      data: Array.isArray(data) ? data : [],
      columns: dtCols,
      deferRender: true,
      paging: true,
      pageLength: 25,
      order: [],
      language: { url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json" },
      footerCallback: function () {
        if ($tblMoves.find("tfoot").length === 0) {
          const tfoot = document.createElement("tfoot");
          const tr = document.createElement("tr");
          for (let i = 0; i < dtCols.length; i++) {
            const th = document.createElement("th");
            tr.appendChild(th);
          }
          tfoot.appendChild(tr);
          $tblMoves.append(tfoot);
        }
        const api = this.api();
        dtCols.forEach((c, idx) => {
          let text = "";
          if (c.type === "number") {
            const total = sumColumn(api, idx);
            text = formatNumber(total);
          }
          $(api.column(idx).footer()).html(text);
        });
      },
    });
  }

  // -------------------------
  // Cards resumen (3)
  // -------------------------
  function renderSummaryCards(summary) {
    if (!$cardsHost.length) return;
    const s = summary || {};
    $cardsHost.empty();

    const cards = [
      {
        icon: "fa-solid fa-barcode",
        color: "primary",
        title: "SKUs importados",
        value: formatNumber(s.skus_importados || 0),
        subtitle: `Nuevos: ${formatNumber(s.skus_nuevos ?? 0)}`,
      },
      {
        icon: "fa-solid fa-boxes-stacked",
        color: "success",
        title: "Ítems importados",
        value: formatNumber(s.items_importados || 0),
        subtitle: `Lotes: ${formatNumber(s.lotes_importados || 0)}${s.lotes_nuevos != null ? ` (${formatNumber(s.lotes_nuevos)} nuevos)` : ""}`,
      },
      {
        icon: "fa-solid fa-layer-group",
        color: "info",
        title: "UC totales",
        value: formatNumber(s.uc_totales || 0),
        subtitle: `Cajas: ${formatNumber(s.cajas || 0)} · UC sueltas: ${formatNumber(s.uc_sueltas_est || 0)}`,
      },
    ];

    cards.forEach((c) => {
      const $card = $(`
        <div class="col-12 col-md-4">
          <div class="card border-${c.color} shadow-sm">
            <div class="card-body d-flex align-items-center">
              <div class="me-3 text-${c.color}">
                <i class="${c.icon} fa-2x"></i>
              </div>
              <div>
                <h6 class="mb-0 text-muted">${c.title}</h6>
                <h5 class="my-1">${c.value}</h5>
                <p class="mb-0 small text-muted">${c.subtitle}</p>
              </div>
            </div>
          </div>
        </div>
      `);
      $cardsHost.append($card);
    });
  }

  // -------------------------
  // Línea de sugerencias (debajo de las cards)
  // -------------------------
  function renderHintsLine(hints) {
    if (!$cardsHost.length) return;
    const h = hints || {};

    $("#plHintsLine").remove();

    const palletsCount = Number(h.pallets_count || 0);
    const positionsCount = Number(h.positions_count || 0);

    const topPal = (Array.isArray(h.pallets_top) ? h.pallets_top : []).slice(0, 6);
    const topPos = (Array.isArray(h.positions_top) ? h.positions_top : []).slice(0, 6);

    const palList = topPal.map(x => `${x.hint} (${x.c})`).join(" · ");
    const posList = topPos.map(x => `${x.hint} (${x.c})`).join(" · ");
    const hasPos = positionsCount > 0 || posList.length > 0;

    const $line = $(`
      <div id="plHintsLine" class="col-12">
        <div class="card border-0 bg-transparent">
          <div class="card-body py-2">
            <div class="d-flex flex-column flex-md-row gap-3 align-items-start align-items-md-center">
              <div class="text-primary">
                <i class="fa-solid fa-pallet"></i>
                <strong>Pallets:</strong> ${formatNumber(palletsCount)}${palList ? ` · <span class="text-muted">${palList}</span>` : ""}
              </div>
              ${hasPos ? `
                <div class="text-info">
                  <i class="fa-solid fa-location-dot"></i>
                  <strong>Posiciones:</strong> ${formatNumber(positionsCount)}${posList ? ` · <span class="text-muted">${posList}</span>` : ""}
                </div>
              ` : ""}
            </div>
          </div>
        </div>
      </div>
    `);

    $cardsHost.append($line);
  }

  // -------------------------
  // Modal de detalle (creación perezosa)
  // -------------------------
  function ensureItemsModal() {
    if (document.getElementById("plItemsModal")) return;

    const html = `
<div class="modal fade" id="plItemsModal" tabindex="-1" aria-labelledby="plItemsModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="plItemsModalLabel"><i class="fa-solid fa-list me-2"></i>Detalle del Packing List</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <div class="table-responsive">
          <table id="tblPLItems" class="table table-striped w-100">
            <thead><tr></tr></thead>
            <tbody></tbody>
            <tfoot></tfoot>
          </table>
        </div>
      </div>
      <div class="modal-footer">
        <small class="text-muted me-auto" id="plItemsCaption"></small>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
      </div>
    </div>
  </div>
</div>`;
    document.body.insertAdjacentHTML("beforeend", html);

    // Aplicar ajustes visuales si estamos en modo oscuro
    const applyModalTheme = () => {
      const modalEl = document.getElementById("plItemsModal");
      if (!modalEl) return;
      const table = modalEl.querySelector('#tblPLItems');
      const modalContent = modalEl.querySelector('.modal-content');
      const btnClose = modalEl.querySelector('.btn-close');

      if (isDarkMode()) {
        if (table && !table.classList.contains('table-dark')) table.classList.add('table-dark');
        if (modalContent) {
          modalContent.classList.add('bg-dark', 'text-white');
        }
        if (btnClose) btnClose.classList.add('btn-close-white');
        // caption text color
        const caption = modalEl.querySelector('#plItemsCaption');
        if (caption) caption.classList.remove('text-muted');
        if (caption) caption.classList.add('text-light');
      } else {
        if (table) table.classList.remove('table-dark');
        if (modalContent) {
          modalContent.classList.remove('bg-dark', 'text-white');
        }
        if (btnClose) btnClose.classList.remove('btn-close-white');
        const caption = modalEl.querySelector('#plItemsCaption');
        if (caption) caption.classList.remove('text-light');
        if (caption) caption.classList.add('text-muted');
      }
    };

    // Aplicar inmediatamente y también al mostrarse (por si cambió el tema mientras)
    applyModalTheme();
    const modalEl = document.getElementById("plItemsModal");
    if (modalEl) {
      modalEl.addEventListener('show.bs.modal', applyModalTheme);
    }
  }

  function openItemsModal(plId) {
    if (!plId) {
      return notify("error", "Detalle", "No hay PL confirmado para ver detalle.");
    }
    ensureItemsModal();

    const $modal = $("#plItemsModal");
    const $tbl = $("#tblPLItems");
    const $caption = $("#plItemsCaption");

    // Limpieza DataTable si ya existía
    if (dtItems) {
      dtItems.clear();
      dtItems.destroy(true);
      dtItems = null;
    }
    $tbl.find("thead tr").empty();
    $tbl.find("tbody").empty();
    $tbl.find("tfoot").empty();

    // Cargar datos
    $.ajax({
      url: joinUrl(`api/operaciones/pl_items.php`),
      method: "GET",
      dataType: "json",
      data: { pl_id: plId },
    })
      .done(function (res) {
        if (!res || res.ok !== true) {
          const msg = (res && (res.message || res.error)) || "No se pudo cargar el detalle";
          return notify("error", "Detalle", msg);
        }

        // Cabecera del modal
        $("#plItemsModalLabel").text(`Detalle del Packing List ${res.packinglist?.codigo || `#${plId}`}`);
        $caption.text(`${res.packinglist?.cliente_ref || ""} · Fecha: ${res.packinglist?.fecha || ""}`);

        // Columnas
        const cols = Array.isArray(res.columns) ? res.columns : [];
        const theadHtml = "<tr>" + cols.map(c => `<th>${(c.title || c.data || "").toString()}</th>`).join("") + "</tr>";
        $tbl.find("thead").html(theadHtml);
        if ($tbl.find("tbody").length === 0) $tbl.append("<tbody></tbody>");
        if ($tbl.find("tfoot").length === 0) {
          const tf = document.createElement("tfoot");
          const tr = document.createElement("tr");
          for (let i = 0; i < cols.length; i++) tr.appendChild(document.createElement("th"));
          tf.appendChild(tr);
          $tbl.append(tf);
        }

        // Defs DataTable
        const dtCols = cols.map((c) => ({
          data: c.data,
          title: c.title || c.data,
          render: function (val, type) {
            if (c.type === "number") {
              if (type === "display" || type === "filter") return formatNumber(val);
            }
            return val;
          },
        }));

        dtItems = $tbl.DataTable({
          data: Array.isArray(res.data) ? res.data : [],
          columns: dtCols,
          deferRender: true,
          paging: true,
          pageLength: 25,
          order: [],
          language: { url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json" },
          footerCallback: function () {
            const api = this.api();
            dtCols.forEach((c, idx) => {
              let text = "";
              if (c.type === "number") {
                const total = sumColumn(api, idx);
                text = formatNumber(total);
              }
              $(api.column(idx).footer()).html(text);
            });
          },
        });

        // Re-apply modal/table theme after DataTable has rendered. DataTables may clone or
        // modify table markup, so ensure dark-mode classes are present (or removed) here.
        (function enforceModalThemeAfterDT() {
          const modalEl2 = document.getElementById('plItemsModal');
          if (!modalEl2) return;
          const table2 = modalEl2.querySelector('#tblPLItems');
          const modalContent2 = modalEl2.querySelector('.modal-content');
          const btnClose2 = modalEl2.querySelector('.btn-close');
          const caption2 = modalEl2.querySelector('#plItemsCaption');

          if (isDarkMode()) {
            if (table2 && !table2.classList.contains('table-dark')) table2.classList.add('table-dark');
            if (modalContent2) modalContent2.classList.add('bg-dark', 'text-white');
            if (btnClose2) btnClose2.classList.add('btn-close-white');
            if (caption2) { caption2.classList.remove('text-muted'); caption2.classList.add('text-light'); }

            // Ensure DataTables-generated elements (like the cloned header) also look correct.
            // Add a dark class to the wrapper so controls inherit readable colors.
            const wrapper = table2 ? table2.closest('.dataTables_wrapper') : null;
            if (wrapper) wrapper.classList.add('bg-dark', 'text-white');
          } else {
            if (table2) table2.classList.remove('table-dark');
            if (modalContent2) modalContent2.classList.remove('bg-dark', 'text-white');
            if (btnClose2) btnClose2.classList.remove('btn-close-white');
            if (caption2) { caption2.classList.remove('text-light'); caption2.classList.add('text-muted'); }
            const wrapper = table2 ? table2.closest('.dataTables_wrapper') : null;
            if (wrapper) wrapper.classList.remove('bg-dark', 'text-white');
          }
        })();

        // Mostrar modal
        const modal = new window.bootstrap.Modal(document.getElementById("plItemsModal"));
        modal.show();
      })
      .fail(function (jqXHR) {
        const j = jqXHR.responseJSON || {};
        const msg = j.message || j.error || "Error de comunicación";
        notify("error", "Detalle", msg);
      });
  }

  // -------------------------
  // Cargar previsualización
  // -------------------------
  function loadPreview(batchId) {
    $.ajax({
      url: joinUrl("api/operaciones/pl_preview.php"),
      method: "GET",
      dataType: "json",
      data: { batch_id: batchId },
    })
      .done(function (res) {
        if (!res || res.ok !== true) {
          const msg = (res && (res.message || res.error)) || "No se pudo cargar la previsualización";
          $btnConfirm.prop("disabled", true);
          return notify("error", "Previsualización", msg);
        }

        const columns = Array.isArray(res.columns) ? res.columns : [];
        const data = Array.isArray(res.data) ? res.data : [];

        buildPreviewTable(columns, data);
        $btnConfirm.prop("disabled", false);

        const okCount = Number(res.rows_ok || data.length || 0);
        const errCount = Number(res.rows_error || 0);
        notify("success", "Previsualización cargada", `Filas: ${formatNumber(okCount)}${errCount ? `, Errores: ${formatNumber(errCount)}` : ""}`);
      })
      .fail(function (jqXHR) {
        const j = jqXHR.responseJSON || {};
        const msg = j.message || j.error || "Error de comunicación";
        $btnConfirm.prop("disabled", true);
        notify("error", "Previsualización", msg);
      });
  }

  // -------------------------
  // Submit: Upload Excel (crea/elabora batch en server)
  // -------------------------
  $form.on("submit", function (e) {
    e.preventDefault();

    const fd = new FormData(this);
    const clienteId = String($("#cliente_id").val() || "").trim();
    if (!clienteId) {
      return notify("error", "Validación", "Debe seleccionar un cliente.");
    }

    $.ajax({
      url: joinUrl("api/operaciones/pl_upload.php"),
      method: "POST",
      data: fd,
      processData: false,
      contentType: false,
      dataType: "json",
    })
      .done(function (res) {
        if (!res || res.ok !== true) {
          const msg = (res && (res.message || res.error)) || "No se pudo importar el archivo";
          $btnConfirm.prop("disabled", true);
          return notify("error", "Importación", msg);
        }
        currentBatch = res.batch_id || res.batch || null;
        notify(
          "success",
          "Importación exitosa",
          `Batch #${currentBatch || "?"}. OK: ${formatNumber(res.rows_ok || 0)}${res.rows_error ? `, Errores: ${formatNumber(res.rows_error)}` : ""}`
        );

        if (!currentBatch) {
          $btnConfirm.prop("disabled", true);
          return notify("error", "Batch inválido", "El servidor no devolvió un batch_id válido.");
        }

        loadPreview(currentBatch);
      })
      .fail(function (jqXHR) {
        const j = jqXHR.responseJSON || {};
        const msg = j.message || j.error || "Error de comunicación";
        $btnConfirm.prop("disabled", true);
        notify("error", "Importación", msg);
      });
  });

  // -------------------------
  // Confirmar ingreso (endpoint REAL)
  // -------------------------
  $btnConfirm.on("click", function () {
    const clienteId = String($("#cliente_id").val() || "").trim();
    if (!clienteId) {
      return notify("error", "Validación", "Debe seleccionar un cliente.");
    }

    const observacion = String($("#observacion").val() || "").trim();

    const payload = {
      batch_id: currentBatch,
      cliente_id: clienteId,
      observacion: observacion,
      // force_new_product: 0,
      // overwrite: 0,
    };

    $.ajax({
      url: joinUrl("api/operaciones/pl_confirm_commit.php"),
      method: "POST",
      dataType: "json",
      data: payload,
    })
      .done(function (res) {
        if (!res || res.ok !== true) {
          const msg = (res && (res.message || res.error)) || "No se pudo confirmar el ingreso";
          return notify("error", "Confirmación", msg);
        }

        // Guardar PL id para el detalle
        lastPLId = res.packinglist_id || null;

        // Mantener la previsualización visible
        const $card = getPreviewCard();
        if ($card.length) $card.show();
        if (dtPreview) dtPreview.draw(false);

        // Deshabilitar confirmar + badge
        $btnConfirm.prop("disabled", true).addClass("disabled");
        if (!$btnConfirm.next(".badge").length) {
          $btnConfirm.after('<span class="badge bg-success ms-2">Confirmado</span>');
        }

        // Insertar/mostrar botón "Ver detalle"
        insertViewDetailButton();

        notify("success", "Ingreso confirmado", `PL #${lastPLId || "-"} consolidado.`);

        // Cards + sugerencias (reales)
        if (res.summary) renderSummaryCards(res.summary);
        if (res.hints) renderHintsLine(res.hints);

        // No movimientos en esta etapa
        buildMovesTable([], []);
      })
      .fail(function (jqXHR) {
        const j = jqXHR.responseJSON || {};
        const msg = j.message || j.error || "Error de comunicación";
        notify("error", "Confirmación", msg);
      });
  });

  // -------------------------
  // UI: botón "Ver detalle"
  // -------------------------
  function insertViewDetailButton() {
    // Lo colocamos al lado del botón Confirmar y del badge
    const $container = $btnConfirm.parent();
    if (!$container.length) return;

    if ($container.find("#btnViewPLDetail").length) return;

    const $btn = $(`
      <a href="recepcion" class="btn btn-primary ms-2"><i class="fa-solid fa-list me-1"></i> Ir a Recepcion</a>
    `);
    $btn.on("click", function () {
      if (!lastPLId) {
        return notify("error", "Detalle", "Aún no hay un PL consolidado para ver.");
      }
      openItemsModal(lastPLId);
    });

    $container.append($btn);
  }

})(window, jQuery);
