/* SOL – Ingresos / Recepción (Entrante)
 * Vista: views/ingresos/recepcion/index.php
 * Requisitos: jQuery, Bootstrap 5, DataTables
 *
 * IDs de la vista:
 *  - Select: #selPLPendiente  (con data-source opcional para pl_pendientes)
 *  - Botones: #btnCargarPL, #btnRecRefresh, #btnRecPrint
 *  - Auto-ubicar: #chkAutoUbicar   (si no hay pos_code, igual vamos a CUARENTENA)
 *  - Tabla: #tblRecepcionPallets (thead con 11 columnas + acciones)
 *
 * Mejora en este paso:
 * - Editor modal para PENDIENTES: ajustar UV/UC a recibir y repartir “buenas” vs “dañadas”.
 * - Si hay “dañadas” se crea un pallet adicional en CUARENTENA.
 * - Reaparece la columna “Estado” y en recibidos puede mostrar “RECIBIDO” o “DAÑADO (Motivo)”.
 * - Sin cambios de API ni DB todavía; motivos no persisten (se mostrarán en UI).
 */

(function () {
  'use strict';

  // ---------- Constantes ----------
  const DEFAULT_DEPOSITO = 'DEP1';
  const CUARENTENA_POS_FULL = 'DEP1-R99-CUAR-01'; // posición por defecto de CUARENTENA

  // ---------- Elementos ----------
  const $selPL      = () => $('#selPLPendiente');
  const $btnCargar  = () => $('#btnCargarPL');
  const $btnGuardar = () => $('#btnGuardarRecepcion');
  const $btnRefresh = () => $('#btnRecRefresh');
  const $btnPrint   = () => $('#btnRecPrint');
  const $autoUbicar = () => $('#chkAutoUbicar');
  const $tabla      = () => $('#tblRecepcionPallets');
  const $posInput   = () => $('#pos-code, #posCode, [data-pos-code]'); // opcional

  const $sum = {
    codigo:     () => $('#sum-pl-codigo'),
    cliente:    () => $('#sum-pl-cliente'),
    fecha:      () => $('#sum-pl-fecha'),
    palletsEsp: () => $('#sum-pallets-esp'),
    itemsEsp:   () => $('#sum-items-esp'),
    uvEsp:      () => $('#sum-uv-esp'),
    uvRec:      () => $('#sum-uv-rec'),
    ucEsp:      () => $('#sum-uc-esp'),
    ucRec:      () => $('#sum-uc-rec'),
    prodUniq:   () => $('#sum-prod-uniq'),
    discrep:    () => $('#sum-discrep'),
  };

  const $panelResumen = () => $('#resumenEsperado');
  const $btnRecalc    = () => $('#btnRecalcTotales');
  const $panelProg    = () => $('#panelProgreso');
  const $barProg      = () => $('#barProgreso');
  const $progRec      = () => $('#prog-recibido');
  const $progEsp      = () => $('#prog-esperado');
  const $boxDisc      = () => $('#boxDiscrepancias');
  const $litDisc      = () => $('#litDiscrepancias');
  const $panelFisico  = () => $('#panelRecepcionFisica');
  const $badgeHdr     = () => $('#badgeHdrSaved');
  const $btnPrintRecep = () => $('#btnImprimirRecepcion');

  function setRecepcionFisicaVisibility(enabled) {
    try {
      const $panel = $panelFisico();
      if (!$panel.length) return;
      if (enabled) {
        $panel.removeClass('d-none').prop('hidden', false).show();
      } else {
        $panel.addClass('d-none').prop('hidden', true).hide();
      }
    } catch (e) { /* ignore */ }
  }

  setRecepcionFisicaVisibility(false);
  try { $btnPrintRecep().addClass('d-none').prop('hidden', true); } catch (e) { /* ignore */ }

  function toggleRecepcionPrintButton(enabled) {
    try {
      const $btn = $btnPrintRecep();
      if (!$btn.length) return;
      if (enabled) {
        $btn.removeClass('d-none').prop('hidden', false);
      } else {
        $btn.addClass('d-none').prop('hidden', true);
      }
    } catch (e) { /* ignore */ }
  }

  function getTodayDateISO() {
    const now = new Date();
    const y = now.getFullYear();
    const m = String(now.getMonth() + 1).padStart(2, '0');
    const d = String(now.getDate()).padStart(2, '0');
    return `${y}-${m}-${d}`;
  }

  function getCurrentTimeHM() {
    const now = new Date();
    const h = String(now.getHours()).padStart(2, '0');
    const mi = String(now.getMinutes()).padStart(2, '0');
    return `${h}:${mi}`;
  }

  // ---------- API ----------
  const API_BASE = '/api/operaciones/recepcion.php';
  const endpoints = {
    plPendientes: (q = '') => `${API_BASE}?meta=pl_pendientes${q ? `&q=${encodeURIComponent(q)}` : ''}`,
    plResumen:    (plId)   => `${API_BASE}?meta=pl_resumen&pl_id=${encodeURIComponent(plId)}`,
    recepListBoth:(plId)   => `${API_BASE}?meta=recepcion_list&pl_id=${encodeURIComponent(plId)}`,
    recepGuardar:          () => `${API_BASE}?meta=recepcion_guardar`,
    recepGuardarItem:      () => `${API_BASE}?meta=recepcion_guardar_item`,
    posDefaults:           () => `${API_BASE}?meta=pos_default_codes`,
  };

  // Cache de posiciones por defecto (buenas/dañadas)
  let posDefaultsCache = null;
  async function getPosDefaults() {
    if (posDefaultsCache) return posDefaultsCache;
    try {
      const res = await getJSON(endpoints.posDefaults());
      if (res && res.ok) {
        posDefaultsCache = res.data || res;
      } else if (res && res.good_pos_code) {
        // json_ok envolvió data, pero por compat usamos raíz si viene directa
        posDefaultsCache = res;
      }
    } catch (e) { posDefaultsCache = null; }
    return posDefaultsCache;
  }

  // ---------- Estado ----------
  let currentPL = { id: null, codigo: null };
  let dt = null;
  // Keep client-side set of PLs that have been fully received during this session
  const completedPLs = new Set();

  // Guardamos motivos elegidos para simular “estado” visible en UI (no persiste aún)
  // clave: pallet_codigo (el que devuelva el backend), valor: {tipo:'DANADO', motivo:'...' }
  const ReceivedNotes = new Map();

  // ---------- Utils ----------
  const fmt = (n) => (n === null || n === undefined ? '' : new Intl.NumberFormat('es-AR').format(n));
  function toast(msg, type = 'info') {
    const $box = $('#app-alert, .app-alert').first();
    if ($box.length) {
      const cls = type === 'success' ? 'alert-success'
               : type === 'warning' ? 'alert-warning'
               : type === 'danger'  ? 'alert-danger'
               :                      'alert-info';
      $box.removeClass('d-none alert-success alert-warning alert-danger alert-info')
          .addClass(`alert ${cls}`).text(msg);
      setTimeout(() => $box.addClass('d-none'), 3500);
    } else {
      if (type === 'danger') alert(msg); else console.log(msg);
    }
  }

  // getJSON robusto
  async function getJSON(url, opts = {}) {
    const res = await fetch(url, {
      credentials: 'same-origin',
      headers: { 'Accept': 'application/json', ...(opts.headers || {}) },
      ...opts,
    });

    const ct = res.headers.get('content-type') || '';
    const text = await res.text();

    if (!res.ok) {
      throw new Error(`HTTP ${res.status} – ${text.slice(0, 300)}`);
    }
    if (text.trim() === '') return { ok: false, error: 'Respuesta vacía del servidor', raw: '' };

    if (ct.includes('application/json')) {
      try { return JSON.parse(text); } catch { return { ok:false, error:'JSON inválido del servidor', raw:text }; }
    }
    try { return JSON.parse(text); } catch { return { ok:false, error:'Respuesta no JSON del servidor', raw:text }; }
  }

  async function postJSON(url, payload) {
    return getJSON(url, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload || {}),
    });
  }

  // Recoger los metadatos del panel (si están presentes) y devolver un objeto
  function collectMeta() {
    const out = {};
    try {
      const f = $('#meta_fecha_ingreso').val(); if (f) out.fecha_ingreso = f;
      const ha = $('#meta_hora_arribo').val(); if (ha) out.hora_arribo = ha;
      const hi = $('#meta_hora_inicio').val(); if (hi) out.hora_inicio = hi;
      const hf = $('#meta_hora_fin').val(); if (hf) out.hora_fin = hf;
      const oc = $('#meta_operarios_cant').val(); if (oc !== undefined && oc !== null && String(oc) !== '') out.operarios_cant = Number(oc);
  const td = $('#meta_tipo_documento').val();
  if (td) out.tipo_documento = String(td).toUpperCase();
      const de = $('#meta_documento_entrada').val(); if (de) out.documento_entrada = de;
      const mv = $('#meta_movil_id').val(); if (mv) out.movil_id = mv;
      const ch = $('#meta_chofer_id').val(); if (ch) out.chofer_id = ch;
      const obs = $('#meta_observacion').val(); if (obs) out.observacion = obs;
    } catch (e) { /* ignore if elements missing */ }
    return out;
  }

  // Campos obligatorios para la recepción
  const META_REQUIRED_FIELDS = [
    { selector: '#meta_fecha_ingreso',     label: 'Fecha de ingreso', type: 'date' },
    { selector: '#meta_hora_arribo',       label: 'Hora de arribo',   type: 'time' },
    { selector: '#meta_hora_inicio',       label: 'Hora de inicio',   type: 'time' },
    { selector: '#meta_hora_fin',          label: 'Hora de fin',      type: 'time' },
    { selector: '#meta_operarios_cant',    label: 'Cantidad de operarios', type: 'number' },
    { selector: '#meta_tipo_documento',    label: 'Tipo de documento', type: 'select' },
    { selector: '#meta_documento_entrada', label: 'Documento',        type: 'text' },
    { selector: '#meta_movil_id',          label: 'Móvil',            type: 'select' },
    { selector: '#meta_chofer_id',         label: 'Chofer',           type: 'select' },
  ];

  function ensureMetaComplete() {
    let missing = [];
    let firstInvalid = null;

    META_REQUIRED_FIELDS.forEach(field => {
      const $el = $(field.selector);
      if (!$el.length) return;

      let raw = $el.val();
      let valid = true;

      switch (field.type) {
        case 'number':
          valid = raw !== undefined && raw !== null && String(raw).trim() !== '' && !isNaN(Number(raw)) && Number(raw) > 0;
          break;
        default:
          raw = typeof raw === 'string' ? raw.trim() : raw;
          valid = raw !== undefined && raw !== null && raw !== '' && raw !== '--' && raw !== '-- Seleccionar --';
          if (valid && field.selector === '#meta_tipo_documento') {
            const allowed = ['FACTURA', 'REMITO'];
            valid = allowed.includes(String(raw).toUpperCase());
          }
          break;
      }

      if (!valid) {
        missing.push(field.label);
        if (!$el.hasClass('is-invalid')) $el.addClass('is-invalid');
        if (!firstInvalid) firstInvalid = $el;
      } else {
        if ($el.hasClass('is-invalid')) $el.removeClass('is-invalid');
      }
    });

    if (missing.length) {
      toast('Completa los metadatos obligatorios: ' + missing.join(', '), 'danger');
      if (firstInvalid && firstInvalid.length && typeof firstInvalid.focus === 'function') {
        try { firstInvalid.trigger('focus'); } catch (e) { /* ignore */ }
      }
      return false;
    }
    return true;
  }

  // Ensure a select (Select2 or native) shows the requested selected option.
  // Retries a few times to handle race conditions where Select2 or catalog population
  // finishes after loadResumen runs.
  function ensureSelectShows($el, id, text, attempt = 0) {
    if (!id) return;
    const maxAttempts = 6;
    const delay = 120; // ms

    try {
      // If Select2 is active on this element
      if ($.fn.select2 && $el.hasClass && $el.hasClass('select2-hidden-accessible')) {
        // If option exists, select it; otherwise create it
        let opt = $el.find('option[value="' + id + '"]');
        if (!opt || opt.length === 0) {
          try {
            const newOpt = new Option(text || id, id, true, true);
            // Append native HTMLOptionElement
            $el.append(newOpt);
          } catch (e) { /* ignore append failures */ }
        } else {
          try { opt.prop('selected', true); } catch(e) { /* ignore */ }
        }
        try {
          $el.val(String(id)).trigger('change');
          // Trigger select2:select to force the displayed selection
          $el.trigger({ type: 'select2:select', params: { data: { id: id, text: text || id } } });
        } catch (e) { /* ignore trigger errors */ }

        // verify Select2 has the data; if not, retry
        try {
          const sdata = $el.select2 ? $el.select2('data') : null;
          const hasText = sdata && ((Array.isArray(sdata) && sdata.length && (sdata[0].id == id || sdata[0].text)) || (sdata.id == id));
          if (hasText) return; // done
        } catch (e) { /* ignore */ }
      } else {
        // native select: ensure option exists and is selected
        let opt = $el.find('option[value="' + id + '"]');
        if (!opt || opt.length === 0) {
          $el.append($('<option>', { value: id, text: text || id, selected: true }));
        } else {
          try { opt.prop('selected', true); } catch(e) { opt[0].selected = true; }
        }
        try { $el.val(String(id)); } catch(e) { /* ignore */ }
        return; // native select done
      }
    } catch (e) {
      // swallow errors to allow retries
    }

    if (attempt < maxAttempts) {
      setTimeout(function(){ ensureSelectShows($el, id, text, attempt + 1); }, delay);
    }
  }

  // ---------- Modal Editor ----------
  function ensureEditorModal() {
    if ($('#rec-editor-modal').length) return;
    const html = `
<div class="modal fade" id="rec-editor-modal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Recepción de ítem (ajustar cantidades)</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <form id="rec-editor-form" class="row g-3">
          <input type="hidden" name="pl_item_id" />
          <div class="col-12">
            <div><strong>Producto:</strong> <span id="m-prod">—</span></div>
            <div><strong>Lote:</strong> <span id="m-lote">—</span></div>
            <div class="text-muted small" id="m-expected">—</div>
            <div class="mt-2 small" id="m-pos-map" style="display:none">
              <span class="badge text-bg-success me-2">Buenas → <span id="m-pos-good">—</span></span>
              <span class="badge text-bg-warning">Dañadas → <span id="m-pos-dam">—</span></span>
            </div>
          </div>
          <div class="col-12">
            <hr class="my-2"/>
          </div>
          <div class="col-6">
            <label class="form-label">UV buenas</label>
            <input type="number" min="0" step="1" class="form-control" name="uv_buenas" value="0" />
          </div>
          <div class="col-6">
            <label class="form-label">UC buenas</label>
            <input type="number" min="0" step="1" class="form-control" name="uc_buenas" value="0" />
          </div>

          <div class="col-6">
            <label class="form-label">UV dañadas</label>
            <input type="number" min="0" step="1" class="form-control" name="uv_danadas" value="0" />
          </div>
          <div class="col-6">
            <label class="form-label">UC dañadas</label>
            <input type="number" min="0" step="1" class="form-control" name="uc_danadas" value="0" />
          </div>

          <div class="col-12">
            <label class="form-label">Motivo (si hay dañadas)</label>
            <select class="form-select" name="motivo">
              <option value="">—</option>
              <option value="DANADO">Daño físico / roto</option>
              <option value="INCONSISTENCIA_DOC">Inconsistencia documental</option>
              <option value="OTRO">Otro</option>
            </select>
          </div>

          <div class="col-12">
            <div class="form-text">
              • Si ingresás menos de lo esperado, el remanente continuará como <strong>PENDIENTE</strong>.<br/>
              • Si ingresás más de lo esperado, este paso no lo confirmará (lo trataremos como caso especial en un siguiente paso).
            </div>
          </div>
        </form>
      </div>
      <div class="modal-footer">
        <button id="rec-editor-submit" type="button" class="btn btn-primary">
          <span class="me-1" aria-hidden="true">✅</span>Confirmar recepción
        </button>
      </div>
    </div>
  </div>
</div>`;
    $('body').append(html);
  }

  function openEditorModal(row) {
    ensureEditorModal();
    const $m = $('#rec-editor-modal');
    const $f = $('#rec-editor-form');

    // Defaults desde la fila pendiente
    const plItemId = row.pl_item_id || row.id || null;
    const lote     = String(row.lote || '');
    const prod     = String(row.producto_txt || '');
    const uvEsp    = Number(row.uv_esp ?? row.uv ?? 0);
    const ucEsp    = Number(row.uc_esp ?? row.uc ?? 0);

    $f[0].reset();
    $f.find('[name=pl_item_id]').val(plItemId);

    $('#m-prod').text(prod || '—');
    $('#m-lote').text(lote || '—');
    $('#m-expected').text(`Esperado: UV ${fmt(uvEsp)} • UC ${fmt(ucEsp)}`);

    // Prefill: por defecto todo “bueno”
    $f.find('[name=uv_buenas]').val(uvEsp || 0);
    $f.find('[name=uc_buenas]').val(ucEsp || 0);
    $f.find('[name=uv_danadas]').val(0);
    $f.find('[name=uc_danadas]').val(0);
    $f.find('[name=motivo]').val('');

    const modal = bootstrap.Modal.getOrCreateInstance($m[0]);
    modal.show();

    // Guardamos “row” en el modal para usar al confirmar
    $m.data('row', row);

    // Cargar y mostrar mapa de posiciones (buenas/dañadas)
    getPosDefaults().then(def => {
      try {
        const good = def?.good_pos_code || '—';
        const dam  = def?.damaged_pos_code || '—';
        $('#m-pos-good').text(good || '—');
        $('#m-pos-dam').text(dam || '—');
        $('#m-pos-map').show();
      } catch (e) { /* ignore */ }
    }).catch(() => {/* ignore */});
  }

  // ---------- Carga de pendientes ----------
  async function loadPendientes() {
    try {
      const $sel = $selPL();
      const srcAttr = $sel.attr('data-source');
      const url = srcAttr && srcAttr.trim() ? srcAttr : endpoints.plPendientes('');
      const { ok, data, error, raw } = await getJSON(url);
      if (!ok) throw new Error(error || raw || 'No se pudo listar PL pendientes');

      const prev = $sel.val();
      $sel.empty().append($('<option>', { value: '', text: 'Seleccione un PL...' }));
      (data || []).forEach(r => {
        // Skip any PLs we've already completed in this session
        try { if (r && r.pl_id && completedPLs.has(String(r.pl_id))) return; } catch (e) { /* ignore */ }
        const pendingCount = Number(r.items_pending || r.items_pending === 0 ? r.items_pending : 0) || 0;
        const pendTxt = pendingCount > 0 ? ` (${pendingCount} pendientes)` : '';
        const txt = `${r.pl_codigo} | ${r.cliente || ''} | ${r.pl_fecha || ''}`;
        const $opt = $('<option>', { value: r.pl_id, text: txt + pendTxt });
        if (pendingCount === 0) $opt.attr('data-completed', '1');
        $sel.append($opt);

      });
      if (prev && $sel.find(`option[value="${prev}"]`).length) $sel.val(prev);
      // If no PLs were added (only the placeholder remains), show friendly message
      const $noMsg = $('#no-pl-msg');
      const hasOptions = $sel.find('option').length > 1;
      if (!hasOptions) {
        $noMsg.removeClass('d-none');
        $sel.prop('disabled', true);
        $btnCargar().prop('disabled', true);
      } else {
        $noMsg.addClass('d-none');
        $sel.prop('disabled', false);
        $btnCargar().prop('disabled', false);
      }
    } catch (e) {
      toast(`Error listando PL pendientes: ${e.message}`, 'danger');
    }
  }

  // No dropdown handler required; selection is done via the select element

  // ---------- Resumen ----------
  async function loadResumen(plId) {
    // show placeholder while loading and ensure it's hidden after (even on error)
    showSummaryLoading(true);
    try {
      const { ok, data, error, raw } = await getJSON(endpoints.plResumen(plId));
      if (!ok) throw new Error(error || raw || 'No se pudo obtener resumen del PL');

      currentPL.id = data.pl_id;
      currentPL.codigo = data.pl_codigo;

      $sum.codigo().text(data.pl_codigo || '—');
      $sum.cliente().text(data.cliente || '—');
      $sum.fecha().text(data.pl_fecha || '—');
  $sum.itemsEsp().text(fmt(data.items_esp || 0));
  // Mostrar items pendientes en el resumen si el endpoint lo provee
  try { $('#sum-items-pend').text(fmt(data.items_pending ?? 0)); } catch (e) { /* ignore */ }
      $sum.uvEsp().text(fmt(data.uv_esp || 0));
      $sum.uvRec().text(fmt(data.uv_recv || 0));
      $sum.ucEsp().text(fmt(data.uc_esp || 0));
      $sum.ucRec().text(fmt(data.uc_recv || 0));
    $sum.prodUniq().text(fmt(data.productos_unicos || 0));
      $sum.discrep().text(fmt(data.discrepancias || 0));
      $sum.palletsEsp().text('—');

      // Progreso por UV
      const uvEsp  = Number(data.uv_esp || 0);
      const uvRecv = Number(data.uv_recv || 0);
      const pct = uvEsp > 0 ? Math.min(100, Math.round((uvRecv / uvEsp) * 100)) : 0;

      $progRec().text(fmt(uvRecv));
      $progEsp().text(fmt(uvEsp));
      $barProg().css('width', `${pct}%`).text(`${pct}%`);

      $panelResumen().prop('hidden', false);
      $panelProg().prop('hidden', false);

      if ((data.discrepancias || 0) > 0) {
        $litDisc().text(fmt(data.discrepancias));
        $boxDisc().prop('hidden', false);
      } else {
        $boxDisc().prop('hidden', true);
      }
      try {
        const itemsPending = Number(data.items_pending ?? 0);
        const uvRecv = Number(data.uv_recv ?? 0);
        toggleRecepcionPrintButton(itemsPending === 0 && uvRecv > 0);
      } catch (e) { toggleRecepcionPrintButton(false); }
      // Header info: cliente, movil, chofer (placeholders in the view)
      try {
        $('#hdr-cliente').text(data.cliente || '-');
        $('#hdr-movil').text(data.movil || '-');
        $('#hdr-chofer').text(data.chofer || '-');
      } catch (e) { /* ignore */ }
      // Populate metadata inputs (so a reload shows saved values)
      try {
        if (data.fecha_ingreso) $('#meta_fecha_ingreso').val(data.fecha_ingreso);
        if (data.hora_arribo) $('#meta_hora_arribo').val(data.hora_arribo);
        if (data.hora_inicio) $('#meta_hora_inicio').val(data.hora_inicio);
        if (data.hora_fin) $('#meta_hora_fin').val(data.hora_fin);
        if (data.operarios_cant !== undefined && data.operarios_cant !== null) $('#meta_operarios_cant').val(data.operarios_cant);
  if (data.tipo_documento) $('#meta_tipo_documento').val(String(data.tipo_documento).toUpperCase());
        if (data.documento_entrada) $('#meta_documento_entrada').val(data.documento_entrada);
        if (data.observacion) $('#meta_observacion').val(data.observacion);
        // Update header saved badge heuristic: consider saved if at least one meta field exists
        (function(){
          const saved = !!(data.fecha_ingreso || data.hora_arribo || data.hora_inicio || data.hora_fin || (data.operarios_cant!=null) || data.tipo_documento || data.documento_entrada || data.movil_id || data.chofer_id || data.observacion);
          setHdrBadge(saved);
        })();
      } catch(e) { /* ignore */ }
      // Preselect móvil / chofer si el PL trae esos valores en el resumen
      try {
        // móvil preselect (Select2 remoto): fetch single and add option
        if (data.movil_id) {
          const mvId = String(data.movil_id);
          const $mov = $('#meta_movil_id');
          if ($.fn.select2 && $mov.hasClass('select2-hidden-accessible')) {
            fetchSingleMovil(mvId).then(function(item){
              try {
                const label = (item && (item.chapa || item.id)) ? (item.chapa || item.id) : mvId;
                // create option and mark it selected/defaultSelected for Select2
                let option = $mov.find('option[value="' + mvId + '"]');
                if (!option || option.length === 0) {
                  option = new Option(label, mvId, true, true);
                  option.selected = true;
                  option.defaultSelected = true;
                  $mov.append(option);
                } else {
                  option.prop('selected', true);
                }
                // notify Select2 to update
                try { 
                  $mov.val(mvId).trigger('change'); 
                  // ensure Select2 updates its displayed selection
                  try { $mov.trigger({ type: 'select2:select', params: { data: { id: mvId, text: label } } }); } catch(e) { /* ignore */ }
                } catch(e) { /* ignore */ }
                // Ensure the visual selection is applied (handles Select2 timing/race conditions)
                try { ensureSelectShows($mov, mvId, label); } catch(e) { /* ignore */ }
              } catch (e) { $mov.val(mvId).trigger('change'); }
            }).catch(function(){ $mov.val(mvId).trigger('change'); });
            // Even if the remote fetch failed, schedule a final verification to show selection
            try { ensureSelectShows($('#meta_movil_id'), mvId, data.movil || mvId); } catch(e) { /* ignore */ }
          } else {
            // If Select2 is not present, ensure the option exists and mark selected
            const existing = $mov.find('option[value="' + mvId + '"]');
            if (existing.length === 0) {
              const label = data.movil || mvId;
              $mov.append($('<option>', { value: mvId, text: label, selected: true }));
            } else {
              existing.prop('selected', true);
            }
            $mov.data('preselect', mvId).val(mvId);
            try { ensureSelectShows($mov, mvId, data.movil || mvId); } catch(e) { /* ignore */ }
          }
        }

        // chofer preselect
        if (data.chofer_id) {
          const chId = String(data.chofer_id);
          const $cho = $('#meta_chofer_id');
          if ($.fn.select2 && $cho.hasClass('select2-hidden-accessible')) {
            fetchSingleChofer(chId).then(function(item){
              try {
                const label = (item && (item.nombre || item.id)) ? (item.nombre || item.id) : chId;
                let option = $cho.find('option[value="' + chId + '"]');
                if (!option || option.length === 0) {
                  option = new Option(label, chId, true, true);
                  option.selected = true;
                  option.defaultSelected = true;
                  $cho.append(option);
                } else {
                  option.prop('selected', true);
                }
                try { 
                  $cho.val(chId).trigger('change'); 
                  // ensure Select2 updates its displayed selection
                  try { $cho.trigger({ type: 'select2:select', params: { data: { id: chId, text: label } } }); } catch(e) { /* ignore */ }
                } catch(e) { /* ignore */ }
                // Ensure the visual selection is applied (handles Select2 timing/race conditions)
                try { ensureSelectShows($cho, chId, label); } catch(e) { /* ignore */ }
              } catch (e) { $cho.val(chId).trigger('change'); }
            }).catch(function(){ $cho.val(chId).trigger('change'); });
            // Final check for native/Select2 visibility
            try { ensureSelectShows($('#meta_chofer_id'), chId, data.chofer || chId); } catch(e) { /* ignore */ }
          } else {
            const existing = $cho.find('option[value="' + chId + '"]');
            if (existing.length === 0) {
              const label = data.chofer || chId;
              $cho.append($('<option>', { value: chId, text: label, selected: true }));
            } else {
              existing.prop('selected', true);
            }
            $cho.data('preselect', chId).val(chId);
            try { ensureSelectShows($cho, chId, data.chofer || chId); } catch(e) { /* ignore */ }
          }
        }
        // If after all attempts the selected option has no visible text, force a repaint by cloning the select
        try {
          (function(){
            const $m = $('#meta_movil_id');
            const selText = ($m.find('option:selected').text() || '').trim();
            if (!$m.length || selText.length > 0) return;
            // clone including events, replace and restore value
            const val = $m.val();
            const $clone = $m.clone(true);
            $m.replaceWith($clone);
            try { $clone.val(val); } catch(e) { /* ignore */ }
          })();
        } catch(e) { /* ignore */ }
        try {
          (function(){
            const $c = $('#meta_chofer_id');
            const selText = ($c.find('option:selected').text() || '').trim();
            if (!$c.length || selText.length > 0) return;
            const val = $c.val();
            const $clone = $c.clone(true);
            $c.replaceWith($clone);
            try { $clone.val(val); } catch(e) { /* ignore */ }
          })();
        } catch(e) { /* ignore */ }
      } catch (e) { /* ignore */ }
    } finally {
      showSummaryLoading(false);
    }
  }

  // --- Badge helpers ---
  function setHdrBadge(saved) {
    try {
      const $b = $badgeHdr();
      if ($b && $b.length) {
        if (saved) {
          $b.removeClass('text-bg-secondary text-bg-warning').addClass('text-bg-success').text('Cabecera: Guardada');
        } else {
          $b.removeClass('text-bg-success text-bg-warning').addClass('text-bg-secondary').text('Cabecera: Pendiente');
        }
      }
    } catch(e) { /* ignore */ }
    try { setRecepcionFisicaVisibility(!!saved); } catch (e) { /* ignore */ }
  }

  function showSummaryLoading(loading) {
    try {
      const placeholders = ['codigo','cliente','fecha','palletsEsp','itemsEsp','uvEsp','uvRec','ucEsp','ucRec','prodUniq','discrep'];
      placeholders.forEach(k => {
        const el = $sum[k] && $sum[k]();
        if (!el || !el.length) return;
        if (loading) {
          el.data('orig', el.text());
          el.text('…');
        } else {
          // Do not restore previous text here: new values are set by loadResumen
          el.removeData('orig');
        }
      });
    } catch (e) { /* ignore */ }
  }

  // ---------- Tabla: fusiona pendientes + recibidos ----------
  async function loadRecepcionList(plId) {
    const { ok, data, error, raw } = await getJSON(endpoints.recepListBoth(plId));
    if (!ok) throw new Error(error || raw || 'No se pudo obtener la lista de recepción');

    let received = Array.isArray(data?.received) ? data.received : [];
    let pending  = Array.isArray(data?.pending)  ? data.pending  : [];

    // Fallbacks: some endpoints may return a flat array or use different keys
    if ((!received || received.length === 0) && (!pending || pending.length === 0)) {
      if (Array.isArray(data)) {
        // server returned an array of pending items directly
        pending = data;
      } else if (Array.isArray(data?.items)) {
        pending = data.items;
      } else if (Array.isArray(data?.data)) {
        pending = data.data;
      }
    }

    // Normalizamos a un esquema común para la grilla
    // helper: obtiene valor esperado intentando varias claves posibles
    function getExpectedVal(obj, baseKey) {
      if (!obj) return null;
      const v = (obj[`${baseKey}_esp`] ?? obj[`${baseKey}_expected`] ?? obj[`${baseKey}_esperado`] ?? obj[baseKey] ?? null);
      if (v === null || v === undefined || v === '') return null;
      // Coerce numeric strings to numbers when possible
      if (typeof v === 'number') return v;
      if (typeof v === 'string') {
        const n = Number(String(v).replace(/\./g, '').replace(/,/g, '.'));
        return isNaN(n) ? null : n;
      }
      return null;
    }

    const rowsReceived = received.map(r => ({
      type: 'received',
      pallet_codigo: r.pallet_codigo,
      producto_txt: r.producto_txt,
      lote: r.lote,
      fecha_produccion: r.fecha_produccion,
      fecha_vencimiento: r.fecha_vencimiento,
      uv_esp: getExpectedVal(r, 'uv') ?? (r.rem_uv ?? r.rem_uc ? (r.rem_uv ?? null) : null),
      uc_esp: getExpectedVal(r, 'uc') ?? (r.rem_uc ?? null),
      // Use the actual received values when present (API returns strings sometimes)
      uv:     (r.uv !== undefined && r.uv !== null) ? (Number(String(r.uv).replace(/\./g,'').replace(/,/g,'.')) || 0) : (getExpectedVal(r, 'uv_recv') ?? null),
      uc:     (r.uc !== undefined && r.uc !== null) ? (Number(String(r.uc).replace(/\./g,'').replace(/,/g,'.')) || 0) : (getExpectedVal(r, 'uc_recv') ?? null),
      estado: r.estado,
      pl_item_id: r.pl_item_id || null,
      ubicacion: r.ubicacion || CUARENTENA_POS_FULL,
      peso: r.peso ?? null,
    }));

    const rowsPending = pending.map(r => ({
      type: 'pending',
      pl_item_id: r.pl_item_id,
      pallet_codigo: r.pallet_codigo,
      producto_txt: r.producto_txt,
      lote: r.lote,
      fecha_produccion: r.fecha_produccion,
      fecha_vencimiento: r.fecha_vencimiento,
      uv_esp: getExpectedVal(r, 'uv') ?? (r.rem_uv ?? null),
      uc_esp: getExpectedVal(r, 'uc') ?? (r.rem_uc ?? null),
      uv:     getExpectedVal(r, 'uv_recv') ?? null,
      uc:     getExpectedVal(r, 'uc_recv') ?? null,
      estado: r.estado,
      ubicacion: '',
      peso: r.peso ?? null,
    }));

    const merged = [...rowsPending, ...rowsReceived];

    // If there are no pending items for this PL, remove it from the pending PL selector
    try {
      const pendingCount = (rowsPending || []).length;
      if (pendingCount === 0 && currentPL && currentPL.id) {
        // Remove option from selector so it no longer appears as pending
        const $sel = $selPL();
        const val = String(currentPL.id);
        $sel.find(`option[value="${val}"]`).remove();
        // Remember this PL as completed for this session to avoid re-adding it
        try { completedPLs.add(String(currentPL.id)); } catch (e) { /* ignore */ }
        toast('Este PL ya se completó y fue quitado de la lista de pendientes.', 'info');
      }
      else {
        // If there are pending items, ensure it isn't incorrectly marked as completed
        try { if (currentPL && currentPL.id) completedPLs.delete(String(currentPL.id)); } catch (e) { /* ignore */ }
      }
    } catch (e) { /* ignore */ }

    const $t = $tabla();
    if (dt) { try { dt.clear().destroy(); } catch {} }
    $t.find('tbody').empty();

    // función local para renderizar diferencias con colores
    function renderDiffCell(v) {
      if (v === null || v === undefined) return '';
      if (v === 0) return '<span class="text-success">OK</span>'; // zero -> green OK
      if (v < 0) return `<span class="text-danger">${fmt(Math.abs(v))}</span>`; // falta (negative diff -> missing)
      // v > 0 => exceso
      return `<span class="text-success">+${fmt(Math.abs(v))}</span>`;
    }

    // helper para sumar columna DataTables (pagina actual)
    function sumColumn(api, colIdx) {
      let total = 0;
      api
        .column(colIdx, { page: 'current' })
        .data()
        .each(function (v) {
          if (v === null || v === undefined || v === '') return;
          // v may be number or string with formatting
          const num = typeof v === 'number' ? v : Number(String(v).replace(/\./g, '').replace(/,/g, '.'));
          if (!isNaN(num)) total += num;
        });
      return total;
    }

    // Simple property-based columns (restore prior working approach)
    const cols = [
      { title: 'Producto',    data: 'producto_txt' },
      { title: 'Lote',        data: 'lote' },
      { title: 'F. Prod.',    data: 'fecha_produccion' },
      { title: 'F. Venc.',    data: 'fecha_vencimiento' },

      { title: 'UV (esp)',    data: 'uv_esp', className: 'text-end', render: (v)=> fmt(v) },
      { title: 'UV (rec)',    data: 'uv',     className: 'text-end', render: (v)=> fmt(v) },
      { title: 'Δ UV',        data: 'uv_diff', className: 'text-end', render: (v)=> renderDiffCell(v) },

      { title: 'UC (esp)',    data: 'uc_esp', className: 'text-end', render: (v)=> fmt(v) },
      { title: 'UC (rec)',    data: 'uc',     className: 'text-end', render: (v)=> fmt(v) },
      { title: 'Δ UC',        data: 'uc_diff', className: 'text-end', render: (v)=> renderDiffCell(v) },

      { title: 'Peso (kg)',   data: 'peso', className: 'text-end', render: (v)=> v!=null ? fmt(v) : '' },
      { title: 'Ubicación',   data: 'ubicacion' },
      { title: 'Estado',      data: 'estado', render: (v, t, row) => row.type === 'pending' ? 'PENDIENTE' : (v || 'RECIBIDO') },
      { title: 'Acciones',    data: null, orderable:false, searchable:false,
        render: (row) => {
          if (row.type === 'pending') {
            const id   = row.pl_item_id || '';
            const lote = row.lote || '';
            const uv   = row.uv_esp || 0;
            const uc   = row.uc_esp || 0;
            return `
              <div class="btn-group btn-group-sm">
                <button type="button"
                        class="btn btn-primary js-editar-item"
                        data-pl-item-id="${id}"
                        data-lote="${String(lote).replace(/"/g,'&quot;')}"
                        data-uv="${uv}"
                        data-uc="${uc}"
                        data-producto="${String(row.producto_txt||'').replace(/"/g,'&quot;')}">
                  Editar
                </button>
                <button type="button"
                        class="btn btn-success js-recibir-item"
                        data-pl-item-id="${id}"
                        data-lote="${String(lote).replace(/"/g,'&quot;')}"
                        data-uv="${uv}"
                        data-uc="${uc}">
                  Recibir ahora
                </button>
              </div>`;
          }
          return ''; // sin botón para recibidos
        }
      },
    ];

    // Ensure footer exists before initializing DataTable so DataTables can use it
    if ($t.find('tfoot').length === 0) {
      const tf = document.createElement('tfoot');
      const tr = document.createElement('tr');
      for (let i = 0; i < cols.length; i++) tr.appendChild(document.createElement('th'));
      tf.appendChild(tr);
      $t.append(tf);
    }

    // Compute uv_diff / uc_diff for each merged row (ensure numbers)
    merged.forEach(r => {
      const euv = r.uv_esp != null ? Number(r.uv_esp) : null;
      const eucc = r.uc_esp != null ? Number(r.uc_esp) : null;
      const ruv = r.uv != null ? Number(r.uv) : null;
      const ruc = r.uc != null ? Number(r.uc) : null;
      r.uv_diff = euv != null ? ((ruv != null ? (ruv - euv) : (0 - euv))) : null;
      r.uc_diff = eucc != null ? ((ruc != null ? (ruc - eucc) : (0 - eucc))) : null;
    });

    dt = $t.DataTable({
      destroy: true,
      data: merged,
      columns: cols,
      deferRender: true,
      processing: false,
      paging: true,
      pageLength: 25,
      lengthChange: false,
      order: [],
      searching: true,
      autoWidth: false,
      scrollX: true,
      scrollCollapse: true,
      responsive: false,
    language: {
      emptyTable: 'No hay datos disponibles en la tabla',
      info: 'Mostrando _START_ a _END_ de _TOTAL_ entradas',
      infoEmpty: 'Mostrando 0 a 0 de 0 entradas',
      lengthMenu: 'Mostrar _MENU_ entradas',
      loadingRecords: 'Cargando…',
      zeroRecords: 'No se encontraron registros coincidentes',
      search: 'Buscar:',
      paginate: { first: 'Primera', previous: 'Anterior', next: 'Siguiente', last: 'Última' },
      aria: { sortAscending: ': activar para ordenar la columna ascendente', sortDescending: ': activar para ordenar la columna descendente' }
    },
      footerCallback: function () {
        const api = this.api();
        const totals = {
          uv_esp: sumColumn(api, 4),
          uv:     sumColumn(api, 5),
          uv_diff: sumColumn(api, 6),
          uc_esp: sumColumn(api, 7),
          uc:     sumColumn(api, 8),
          uc_diff: sumColumn(api, 9),
        };

        // Note: summary cards for UV/UC recibidas and discrepancias are provided by the backend
        // via loadResumen(plId) and should remain authoritative. We do not overwrite them here.

        const footerCells = $t.find('tfoot tr th');
        if (footerCells && footerCells.length) {
          $(footerCells[0]).text('Totales:');
          $(footerCells[4]).html(fmt(totals.uv_esp));
          $(footerCells[5]).html(fmt(totals.uv));
          $(footerCells[6]).html(fmt(totals.uv_diff));
          $(footerCells[7]).html(fmt(totals.uc_esp));
          $(footerCells[8]).html(fmt(totals.uc));
          $(footerCells[9]).html(fmt(totals.uc_diff));
        }
      }
    });
  // Hide the footer row (we prefer summary cards as the source of truth)
  try { $t.find('tfoot').hide(); } catch (e) { /* ignore */ }
    // Force horizontal scrolling on wrapper and prevent table cells wrapping
    try {
      const wrapper = $t.closest('.dataTables_wrapper');
      if (wrapper && wrapper.length) wrapper.css('overflow-x', 'auto');
      $t.css('white-space', 'nowrap');
    } catch (e) { /* ignore */ }
  }

  // ---------- Acción: Abrir editor ----------
  $(document).on('click', '.js-editar-item', function () {
    const $btn = $(this);
    const row = {
      type: 'pending',
      pl_item_id: Number($btn.data('pl-item-id') || 0),
      lote: String($btn.data('lote') || ''),
      uv: Number($btn.data('uv') || 0),
      uc: Number($btn.data('uc') || 0),
      producto_txt: String($btn.data('producto') || ''),
    };
    if (!row.pl_item_id) { toast('Ítem inválido.', 'warning'); return; }
    openEditorModal(row);
  });

  // ---------- Acción: Confirmar editor ----------
  $(document).on('click', '#rec-editor-submit', async function () {
    const $m = $('#rec-editor-modal');
    const $f = $('#rec-editor-form');
    const row = $m.data('row') || {};
    const plItemId = Number($f.find('[name=pl_item_id]').val() || 0);
    if (!currentPL.id || !plItemId) return;

    if (!ensureMetaComplete()) return;

    const uvBuenas = Math.max(0, Number($f.find('[name=uv_buenas]').val() || 0));
    const ucBuenas = Math.max(0, Number($f.find('[name=uc_buenas]').val() || 0));
    const uvDanadas= Math.max(0, Number($f.find('[name=uv_danadas]').val() || 0));
    const ucDanadas= Math.max(0, Number($f.find('[name=uc_danadas]').val() || 0));
    const motivo   = String($f.find('[name=motivo]').val() || '');

    // Posición manual si existe, si no la default
    let posCodeOk = '';
    const $pos = $posInput();
    if ($pos.length && $pos.val()) posCodeOk = String($pos.val()).trim();
    else posCodeOk = CUARENTENA_POS_FULL; // si no proveen pos, mandamos igual a CUARENTENA (negocio actual)

    // Construimos el plan con 1 o 2 pallets según haya dañadas
    const rows = [];
    if (uvBuenas > 0 || ucBuenas > 0) {
      rows.push({
        pallet_codigo: null,
        pos_code: posCodeOk,
        items: [{ pl_item_id: plItemId, lote_codigo: String(row.lote||''), uv: uvBuenas, uc: ucBuenas }]
      });
    }
    if (uvDanadas > 0 || ucDanadas > 0) {
      rows.push({
        pallet_codigo: null,
        pos_code: CUARENTENA_POS_FULL,
        meta_estado: 'DANADO',
        items: [{ pl_item_id: plItemId, lote_codigo: String(row.lote||''), uv: uvDanadas, uc: ucDanadas }]
      });
    }

    if (rows.length === 0) {
      toast('No hay cantidades a recibir.', 'warning');
      return;
    }

    const payload = {
      pl_id: currentPL.id,
      deposito_code: DEFAULT_DEPOSITO,
      pos_code: rows[0].pos_code, // coherencia con contrato
      auto_ubicar: false,
      rows
    };
    // Añadir metadatos si el usuario los completó
    try { Object.assign(payload, collectMeta()); } catch (e) { /* ignore */ }

    const $btn = $('#rec-editor-submit');
    const originalHtml = $btn.html();
    $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1"></span>Guardando…');

    try {
      const res = await postJSON(endpoints.recepGuardar(), payload);
      if (!res || res.ok === false) {
        const raw = res ? (res.error || res.debug || res.raw || '') : '';
        // Mensaje específico si el SP devolvió excedente del plan
        const specific = /excede|excede.*pendientes|excede.*cantidad|excede.*items|excede|exced/i.test(String(raw));
        const msg = specific
          ? 'El plan excede las cantidades pendientes para uno o más ítems. Ajustá UV/UC e intentá de nuevo.'
          : (res ? String(raw || 'Fallo al recibir').slice(0, 400) : 'Fallo al recibir');
        toast(msg, 'danger');
        return; // no lanzar excepción para evitar doble alerta
      }

      // Si hubo “dañadas”, registramos nota en memoria para que la tabla muestre “DAÑADO (Motivo)”
      if (motivo && res?.data?.pallet_codes?.mapping) {
        // mapping: [{pallet_id, from, to}]
        const map = res.data.pallet_codes.mapping || [];
        // no sabemos cuál corresponde a dañadas; heurística: si hubo 2 rows, el último fue CUARENTENA ⇒ dañadas
        if (rows.length === 2 && map.length >= 2) {
          const damagedCode = String(map[map.length - 1].to || map[map.length - 1].from || '').trim();
          if (damagedCode) {
            ReceivedNotes.set(damagedCode, { tipo: 'DANADO', motivo });
          }
        }
      }

      // Mostrar warnings del backend si existen
      try {
        const warns = res.warnings || res.data?.warnings || [];
        (Array.isArray(warns) ? warns : []).forEach(w => toast(String(w), 'warning'));
      } catch (e) { /* ignore */ }
      toast('Recepción guardada.', 'success');
      bootstrap.Modal.getOrCreateInstance($m[0]).hide();
      if (currentPL.id) {
        await loadResumen(currentPL.id);
        await loadRecepcionList(currentPL.id);
      }
    } catch (e) {
      toast(`Error: ${e.message}`, 'danger');
    } finally {
      $btn.prop('disabled', false).html(originalHtml);
    }
  });

  // ---------- Acción: Recibir ahora (todo como “buenas”) ----------
  $(document).on('click', '.js-recibir-item', async function () {
    const $btn = $(this);
    const plItemId = Number($btn.data('pl-item-id') || 0);
    if (!currentPL.id || !plItemId) return;

    if (!ensureMetaComplete()) return;

    const lote = String($btn.data('lote') || '').trim();
    const uv   = Number($btn.data('uv') || 0);
    const uc   = Number($btn.data('uc') || 0);

    // Posición manual si existe, si no CUARENTENA por default
    let posCode = '';
    const $pos = $posInput();
    if ($pos.length && $pos.val()) {
      posCode = String($pos.val()).trim();
    } else {
      posCode = CUARENTENA_POS_FULL;
    }

    // Usar endpoint recepcion_guardar_item que autogenera pallets según producto
    const payload = {
      pl_id: currentPL.id,
      pl_item_id: plItemId,
      pos_code: posCode,
      auto_ubicar: !posCode || posCode === CUARENTENA_POS_FULL
    };
    try { Object.assign(payload, collectMeta()); } catch (e) { /* ignore */ }

    const originalHtml = $btn.html();
    $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1"></span>Recibiendo…');

    try {
      const res = await postJSON(endpoints.recepGuardarItem(), payload);
      if (!res || res.ok === false) {
        const msg = res
          ? String(res.error || res.debug || res.raw || 'Fallo al recibir item').slice(0, 300)
          : 'Fallo al recibir item';
        throw new Error(msg);
      }
      const palletCount = res.plan_count || 0;
      toast(`Item recibido: ${palletCount} pallet(s) creado(s) automáticamente.`, 'success');
      if (currentPL.id) {
        await loadResumen(currentPL.id);
        await loadRecepcionList(currentPL.id);
      }
    } catch (e) {
      toast(`Error al recibir item: ${e.message}`, 'danger');
    } finally {
      $btn.prop('disabled', false).html(originalHtml);
    }
  });

  // ---------- Botones ----------
  $btnCargar().on('click', async function () {
    const $btn = $btnCargar();
    if ($btn.prop('disabled')) return;
    const originalHtml = $btn.html();
    const plId = Number($selPL().val() || 0);
    if (!plId) { toast('Seleccione un PL.', 'warning'); return; }
    try {
      $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span>Cargando…');
      await loadResumen(plId);
      await loadRecepcionList(plId);
      try { localStorage.setItem('rec_last_pl', String(plId)); } catch(e) { /* ignore */ }
    } catch (e) {
      toast(e.message || 'Error al cargar PL', 'danger');
    } finally {
      $btn.prop('disabled', false).html(originalHtml);
    }
  });

  $btnRefresh().on('click', async function () {
    const selected = $selPL().val() || '';
    await loadPendientes();
    if (selected) { $selPL().val(selected); }
    if (currentPL.id) {
      await loadResumen(currentPL.id);
      await loadRecepcionList(currentPL.id);
    }
  });

  $btnPrint().on('click', function () {
    window.print();
  });

  $(document).on('click', '#btnFechaIngresoHoy', function () {
    const $input = $('#meta_fecha_ingreso');
    if (!$input.length) return;
    $input.val(getTodayDateISO()).trigger('change');
    if ($input.hasClass('is-invalid')) $input.removeClass('is-invalid');
  });

  $(document).on('click', '.js-set-time-now', function () {
    const targetId = $(this).data('target');
    if (!targetId) return;
    const $input = $('#' + targetId);
    if (!$input.length) return;
    $input.val(getCurrentTimeHM()).trigger('change');
    if ($input.hasClass('is-invalid')) $input.removeClass('is-invalid');
  });

  $(document).on('click', '#btnImprimirRecepcion', function () {
    if (!currentPL.id) {
      toast('Cargá un PL antes de imprimir la recepción.', 'warning');
      return;
    }
    const url = `/api/operaciones/recepcion_pdf.php?pl_id=${encodeURIComponent(currentPL.id)}`;
    window.open(url, '_blank', 'noopener');
  });

  // Recalculate totals button
  $btnRecalc().on('click', async function () {
    if (!currentPL.id) { toast('No hay PL cargado.', 'warning'); return; }
    const $btn = $btnRecalc();
    if ($btn.prop('disabled')) return; // already running
    const originalHtml = $btn.html();
    try {
      $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span>Recalculando…');
      await loadResumen(currentPL.id);
      toast('Totales recalculados.', 'success');
    } catch (e) {
      toast(`Error recalculando totales: ${e.message}`, 'danger');
    } finally {
      $btn.prop('disabled', false).html(originalHtml);
    }
  });

  // ---------- Acción: Guardar cabecera (persistir metadatos sin ejecutar SP) ----------
  $(document).on('click', '#btnGuardarCabecera', async function () {
    if (!currentPL.id) { toast('Seleccione un PL y cárguelo primero.', 'warning'); return; }
    if (!ensureMetaComplete()) return;
    const payload = { pl_id: currentPL.id };
    try { Object.assign(payload, collectMeta()); } catch (e) { /* ignore */ }

    const $btn = $('#btnGuardarCabecera');
    const orig = $btn.html();
    $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1"></span>Guardando…');
    try {
      const res = await postJSON(API_BASE + '?meta=recepcion_guardar_hdr', payload);
      if (!res || res.ok === false) {
        const msg = res ? String(res.error || res.debug || res.raw || 'Fallo al guardar cabecera').slice(0,300) : 'Fallo al guardar cabecera';
        throw new Error(msg);
      }
      toast('Cabecera guardada.', 'success');
    // Flip badge to saved immediately
    try { setHdrBadge(true); } catch(e) { /* ignore */ }
  // No persistimos flag; panel físico siempre visible por ahora
      try { localStorage.setItem('rec_last_pl', String(currentPL.id)); } catch(e) { /* ignore */ }
      // Refresh resumen and list to reflect persisted metadata
      if (currentPL.id) {
        await loadResumen(currentPL.id);
        await loadRecepcionList(currentPL.id);
      }
    } catch (e) {
      toast(`Error guardando cabecera: ${e.message}`, 'danger');
    } finally {
      $btn.prop('disabled', false).html(orig);
    }
  });

  // ---------- Acción: Guardar recepción (recibir TODOS los items pendientes con autogeneración de pallets) ----------
  $btnGuardar().on('click', async function () {
    if (!currentPL.id) { toast('Seleccione un PL y cárguelo primero.', 'warning'); return; }
    if (!dt) { toast('No hay datos cargados.', 'warning'); return; }

    if (!ensureMetaComplete()) return;

    // Recolectar todas las filas (DataTables API)
    const dataApi = dt.rows().data();
    const all = [];
    for (let i = 0; i < dataApi.length; i++) all.push(dataApi[i]);

    const pendings = all.filter(r => r && r.type === 'pending');
    if (!pendings || pendings.length === 0) { toast('No hay ítems pendientes para recibir.', 'info'); return; }

    // Confirmación si son muchos ítems
    if (pendings.length > 25) {
      if (!confirm(`Se van a recibir ${pendings.length} ítems (con autogeneración de pallets). ¿Desea continuar?`)) return;
    }

    const $pos = $posInput();
    const manualPos = ($pos.length && $pos.val()) ? String($pos.val()).trim() : '';
    const defaultPos = manualPos || CUARENTENA_POS_FULL;
    const autoUbicar = !manualPos || manualPos === CUARENTENA_POS_FULL;
    
    // Recoger metadatos una vez para todos los ítems
    let metaPayload = {};
    try { metaPayload = collectMeta(); } catch (e) { /* ignore */ }

    const $btn = $btnGuardar();
    const originalHtml = $btn.html();
    $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1"></span>Guardando…');

    let totalPallets = 0;
    let successCount = 0;
    let errorCount = 0;
    const errors = [];

    try {
      // Procesar cada ítem pendiente usando el endpoint de autogeneración
      for (let i = 0; i < pendings.length; i++) {
        const r = pendings[i];
        const plItemId = r.pl_item_id || r.id || null;
        if (!plItemId) continue;

        const uv = (r.uv != null ? Number(r.uv) : (r.uv_esp != null ? Number(r.uv_esp) : 0)) || 0;
        const uc = (r.uc != null ? Number(r.uc) : (r.uc_esp != null ? Number(r.uc_esp) : 0)) || 0;
        if (uv === 0 && uc === 0) continue; // saltar si no hay cantidades

        const payload = {
          pl_id: currentPL.id,
          pl_item_id: plItemId,
          pos_code: defaultPos,
          auto_ubicar: autoUbicar,
          ...metaPayload
        };

        // Actualizar progreso en el botón
        $btn.html(`<span class="spinner-border spinner-border-sm me-1"></span>Recibiendo ${i + 1}/${pendings.length}…`);

        try {
          const res = await postJSON(endpoints.recepGuardarItem(), payload);
          if (!res || res.ok === false) {
            const msg = res ? String(res.error || res.debug || res.raw || 'Error').slice(0, 200) : 'Error';
            errors.push(`Item ${plItemId}: ${msg}`);
            errorCount++;
          } else {
            const palletCount = res.plan_count || 0;
            totalPallets += palletCount;
            successCount++;
          }
        } catch (e) {
          errors.push(`Item ${plItemId}: ${e.message}`);
          errorCount++;
        }
      }

      // Mostrar resultados
      if (errorCount > 0) {
        const errorMsg = errors.slice(0, 5).join('\n') + (errors.length > 5 ? `\n... y ${errors.length - 5} más` : '');
        toast(`Recepción completada con errores.\nÉxitos: ${successCount}, Errores: ${errorCount}\n${errorMsg}`, 'warning');
      } else {
        toast(`Recepción masiva completada: ${successCount} ítems recibidos, ${totalPallets} pallets creados automáticamente.`, 'success');
      }

      // Recargar datos
      if (currentPL.id) {
        await loadResumen(currentPL.id);
        await loadRecepcionList(currentPL.id);
      }
    } catch (e) {
      toast(`Error al guardar recepción: ${e.message}`, 'danger');
    } finally {
      $btn.prop('disabled', false).html(originalHtml);
    }
  });

  // ---------- Init ----------
  // Cargar catálogos por AJAX para poblar selects de móvil y chofer
  function fetchSingleMovil(id) {
    return $.getJSON('/api/parametros/moviles_select.php?id=' + encodeURIComponent(id)).then(function(r){
      return (r && r.data && r.data[0]) ? r.data[0] : null;
    }).catch(function(){ return null; });
  }

  function fetchSingleChofer(id) {
    return $.getJSON('/api/parametros/choferes.php?id=' + encodeURIComponent(id)).then(function(r){
      return (r && r.data && r.data[0]) ? r.data[0] : null;
    }).catch(function(){ return null; });
  }

  function loadCatalogsForRecepcion() {
    // For testing: disable Select2 and always use native <select> population.
    // This avoids Select2 timing/display issues while debugging selection behavior.
    try {
      try {
        if ($.fn.select2) {
          try { if ($('#meta_movil_id').data('select2')) $('#meta_movil_id').select2('destroy'); } catch(e) {}
          try { if ($('#meta_chofer_id').data('select2')) $('#meta_chofer_id').select2('destroy'); } catch(e) {}
        }
      } catch(e) { /* ignore */ }

      // Populate móvil & chofer selects with a small set of native options for fast page load.
      // Additionally initialize Select2 lazily on focus to enable remote searching without blocking initial render.
      try {
        // small initial load (first N items) so the user sees some options immediately
        $.getJSON('/api/parametros/moviles_select.php').done(function(res){
          const $mov = $('#meta_movil_id');
          $mov.empty().append($('<option>',{value:'',text:'--'}));
          if (res && Array.isArray(res.data)) {
            res.data.forEach(function(it){ $mov.append($('<option>',{value:it.id,text:it.chapa||it.id})); });
          }
          try { const pre = $mov.data('preselect'); if (pre) { $mov.val(String(pre)); } } catch(e) { /* ignore */ }
        }).fail(function(){ /* ignore */ });

        $.getJSON('/api/parametros/choferes.php').done(function(res){
          const $cho = $('#meta_chofer_id');
          $cho.empty().append($('<option>',{value:'',text:'--'}));
          if (res && Array.isArray(res.data)) {
            res.data.forEach(function(it){ $cho.append($('<option>',{value:it.id,text:it.nombre||it.id})); });
          }
          try { const pre = $cho.data('preselect'); if (pre) { $cho.val(String(pre)); } } catch(e) { /* ignore */ }
        }).fail(function(){ /* ignore */ });

        // Lazy Select2 initializer: called on focus to avoid loading select2 and remote requests on page load.
        function initLazySelect2($el, url, mapFn, minChars) {
          minChars = (minChars == null) ? 2 : minChars;
          // Initialize Select2 on first click to avoid loading Select2 on page load
          $el.one('click.lazySelect2', function(){
            try {
              if (!($.fn && $.fn.select2)) return; // no select2 available
              // Destroy any existing remnants
              try { $el.select2('destroy'); } catch(e) {}
              $el.select2({
                width: '100%',
                placeholder: $el.attr('placeholder') || 'Buscar...',
                ajax: {
                  url: url,
                  dataType: 'json',
                  delay: 300,
                  data: function(params) { return { q: params.term, page: params.page || 1 }; },
                  processResults: function(data, params) {
                    params.page = params.page || 1;
                    const rows = data && data.data ? data.data : [];
                    return {
                      results: rows.map(mapFn),
                      pagination: { more: (rows.length >= 20) }
                    };
                  },
                  cache: true
                },
                minimumInputLength: minChars,
                allowClear: true
              });
            } catch(e) { /* ignore */ }
          });
        }

        // Initialize lazy select2 on focus for both selects
        initLazySelect2($('#meta_movil_id'), '/api/parametros/moviles_select.php', function(r){ return { id: r.id, text: r.chapa || r.id }; }, 2);
        initLazySelect2($('#meta_chofer_id'), '/api/parametros/choferes.php', function(r){ return { id: r.id, text: r.nombre || r.id }; }, 2);

      } catch(e) { /* ignore */ }

    } catch (e) { /* ignore */ }
  }

  $(async function () {
    // Defensive: remove any lingering Select2 UI containers and destroy instances so native selects are visible.
    try {
      // Try to destroy Select2 instances unconditionally (some versions don't set data('select2'))
      ['#meta_movil_id', '#meta_chofer_id', '#meta_tipo_documento', '#selPLPendiente'].forEach(function(sel){
        try { $(sel).select2('destroy'); } catch(e) { /* ignore if not initialized */ }
        try { $(sel).removeClass('select2-hidden-accessible'); } catch(e) {}
        try { $(sel).removeAttr('data-select2-id aria-hidden tabindex'); } catch(e) {}
        try { $(sel).attr('aria-hidden', null); } catch(e) {}
        try { $(sel).nextAll('.select2, .select2-container, .select2-container--default').remove(); } catch(e) {}
        try { $(sel).show(); } catch(e) {}
      });
      // Also remove any global Select2 containers left behind (for debugging/testing it's ok)
      try { $('.select2, .select2-container, .select2-container--default').remove(); } catch(e) {}
      // Ensure the PL select is enabled and has a placeholder option so user can interact
      try {
        const $pl = $('#selPLPendiente');
        if ($pl && $pl.length) {
          if ($pl.find('option').length === 0) $pl.append($('<option>',{value:'',text:'Seleccione un PL...'}));
          $pl.prop('disabled', false).show();
          $pl.closest('.col-12, .col-12.col-md-6').show();
        }
      } catch(e) { /* ignore */ }
    } catch(e) { /* ignore */ }
    // Try to restore last opened PL (stored in localStorage) and auto-load it
    const LAST_PL_KEY = 'rec_last_pl';
    const lastPl = (function(){ try { return localStorage.getItem(LAST_PL_KEY); } catch(e){ return null; } })();
    await loadPendientes();
    try { loadCatalogsForRecepcion(); } catch (e) { console.warn('Error cargando catálogos:', e); }

    // limpiar resaltado de validación al editar campos de metadatos
    META_REQUIRED_FIELDS.forEach(field => {
      const $el = $(field.selector);
      if (!$el.length) return;
      $el.on('input change blur', function () {
        if ($(this).hasClass('is-invalid')) $(this).removeClass('is-invalid');
        if (field.selector === '#meta_tipo_documento') {
          const val = $(this).val();
          if (val) $(this).val(String(val).toUpperCase());
        }
      });
    });

    // If we have a previously opened PL, and it exists in the select, auto-load resumen + list
    try {
      if (lastPl) {
        const $sel = $selPL();
        if ($sel.find(`option[value="${lastPl}"]`).length) {
          $sel.val(String(lastPl));
          // emulate clicking CargarPL: set currentPL and load
          await loadResumen(Number(lastPl));
          await loadRecepcionList(Number(lastPl));
        } else {
          // if option not present, remove stale storage
          try { localStorage.removeItem(LAST_PL_KEY); } catch(e) { /* ignore */ }
        }
      }
    } catch(e) { console.warn('Error restaurando PL previo:', e); }
  });
})();
