// DataTables: Inventario → Existencias (resumen por producto)
$(document).ready(function () {
  const tableId = "existenciasTable";
  const ajaxUrl = "/api/inventario/existencias.php";

  const numberFormatter = new Intl.NumberFormat("es-AR", {
    minimumFractionDigits: 0,
    maximumFractionDigits: 0,
  });

  const toNumber = (value) => {
    const numeric = Number(value);
    return Number.isFinite(numeric) ? numeric : 0;
  };

  const formatNumber = (value) => numberFormatter.format(toNumber(value));

  const renderNumber = (data) => formatNumber(data);

  const defaultTotals = {
    productos: 0,
    clientes: 0,
    operativas: 0,
    pallets: 0,
    cajas: 0,
    unidades_sueltas: 0,
    unidades: 0,
    stock: 0,
    reservados: 0,
    disponibles: 0,
  };

  const cardElements = {
    productos: $("[data-metric='productos']"),
    pallets: $("[data-metric='pallets']"),
    cajas: $("[data-metric='cajas']"),
    unidades_sueltas: $("[data-metric='unidades_sueltas']"),
    stock: $("[data-metric='stock']"),
    reservados: $("[data-metric='reservados']"),
    disponibles: $("[data-metric='disponibles']"),
  };

  const ensureMetricText = (filteredValue, grandValue) => {
    if (
      grandValue !== null &&
      grandValue !== undefined &&
      grandValue !== filteredValue
    ) {
      return `${formatNumber(filteredValue)} / ${formatNumber(grandValue)}`;
    }

    return formatNumber(filteredValue);
  };

  let lastGrandTotals = { ...defaultTotals };

  const columns = [
    {
      data: null,
      defaultContent: "",
      className: "select-checkbox",
      orderable: false,
      searchable: false,
      title: "",
      width: "20px",
    },
    { data: "cliente", title: "Cliente" },
    { data: "operativa", title: "Operativa" },
    { data: "sku", title: "SKU" },
    { data: "denominacion", title: "Producto" },
    {
      data: "pallets",
      title: "Pallets",
      className: "text-end",
      render: renderNumber,
    },
    {
      data: "cajas",
      title: "Cajas",
      className: "text-end",
      render: renderNumber,
    },
    {
      data: "unidades_sueltas",
      title: "Unidades sueltas (UC)",
      className: "text-end",
      render: renderNumber,
    },
    {
      data: "stock",
      title: "Stock",
      className: "text-end",
      render: renderNumber,
    },
    {
      data: "reservados",
      title: "Reservados",
      className: "text-end text-warning",
      render: renderNumber,
    },
    {
      data: "disponibles",
      title: "Disponibles",
      className: "text-end text-success fw-semibold",
      render: renderNumber,
    },
  ];

  const $table = $("#" + tableId);
  $table.find("tfoot").remove();

  if ($table.find("thead").length === 0 || $table.find("thead tr").length === 0) {
    let thead = "<thead><tr>";
    columns.forEach((col) => {
      thead += `<th>${col.title ? col.title : ""}</th>`;
    });
    thead += "</tr></thead>";
    $table.prepend(thead);
  }

  const updateTotals = (totals, grandTotals = null) => {
    const safeTotals = { ...defaultTotals, ...(totals || {}) };
    const safeGrandTotals = grandTotals
      ? { ...defaultTotals, ...grandTotals }
      : null;

    Object.entries(cardElements).forEach(([key, $element]) => {
      if (!$element.length) {
        return;
      }

      const filteredValue = safeTotals[key] ?? 0;
      const grandValue = safeGrandTotals ? safeGrandTotals[key] ?? 0 : null;
      $element.text(ensureMetricText(filteredValue, grandValue));
    });
  };

  updateTotals(defaultTotals, defaultTotals);

  const computeFilteredTotals = (api) => {
    const totals = { ...defaultTotals };
    const clientes = new Set();
    const operativas = new Set();

    const filteredRows = api.rows({ search: "applied" });
    totals.productos = filteredRows.count();

    filteredRows.data().each((row) => {
      if (!row) {
        return;
      }

      const cliente = typeof row.cliente === "string" ? row.cliente.trim() : String(row.cliente ?? "").trim();
      if (cliente) {
        clientes.add(cliente);
      }

      const operativa = typeof row.operativa === "string" ? row.operativa.trim() : String(row.operativa ?? "").trim();
      if (operativa) {
        operativas.add(operativa);
      }

      totals.pallets += toNumber(row.pallets);
      totals.cajas += toNumber(row.cajas);
      totals.unidades_sueltas += toNumber(
        row.unidades_sueltas ?? row.uc ?? row.uc_sueltas ?? row.unidades
      );

      const stock = toNumber(row.stock ?? row.unidades);
      const unidades = toNumber(row.unidades ?? stock);
      const reservados = toNumber(row.reservados);
      const disponiblesSource =
        row.disponibles !== undefined
          ? row.disponibles
          : stock - reservados;

      totals.stock += stock;
      totals.unidades += unidades;
      totals.reservados += reservados;
      totals.disponibles += toNumber(disponiblesSource);
    });

    totals.clientes = clientes.size;
    totals.operativas = operativas.size;

    return totals;
  };

  const dtOptions = {
    ajax: {
      url: ajaxUrl,
      dataSrc: function (json) {
        if (json && json.ok !== false) {
          const grand = json.grandTotals || json.totals || null;
          lastGrandTotals = { ...defaultTotals, ...(grand || {}) };
          if (json.totals) {
            updateTotals(json.totals, lastGrandTotals);
          }
          return Array.isArray(json.rows) ? json.rows : [];
        }

        lastGrandTotals = { ...defaultTotals };
        updateTotals(null, lastGrandTotals);
        if (json && json.error) {
          // eslint-disable-next-line no-console
          console.error("[Existencias]", json.error, json.detail || "");
        }
        return [];
      },
    },
    select: { style: "os", selector: "td.select-checkbox" },
    searching: true,
    ordering: true,
  };

  const dtResult = initializeBaseDataTable({
    tableId,
    ajaxUrl,
    columns,
    editorFields: false,
    order: [
      [1, "asc"],
      [2, "asc"],
      [3, "asc"],
    ],
    dataTableOptions: dtOptions,
    tableButtons: [],
    editorButtons: [],
  });

  if (dtResult && dtResult.table) {
    dtResult.table.on("xhr.dt", function (_event, _settings, json) {
      if (!json || json.ok === false) {
        updateTotals(null, lastGrandTotals);
        return;
      }

      const grand = json.grandTotals || json.totals || null;
      lastGrandTotals = { ...defaultTotals, ...(grand || {}) };
      const totals = json.totals || grand || defaultTotals;
      updateTotals(totals, lastGrandTotals);
    });

    dtResult.table.on("draw.dt", function () {
      const api = dtResult.table;
      const filteredTotals = computeFilteredTotals(api);
      updateTotals(filteredTotals, lastGrandTotals);
    });
  }
});
