/* public/page-scripts/pages/inventario/min_max_scripts.js */
/* global $, Intl */

$(function () {
  const tableId = "#minMaxTable";
  const $form = $("#frmFiltrosMinMax");

  // --- Formato numérico (PY) ---
  const fmtInt = (v) => {
    const n = Number(v ?? 0);
    try {
      return new Intl.NumberFormat("es-PY").format(n);
    } catch {
      return (n || 0).toString();
    }
  };

  // --- Badge de estado ---
  function renderEstado(val) {
    const t = String(val || "").toUpperCase();
    let cls = "badge bg-secondary";
    if (t === "BAJO") cls = "badge bg-danger";
    else if (t === "OK") cls = "badge bg-success";
    else if (t === "SOBRE") cls = "badge bg-warning text-dark";
    return `<span class="${cls}">${t || "-"}</span>`;
  }

  function initSelect2() {
    if (!$.fn.select2) return; // por si falla el CDN
    const opts = {
      width: "100%",
      allowClear: true,
      placeholder: "(Todos)",
      dropdownParent: $("#frmFiltrosMinMax"),
    };
    $("#fCliente").select2(opts);
    $("#fOperativa").select2(opts);
  }

  // --- Cargar combos (clientes / operativas) ---
  function cargarCombos() {
    // Clientes
    $.getJSON("/api/inventario/min_max.php?meta=clientes").done((res) => {
      const $c = $("#fCliente");
      if (res && Array.isArray(res.clientes)) {
        res.clientes.forEach((r) =>
          $c.append(new Option(`${r.razon_social} (ID ${r.id})`, r.id))
        );
        $c.trigger("change.select2");
      }
    });

    // Operativas
    $.getJSON("/api/inventario/min_max.php?meta=operativas").done((res) => {
      const $o = $("#fOperativa");
      if (res && Array.isArray(res.operativas)) {
        res.operativas.forEach((r) =>
          $o.append(new Option(`${r.nombre} (ID ${r.id})`, r.id))
        );
        $o.trigger("change.select2");
      }
    });
  }

  function actualizarResumen(rows) {
    let bajo = 0,
      ok = 0,
      sobre = 0;
    (rows || []).forEach((r) => {
      const s = String(r.estado || "").toUpperCase();
      if (s === "BAJO") bajo++;
      else if (s === "SOBRE") sobre++;
      else ok++;
    });
    $("#badgeBajo").text(`Bajo mínimo: ${bajo}`);
    $("#badgeOk").text(`En rango: ${ok}`);
    $("#badgeSobre").text(`Sobre máximo: ${sobre}`);
  }
  // --- DataTable (server-side) ---
  const dt = $(tableId).DataTable({
    processing: true,
    serverSide: true,
    searching: false,
    lengthChange: true,
    pageLength: 25,
    ajax: {
      url: "/api/inventario/min_max.php",
      type: "GET",
      data: function (d) {
        // filtros → backend
        const params = Object.fromEntries(new FormData($form[0]).entries());
        return Object.assign({}, d, params, { action: "list" });
      },
      dataSrc: function (json) {
        return json.data || [];
      },
    },
    columns: [
      { data: "cliente", title: "Cliente" },
      { data: "operativa", title: "Operativa" },
      { data: "sku", title: "SKU" },
      { data: "denominacion", title: "Producto" },

      {
        data: "minimo",
        title: "Mínimo",
        className: "text-end",
        render: (d) => fmtInt(d),
      },
      {
        data: "maximo",
        title: "Máximo",
        className: "text-end",
        render: (d) => fmtInt(d),
      },

      {
        data: "stock",
        title: "Stock",
        className: "text-end",
        render: (d) => fmtInt(d),
      },
      {
        data: "reservados",
        title: "Reservados",
        className: "text-end",
        render: (d) => fmtInt(d),
      },
      {
        data: "disponibles",
        title: "Disponibles",
        className: "text-end",
        render: (d) => fmtInt(d),
      },

      // Diferencias útiles
      {
        data: "faltante_min",
        title: "Faltante a Mín",
        className: "text-end",
        render: (d) => fmtInt(d),
      },
      {
        data: "exceso_max",
        title: "Exceso sobre Máx",
        className: "text-end",
        render: (d) => fmtInt(d),
      },

      { data: "estado", title: "Estado", render: (d) => renderEstado(d) },
    ],
    order: [
      [1, "asc"],
      [2, "asc"],
    ], // Operativa, SKU
    language: {
      url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json",
    },

    // Totales en footer (página actual)
    footerCallback: function () {
      const api = this.api();
      const sumCol = (idx) =>
        api
          .column(idx, { page: "current" })
          .data()
          .reduce((a, b) => (Number(a) || 0) + (Number(b) || 0), 0);

      // Índices de columnas numéricas:
      // 4:minimo 5:maximo 6:stock 7:reservados 8:disponibles 9:faltante_min 10:exceso_max
      const cols = {
        minimo: 4,
        maximo: 5,
        stock: 6,
        reserv: 7,
        disp: 8,
        falt: 9,
        exc: 10,
      };

      $(api.column(cols.minimo).footer()).html(fmtInt(sumCol(cols.minimo)));
      $(api.column(cols.maximo).footer()).html(fmtInt(sumCol(cols.maximo)));
      $(api.column(cols.stock).footer()).html(fmtInt(sumCol(cols.stock)));
      $(api.column(cols.reserv).footer()).html(fmtInt(sumCol(cols.reserv)));
      $(api.column(cols.disp).footer()).html(fmtInt(sumCol(cols.disp)));
      $(api.column(cols.falt).footer()).html(fmtInt(sumCol(cols.falt)));
      $(api.column(cols.exc).footer()).html(fmtInt(sumCol(cols.exc)));
    },
  });

  // Botón PDF: abrir con los filtros actuales + orden del DataTable
  $("#btnPdfMinMax").on("click", function () {
    const base = "/api/inventario/min_max_pdf.php";
    const formEl = document.getElementById("frmFiltrosMinMax");
    const data = formEl
      ? Object.fromEntries(new FormData(formEl).entries())
      : {};

    const qp = new URLSearchParams();

    // Solo agregamos filtros con valor
    Object.entries(data).forEach(([k, v]) => {
      if (v !== null && v !== "") qp.append(k, v);
    });

    // Incluir el orden actual del DataTable (ej: [[1,"asc"],[2,"desc"]])
    if (typeof dt?.order === "function") {
      const order = dt.order();
      if (order && order.length) {
        qp.append("dt_order", JSON.stringify(order));
      }
    }

    const href = qp.toString() ? `${base}?${qp.toString()}` : base;
    $(this).attr("href", href);
    // no prevenimos el default: <a target="_blank"> abrirá el PDF en nueva pestaña
  });

  // Al recibir datos del servidor, actualizar badges con la página actual
  $(tableId).on("xhr.dt", function (e, settings, json) {
    actualizarResumen(json && json.data ? json.data : []);
  });

  // Primera carga
  initSelect2();
  cargarCombos();

  // Aplicar filtros → recargar
  $form.on("submit", function (e) {
    e.preventDefault();
    dt.ajax.reload();
  });

  cargarCombos();
});
