// DataTables: Inventario → Movimientos de inventario
$(document).ready(function () {
  const tableId = "movimientosTable";
  const ajaxUrl = "/api/inventario/movimientos.php";

  const $filterDesde = $("#movimientosFilterDesde");
  const $filterHasta = $("#movimientosFilterHasta");
  const $applyFilters = $("#movimientosApplyFilters");
  const $resetFilters = $("#movimientosResetFilters");
  const $pdfLink = $("#movimientosPdfLink");

  const numberFormatter0 = new Intl.NumberFormat("es-AR", {
    minimumFractionDigits: 0,
    maximumFractionDigits: 0,
  });

  const numberFormatter2 = new Intl.NumberFormat("es-AR", {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2,
  });

  const toNumber = (value) => {
    if (value === null || value === undefined || value === "") {
      return 0;
    }
    const numeric = Number(value);
    return Number.isFinite(numeric) ? numeric : 0;
  };

  const formatUnits = (value) => numberFormatter0.format(toNumber(value));
  const formatFixed2 = (value) => numberFormatter2.format(toNumber(value));

  const formatNullableUnits = (value) => {
    if (value === null || value === undefined || value === "") {
      return "—";
    }
    return formatUnits(value);
  };

  const formatNullableFixed2 = (value) => {
    if (value === null || value === undefined || value === "") {
      return "—";
    }
    return formatFixed2(value);
  };

  const renderUnits = (data) => formatUnits(data);
  const renderDecimal = (data) => (data === null || data === undefined ? "—" : formatFixed2(data));

  const optionalTotalFields = [
    "saldo_inicial_pallets",
    "saldo_inicial_cajas",
    "saldo_inicial_unidades_sueltas",
    "ingresos_pallets",
    "ingresos_cajas",
    "ingresos_unidades_sueltas",
    "salidas_pallets",
    "salidas_cajas",
    "salidas_unidades_sueltas",
    "saldo_final_pallets",
    "saldo_final_cajas",
    "saldo_final_unidades_sueltas",
    "inventario_fisico",
    "inventario_fisico_pallets",
    "inventario_fisico_cajas",
    "inventario_fisico_unidades_sueltas",
    "diferencia_unidades",
    "valor_diferencia",
  ];

  const defaultTotals = {
    productos: 0,
    saldo_inicial: 0,
    saldo_inicial_pallets: null,
    saldo_inicial_cajas: null,
    saldo_inicial_unidades_sueltas: null,
    ingresos: 0,
    ingresos_pallets: null,
    ingresos_cajas: null,
    ingresos_unidades_sueltas: null,
    salidas: 0,
    salidas_pallets: null,
    salidas_cajas: null,
    salidas_unidades_sueltas: null,
    saldo_final: 0,
    saldo_final_pallets: null,
    saldo_final_cajas: null,
    saldo_final_unidades_sueltas: null,
    inventario_fisico: null,
    inventario_fisico_pallets: null,
    inventario_fisico_cajas: null,
    inventario_fisico_unidades_sueltas: null,
    diferencia_unidades: null,
    valor_unidades: 0,
    valor_diferencia: null,
  };

  const cardElements = {
    productos: $("[data-metric='productos']"),
    ingresos: $("[data-metric='ingresos']"),
    salidas: $("[data-metric='salidas']"),
    saldo_final: $("[data-metric='saldo_final']"),
  };

  const metricFormatters = {
    productos: formatUnits,
    ingresos: formatUnits,
    salidas: formatUnits,
    saldo_final: formatUnits,
  };

  const ensureMetricText = (filteredValue, grandValue, formatter) => {
    const format = formatter || formatUnits;

    if (filteredValue === null || filteredValue === undefined) {
      if (grandValue === null || grandValue === undefined) {
        return "—";
      }
      return `— / ${format(grandValue)}`;
    }

    const filteredText = format(filteredValue);
    if (grandValue === null || grandValue === undefined) {
      return filteredText;
    }

    const filteredNumber = toNumber(filteredValue);
    const grandNumber = toNumber(grandValue);
    if (filteredNumber !== grandNumber) {
      return `${filteredText} / ${format(grandValue)}`;
    }

    return filteredText;
  };

  const footerFormatters = {
    saldo_inicial_pallets: formatNullableUnits,
    saldo_inicial_cajas: formatNullableUnits,
    saldo_inicial_unidades_sueltas: formatNullableUnits,
    saldo_inicial: formatUnits,
    ingresos_pallets: formatNullableUnits,
    ingresos_cajas: formatNullableUnits,
    ingresos_unidades_sueltas: formatNullableUnits,
    ingresos: formatUnits,
    salidas_pallets: formatNullableUnits,
    salidas_cajas: formatNullableUnits,
    salidas_unidades_sueltas: formatNullableUnits,
    salidas: formatUnits,
    saldo_final_pallets: formatNullableUnits,
    saldo_final_cajas: formatNullableUnits,
    saldo_final_unidades_sueltas: formatNullableUnits,
    saldo_final: formatUnits,
    inventario_fisico_pallets: formatNullableUnits,
    inventario_fisico_cajas: formatNullableUnits,
    inventario_fisico_unidades_sueltas: formatNullableUnits,
    inventario_fisico: formatNullableUnits,
    diferencia_unidades: formatNullableUnits,
    valor_unidades: formatFixed2,
    valor_diferencia: formatNullableFixed2,
  };

  let lastGrandTotals = { ...defaultTotals };
  let lastFilters = {
    desde: ($filterDesde.val() || "").trim(),
    hasta: ($filterHasta.val() || "").trim(),
  };

  const buildPdfUrl = (filters) => {
    const params = new URLSearchParams();
    if (filters.desde) {
      params.set("desde", filters.desde);
    }
    if (filters.hasta) {
      params.set("hasta", filters.hasta);
    }

    const query = params.toString();
    return `/api/inventario/movimientos_pdf.php${query ? `?${query}` : ""}`;
  };

  const refreshPdfLink = () => {
    const url = buildPdfUrl(lastFilters);
    $pdfLink.attr("href", url);
  };

  const columns = [
    {
      data: null,
      defaultContent: "",
      className: "select-checkbox",
      orderable: false,
      searchable: false,
      title: "",
      width: "20px",
    },
    { data: "codigo", title: "Código" },
    { data: "descripcion", title: "Descripción" },
    {
      data: "cajas_por_pallet",
      title: "Cajas / pallet",
      className: "text-end",
      render: formatNullableFixed2,
    },
    {
      data: "unidades_por_caja",
      title: "Unid. / caja",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_inicial_pallets",
      title: "Saldo inicial · Pallets",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_inicial_cajas",
      title: "Saldo inicial · Cajas",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_inicial_unidades_sueltas",
      title: "Saldo inicial · Unid. sueltas",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_inicial",
      title: "Saldo inicial · Total unid.",
      className: "text-end fw-semibold",
      render: renderUnits,
    },
    {
      data: "ingresos_pallets",
      title: "Ingresos · Pallets",
      className: "text-end text-success",
      render: formatNullableUnits,
    },
    {
      data: "ingresos_cajas",
      title: "Ingresos · Cajas",
      className: "text-end text-success",
      render: formatNullableUnits,
    },
    {
      data: "ingresos_unidades_sueltas",
      title: "Ingresos · Unid. sueltas",
      className: "text-end text-success",
      render: formatNullableUnits,
    },
    {
      data: "ingresos",
      title: "Ingresos · Total unid.",
      className: "text-end text-success fw-semibold",
      render: renderUnits,
    },
    {
      data: "salidas_pallets",
      title: "Salidas · Pallets",
      className: "text-end text-danger",
      render: formatNullableUnits,
    },
    {
      data: "salidas_cajas",
      title: "Salidas · Cajas",
      className: "text-end text-danger",
      render: formatNullableUnits,
    },
    {
      data: "salidas_unidades_sueltas",
      title: "Salidas · Unid. sueltas",
      className: "text-end text-danger",
      render: formatNullableUnits,
    },
    {
      data: "salidas",
      title: "Salidas · Total unid.",
      className: "text-end text-danger fw-semibold",
      render: renderUnits,
    },
    {
      data: "saldo_final_pallets",
      title: "Saldo final · Pallets",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_final_cajas",
      title: "Saldo final · Cajas",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_final_unidades_sueltas",
      title: "Saldo final · Unid. sueltas",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "saldo_final",
      title: "Saldo final · Total unid.",
      className: "text-end fw-semibold",
      render: renderUnits,
    },
    {
      data: "inventario_fisico_pallets",
      title: "Inventario físico · Pallets",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "inventario_fisico_cajas",
      title: "Inventario físico · Cajas",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "inventario_fisico_unidades_sueltas",
      title: "Inventario físico · Unid. sueltas",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "inventario_fisico",
      title: "Inventario físico · Total unid.",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "diferencia_unidades",
      title: "Diferencia (unid.)",
      className: "text-end",
      render: formatNullableUnits,
    },
    {
      data: "valor_unidades",
      title: "Valor unidades",
      className: "text-end",
      render: renderDecimal,
    },
    {
      data: "valor_diferencia",
      title: "Valor diferencia",
      className: "text-end",
      render: renderDecimal,
    },
  ];

  const $table = $("#" + tableId);

  if ($table.find("thead").length === 0) {
    $table.prepend("<thead></thead>");
  }

  const headerHtml = `
    <tr>
      <th rowspan="2"></th>
      <th rowspan="2">Código</th>
      <th rowspan="2">Descripción</th>
      <th colspan="2">Presentación</th>
      <th colspan="4">Saldo inicial</th>
      <th colspan="4" class="text-success">Ingresos</th>
      <th colspan="4" class="text-danger">Salidas</th>
      <th colspan="4">Saldo final</th>
      <th colspan="4">Inventario físico</th>
      <th rowspan="2">Dif. unidades</th>
      <th rowspan="2">Valor unidades</th>
      <th rowspan="2">Valor diferencia</th>
    </tr>
    <tr>
      <th class="text-end">Cajas / pallet</th>
      <th class="text-end">Unid. / caja</th>
      <th class="text-end">Pallets</th>
      <th class="text-end">Cajas</th>
      <th class="text-end">Unid. sueltas</th>
      <th class="text-end">Total unid.</th>
      <th class="text-end text-success">Pallets</th>
      <th class="text-end text-success">Cajas</th>
      <th class="text-end text-success">Unid. sueltas</th>
      <th class="text-end text-success">Total unid.</th>
      <th class="text-end text-danger">Pallets</th>
      <th class="text-end text-danger">Cajas</th>
      <th class="text-end text-danger">Unid. sueltas</th>
      <th class="text-end text-danger">Total unid.</th>
      <th class="text-end">Pallets</th>
      <th class="text-end">Cajas</th>
      <th class="text-end">Unid. sueltas</th>
      <th class="text-end">Total unid.</th>
      <th class="text-end">Pallets</th>
      <th class="text-end">Cajas</th>
      <th class="text-end">Unid. sueltas</th>
      <th class="text-end">Total unid.</th>
    </tr>
  `;
  $table.find("thead").html(headerHtml);

  if ($table.find("tfoot").length === 0) {
    $table.append("<tfoot></tfoot>");
  }

  const footerHtml = `
    <tr>
      <th></th>
      <th colspan="2" class="text-end">Totales</th>
      <th class="text-end">—</th>
      <th class="text-end">—</th>
      <th class="text-end" data-footer="saldo_inicial_pallets">—</th>
      <th class="text-end" data-footer="saldo_inicial_cajas">—</th>
      <th class="text-end" data-footer="saldo_inicial_unidades_sueltas">—</th>
      <th class="text-end" data-footer="saldo_inicial">0</th>
      <th class="text-end" data-footer="ingresos_pallets">—</th>
      <th class="text-end" data-footer="ingresos_cajas">—</th>
      <th class="text-end" data-footer="ingresos_unidades_sueltas">—</th>
      <th class="text-end" data-footer="ingresos">0</th>
      <th class="text-end" data-footer="salidas_pallets">—</th>
      <th class="text-end" data-footer="salidas_cajas">—</th>
      <th class="text-end" data-footer="salidas_unidades_sueltas">—</th>
      <th class="text-end" data-footer="salidas">0</th>
      <th class="text-end" data-footer="saldo_final_pallets">—</th>
      <th class="text-end" data-footer="saldo_final_cajas">—</th>
      <th class="text-end" data-footer="saldo_final_unidades_sueltas">—</th>
      <th class="text-end" data-footer="saldo_final">0</th>
      <th class="text-end" data-footer="inventario_fisico_pallets">—</th>
      <th class="text-end" data-footer="inventario_fisico_cajas">—</th>
      <th class="text-end" data-footer="inventario_fisico_unidades_sueltas">—</th>
      <th class="text-end" data-footer="inventario_fisico">—</th>
      <th class="text-end" data-footer="diferencia_unidades">—</th>
      <th class="text-end" data-footer="valor_unidades">0,00</th>
      <th class="text-end" data-footer="valor_diferencia">—</th>
    </tr>
  `;
  $table.find("tfoot").html(footerHtml);

  const updateMetricCards = (totals, grandTotals = null) => {
    Object.entries(cardElements).forEach(([key, $element]) => {
      if (!$element.length) {
        return;
      }
      const formatter = metricFormatters[key] || formatUnits;
      const filteredValue = totals ? totals[key] : null;
      const grandValue = grandTotals ? grandTotals[key] : null;
      $element.text(ensureMetricText(filteredValue, grandValue, formatter));
    });
  };

  const updateFooters = (totals) => {
    Object.entries(footerFormatters).forEach(([key, formatter]) => {
      const $footer = $table.find(`[data-footer="${key}"]`);
      if (!$footer.length) {
        return;
      }
      const value = totals ? totals[key] : null;
      const text = formatter === formatNullableUnits || formatter === formatNullableFixed2
        ? formatter(value)
        : formatter(value ?? 0);
      $footer.text(text);
    });
  };

  const updateTotals = (totals, grandTotals = null) => {
    updateFooters(totals);
    updateMetricCards(totals, grandTotals);
  };

  const computeFilteredTotals = (api) => {
    const totals = {
      productos: 0,
      saldo_inicial: 0,
      ingresos: 0,
      salidas: 0,
      saldo_final: 0,
      valor_unidades: 0,
    };

    const optionalPresence = {};
    optionalTotalFields.forEach((field) => {
      totals[field] = 0;
      optionalPresence[field] = false;
    });

    const filteredRows = api.rows({ search: "applied" });
    totals.productos = filteredRows.count();

    filteredRows.data().each((row) => {
      if (!row) {
        return;
      }

      totals.saldo_inicial += toNumber(row.saldo_inicial);
      totals.ingresos += toNumber(row.ingresos);
      totals.salidas += toNumber(row.salidas);
      totals.saldo_final += toNumber(row.saldo_final);

      if (row.valor_unidades !== null && row.valor_unidades !== undefined && row.valor_unidades !== "") {
        totals.valor_unidades += toNumber(row.valor_unidades);
      }

      optionalTotalFields.forEach((field) => {
        const value = row[field];
        if (value !== null && value !== undefined && value !== "") {
          totals[field] += toNumber(value);
          optionalPresence[field] = true;
        }
      });
    });

    optionalTotalFields.forEach((field) => {
      if (!optionalPresence[field]) {
        totals[field] = null;
      }
    });

    return totals;
  };

  const dtOptions = {
    ajax: {
      url: ajaxUrl,
      data: function (params) {
        params.desde = ($filterDesde.val() || "").trim();
        params.hasta = ($filterHasta.val() || "").trim();
      },
      dataSrc: function (json) {
        if (json && json.ok !== false) {
          const grand = json.grandTotals || json.totals || null;
          lastGrandTotals = { ...defaultTotals, ...(grand || {}) };

          const filteredFromServer = json.totals
            ? { ...defaultTotals, ...json.totals }
            : null;
          updateTotals(filteredFromServer, lastGrandTotals);

          if (json.meta && json.meta.filters) {
            if (typeof json.meta.filters.desde === "string") {
              lastFilters.desde = json.meta.filters.desde;
              $filterDesde.val(json.meta.filters.desde);
            }
            if (typeof json.meta.filters.hasta === "string") {
              lastFilters.hasta = json.meta.filters.hasta;
              $filterHasta.val(json.meta.filters.hasta);
            }
          }

          refreshPdfLink();

          return Array.isArray(json.rows) ? json.rows : [];
        }

        lastGrandTotals = { ...defaultTotals };
        updateTotals(null, lastGrandTotals);
        if (json && json.error) {
          // eslint-disable-next-line no-console
          console.error("[Inventario movimientos]", json.error, json.detail || "");
        }
        refreshPdfLink();
        return [];
      },
    },
    select: { style: "os", selector: "td.select-checkbox" },
    searching: true,
    ordering: true,
    footerCallback: function () {
      const api = this.api();
      const filteredTotals = computeFilteredTotals(api);
      updateTotals(filteredTotals, lastGrandTotals);
    },
  };

  const { table } = initializeBaseDataTable({
    tableId,
    ajaxUrl,
    columns,
    editorFields: false,
    order: [
      [1, "asc"],
      [2, "asc"],
    ],
    dataTableOptions: dtOptions,
    tableButtons: [],
    editorButtons: [],
  });

  const reloadWithFilters = () => {
    lastFilters = {
      desde: ($filterDesde.val() || "").trim(),
      hasta: ($filterHasta.val() || "").trim(),
    };
    refreshPdfLink();
    updateTotals(null, lastGrandTotals);
    table.ajax.reload();
  };

  $applyFilters.on("click", () => {
    reloadWithFilters();
  });

  $resetFilters.on("click", () => {
    $filterDesde.val("");
    $filterHasta.val("");
    lastFilters = { desde: "", hasta: "" };
    refreshPdfLink();
    reloadWithFilters();
  });

  $pdfLink.on("click", (event) => {
    event.preventDefault();
    const url = $pdfLink.attr("href");
    if (url) {
      window.open(url, "_blank");
    }
  });

  refreshPdfLink();

  table.on("xhr.dt", (event, settings, json) => {
    if (json && json.ok !== false) {
      if (json.meta && json.meta.filters) {
        if (typeof json.meta.filters.desde === "string") {
          lastFilters.desde = json.meta.filters.desde;
          $filterDesde.val(json.meta.filters.desde);
        }
        if (typeof json.meta.filters.hasta === "string") {
          lastFilters.hasta = json.meta.filters.hasta;
          $filterHasta.val(json.meta.filters.hasta);
        }
      }

      lastGrandTotals = {
        ...defaultTotals,
        ...((json.grandTotals || json.totals) || {}),
      };

      const filteredTotals = json.totals
        ? { ...defaultTotals, ...json.totals }
        : null;

      updateTotals(filteredTotals, lastGrandTotals);
      refreshPdfLink();
    } else {
      lastGrandTotals = { ...defaultTotals };
      updateTotals(null, lastGrandTotals);
      refreshPdfLink();
    }
  });
});
