/**
 * SOL · Reporte de Almacenamiento
 */
(function (window, $) {
  'use strict';

  const baseUrl =
    document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
    (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');
  const joinUrl = (path) => `${baseUrl.replace(/\/$/, '')}/${String(path || '').replace(/^\//, '')}`;
  const apiUrl = joinUrl('api/reportes/almacenamiento.php');

  const formatNumber = (value, decimals = 0) => {
    const num = Number(value ?? 0);
    if (!Number.isFinite(num)) {
      return '-';
    }
    try {
      return new Intl.NumberFormat('es-AR', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals,
      }).format(num);
    } catch (err) {
      return num.toFixed(decimals);
    }
  };

  const formatPercent = (value, decimals = 1) => {
    if (value === null || value === undefined || Number.isNaN(Number(value))) {
      return '-';
    }
    return `${formatNumber(value, decimals)}%`;
  };

  const formatDate = (value) => {
    if (!value) {
      return '-';
    }
    try {
      const date = new Date(value);
      if (Number.isNaN(date.getTime())) {
        return value;
      }
      return new Intl.DateTimeFormat('es-AR').format(date);
    } catch (err) {
      return value;
    }
  };

  const notifier = (type, message) => {
    if (typeof Swal !== 'undefined' && typeof Swal.fire === 'function') {
      Swal.fire({ icon: type, title: message, timer: 2600, showConfirmButton: false });
      return;
    }
    if (type === 'error') {
      console.error('[Almacenamiento]', message);
    } else if (type === 'warning') {
      console.warn('[Almacenamiento]', message);
    } else {
      console.info('[Almacenamiento]', message);
    }
  };

  let dataTable = null;
  let currentRows = [];
  let currentSummary = {};

  $(function () {
    initTable();
    bindEvents();
    loadCombos();
  });

  function initTable() {
    dataTable = $('#tblAlmacenamiento').DataTable({
      data: [],
      columns: [
        { data: 'sku', title: 'SKU', defaultContent: '-' },
        { data: 'producto', title: 'Producto', defaultContent: '-' },
        { data: 'cliente', title: 'Cliente', defaultContent: '-' },
        { data: 'deposito', title: 'Depósito', defaultContent: '-' },
        {
          data: 'conteo_fecha',
          title: 'Fecha conteo',
          render: (value) => formatDate(value),
        },
        {
          data: 'conteo_unidades',
          className: 'text-end',
          title: 'Inventario físico (UC)',
          render: (value) => (value === null ? '-' : formatNumber(value)),
        },
        {
          data: 'stock_unidades',
          className: 'text-end',
          title: 'Stock actual (UC)',
          render: (value) => (value === null ? '-' : formatNumber(value)),
        },
        {
          data: 'delta_unidades',
          className: 'text-end',
          title: 'Diferencia (UC)',
          render: (value) => (value === null ? '-' : formatNumber(value)),
        },
        {
          data: 'delta_pct',
          className: 'text-end',
          title: 'Variación %',
          render: (value) => (value === null ? '-' : formatPercent(value, 2)),
        },
        {
          data: 'exactitud_pct',
          className: 'text-end',
          title: 'Exactitud %',
          render: (value) => (value === null ? '-' : formatPercent(value, 2)),
        },
        {
          data: 'semaforo',
          className: 'text-center table-semaforo',
          title: 'Semáforo',
          orderable: false,
          render: (value) => renderSemaforo(value),
        },
        { data: 'observaciones', title: 'Observaciones', defaultContent: '-' },
      ],
      order: [[9, 'desc']],
      pageLength: 25,
      lengthMenu: [25, 50, 100, 250],
      language: { url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json' },
    });
  }

  function renderSemaforo(value) {
    const map = {
      OK: { cls: 'ok', label: 'Dentro de tolerancia' },
      ALERTA: { cls: 'alerta', label: 'Alerta' },
      CRITICO: { cls: 'critico', label: 'Crítico' },
      SIN_CONTEO: { cls: 'sin-datos', label: 'Sin conteo' },
      SIN_DATOS: { cls: 'sin-datos', label: 'Sin datos' },
    };
    const info = map[value] || map.SIN_DATOS;
    return `
      <span class="text-nowrap">
        <span class="dot ${info.cls}"></span>
        <span>${info.label}</span>
      </span>
    `;
  }

  function bindEvents() {
    $('#frmAlmacenamiento').on('submit', function (event) {
      event.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function () {
      const form = document.getElementById('frmAlmacenamiento');
      if (form) {
        form.reset();
      }
      $('#deposito_id').val('');
      $('#cliente_id').val('');
      $('#tolerancia_pct').val('2');
      $('#rotacion_dias').val('30');
      loadData();
    });

    $('#btnExportCsv').on('click', function () {
      if (!currentRows.length) {
        notifier('warning', 'No hay datos para exportar');
        return;
      }
      exportCsv();
    });
  }

  function loadCombos() {
    $.getJSON(apiUrl, { meta: 'filters' })
      .done(function (response) {
        if (!response || response.ok === false) {
          notifier('warning', response?.error || 'No se pudieron cargar los filtros');
          return;
        }
        fillSelect('#deposito_id', response.depositos, 'Todos');
        fillSelect('#cliente_id', response.clientes, 'Todos');
      })
      .fail(function () {
        notifier('error', 'Error de conexión al cargar filtros');
      });
  }

  function fillSelect(selector, items, placeholder) {
    const $select = $(selector);
    if (!$select.length) return;
    const current = $select.val();
    $select.find('option').not(':first').remove();
    if (Array.isArray(items)) {
      items.forEach(function (item) {
        const id = item.id ?? '';
        const code = item.code ?? '';
        const razon = item.razon_social ?? '';
        const nombre = item.nombre ?? '';
        let label = razon || nombre || code;
        if (code && (razon || nombre)) {
          label = `${code} · ${razon || nombre}`;
        }
        if (!label) {
          label = `ID ${id}`;
        }
        $select.append(new Option(label, id));
      });
    }
    if (current && $select.find(`option[value="${current}"]`).length) {
      $select.val(current);
    } else {
      $select.val('');
    }
  }

  function loadData() {
    const params = Object.fromEntries(new FormData(document.getElementById('frmAlmacenamiento')).entries());

    setLoadingState(true);
    $.ajax({
      url: apiUrl,
      method: 'GET',
      data: params,
      dataType: 'json',
    })
      .done(function (response) {
        if (!response || response.ok === false) {
          notifier('error', response?.error || 'No se pudo generar el reporte');
          resetView();
          return;
        }
        currentRows = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || {};

        if (dataTable) {
          dataTable.clear();
          dataTable.rows.add(currentRows);
          dataTable.draw();
        }

        updateSummary(currentSummary, response.filters || {});
        updateIndicators(response.indicators || []);
        $('#btnExportCsv').prop('disabled', currentRows.length === 0);
      })
      .fail(function () {
        notifier('error', 'Error de conexión al servidor');
        resetView();
      })
      .always(function () {
        setLoadingState(false);
      });
  }

  function setLoadingState(isLoading) {
    const $tbody = $('#tblAlmacenamiento tbody');
    if (isLoading) {
      $tbody.html(`
        <tr class="text-muted">
          <td colspan="12" class="text-center py-4">
            <div class="spinner-border spinner-border-sm me-2" role="status"></div>
            Calculando comparativo...
          </td>
        </tr>
      `);
    }
  }

  function resetView() {
    currentRows = [];
    currentSummary = {};
    if (dataTable) {
      dataTable.clear().draw();
    }
    updateSummary({}, {});
    updateIndicators([]);
    $('#btnExportCsv').prop('disabled', true);
  }

  function updateSummary(summary, filters) {
    const exactitud = summary?.exactitud_global ?? null;
    const fuera = summary?.skus_fuera_tolerancia ?? 0;
    const delta = summary?.delta_total ?? 0;
    const ultima = summary?.ultima_fecha_conteo ?? null;

    $('#badgeSkus').text(`${currentRows.length} SKU${currentRows.length === 1 ? '' : 's'}`);
    $('#summaryExactitud').text(exactitud !== null ? formatPercent(exactitud, 2) : '-');
    $('#summaryFuera').text(formatNumber(fuera));
    $('#summaryDelta').text(formatNumber(delta));
    $('#summaryUltimo').text(formatDate(ultima));

    const tolerancia = filters?.tolerancia_pct ?? summary?.tolerancia ?? null;
    const ultimaNota = ultima ? `Último conteo: ${formatDate(ultima)}.` : 'Sin inventarios recientes registrados.';
    const toleranciaNota = tolerancia ? ` Tolerancia actual: ±${formatNumber(tolerancia, 1)}%.` : '';
    $('#summaryNotas').text(`${ultimaNota}${toleranciaNota}`);
  }

  function updateIndicators(indicadores) {
    const cards = $('#indicadoresContainer .card-indicador');
    cards.each(function (index) {
      const card = $(this);
      const data = indicadores[index] || null;
      const estadoClass = mapEstadoToClass(data?.estado);
      const estadoBadge = mapEstadoToBadge(data?.estado);
      const estadoValor = mapEstadoToValueClass(data?.estado);

      card.removeClass('ok alerta critico sin-datos').addClass(estadoClass);
      card.find('h6').text(data?.label || 'Indicador');
      const valorField = card.find('h3[data-field="valor"]');
      valorField
        .removeClass('text-success text-warning text-danger text-muted')
        .addClass(estadoValor)
        .text(formatIndicatorValue(data));
      card.find('small[data-field="descripcion"]').text(data?.descripcion || 'Sin datos disponibles');
      const metaText = data?.meta !== undefined ? `Meta ${formatIndicatorMeta(data.meta, data?.clave)}` : 'Sin meta';
      card
        .find('.badge')
        .attr('class', `badge badge-indicador ${estadoBadge}`)
        .text(metaText);
    });
  }

  function mapEstadoToClass(estado) {
    switch (estado) {
      case 'OK':
        return 'ok';
      case 'ALERTA':
        return 'alerta';
      case 'CRITICO':
        return 'critico';
      default:
        return 'sin-datos';
    }
  }

  function mapEstadoToBadge(estado) {
    switch (estado) {
      case 'OK':
        return 'bg-success';
      case 'ALERTA':
        return 'bg-warning text-dark';
      case 'CRITICO':
        return 'bg-danger';
      default:
        return 'bg-secondary';
    }
  }

  function mapEstadoToValueClass(estado) {
    switch (estado) {
      case 'OK':
        return 'text-success';
      case 'ALERTA':
        return 'text-warning';
      case 'CRITICO':
        return 'text-danger';
      default:
        return 'text-muted';
    }
  }

  function formatIndicatorValue(indicador) {
    if (!indicador || indicador.valor === null || indicador.valor === undefined) {
      return '-';
    }
    if (indicador.clave === 'exactitud' || indicador.clave === 'ocupacion') {
      return formatPercent(indicador.valor, 1);
    }
    if (indicador.clave === 'rotacion') {
      return `${formatNumber(indicador.valor, 2)}x`;
    }
    return formatNumber(indicador.valor, 1);
  }

  function formatIndicatorMeta(meta, key) {
    if (meta === null || meta === undefined) {
      return '-';
    }
    if (key === 'exactitud' || key === 'ocupacion') {
      return formatPercent(meta, 1);
    }
    if (key === 'rotacion') {
      return `${formatNumber(meta, 2)}x`;
    }
    return formatNumber(meta, Number.isInteger(meta) ? 0 : 1);
  }

  function exportCsv() {
    const headers = [
      'SKU',
      'Producto',
      'Cliente',
      'Depósito',
      'Fecha conteo',
      'Inventario físico (UC)',
      'Stock actual (UC)',
      'Diferencia (UC)',
      'Variación %',
      'Exactitud %',
      'Semáforo',
      'Observaciones',
    ];

    const rows = currentRows.map((row) => [
      row.sku,
      row.producto,
      row.cliente,
      row.deposito,
      row.conteo_fecha || '',
      row.conteo_unidades ?? '',
      row.stock_unidades ?? '',
      row.delta_unidades ?? '',
      row.delta_pct ?? '',
      row.exactitud_pct ?? '',
      row.semaforo,
      row.observaciones,
    ]);

    const csvLines = [headers.join(';')].concat(
      rows.map((fields) =>
        fields
          .map((field) => {
            if (field === null || field === undefined) {
              return '';
            }
            const value = String(field).replace(/"/g, '""');
            return value.includes(';') || value.includes('\n') ? `"${value}"` : value;
          })
          .join(';')
      )
    );

    const blob = new Blob([csvLines.join('\n')], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `reporte_almacenamiento_${new Date().toISOString().slice(0, 10)}.csv`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
  }
})(window, jQuery);
